/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.core.Failure;
import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Agent;
import java.util.Objects;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Notifies an agent that a failure has occurred when it tried to spawn another agent.
 * The event is fired within the default space of the context in which the spawn tentative was.
 * Subclasses of this event could be defined in order to specify more in detail the failure.
 * Only the spawning agent receives this event.
 * 
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 * @since 0.12
 */
@SarlSpecification("0.12")
@SarlElementType(15)
@SuppressWarnings("all")
public class AgentSpawnFailure extends Failure {
  /**
   * The identifier of the context in which the spawn has failed.
   */
  public final UUID contextId;
  
  /**
   * The type of agent for which the spawn has failed.
   */
  public final Class<? extends Agent> agentType;
  
  /**
   * Construct the event.
   * 
   * @param source the address of the emitter.
   * @param contextId the identifier of the context in which the spawn has failed.
   * @param agentType the type of the agent for which the spawn has failed.
   * @param cause the cause of the failure.
   */
  @DefaultValueSource
  public AgentSpawnFailure(@DefaultValue("io.sarl.core.AgentSpawnFailure#NEW_0") final Address source, final UUID contextId, final Class<? extends Agent> agentType, final Object cause) {
    super(source, cause);
    this.contextId = contextId;
    this.agentType = agentType;
  }
  
  /**
   * Default value for the parameter source
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static Address $DEFAULT_VALUE$NEW_0() {
    return null;
  }
  
  /**
   * Construct the event.
   * 
   * @optionalparam source the address of the emitter.
   * @optionalparam contextId the identifier of the context in which the spawn has failed.
   * @optionalparam agentType the type of the agent for which the spawn has failed.
   * @optionalparam cause the cause of the failure.
   */
  @DefaultValueUse("io.sarl.lang.core.Address,java.util.UUID,java.lang.Class,java.lang.Object")
  @SyntheticMember
  public AgentSpawnFailure(final UUID contextId, final Class<? extends Agent> agentType, final Object cause) {
    this($DEFAULT_VALUE$NEW_0(), contextId, agentType, cause);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    AgentSpawnFailure other = (AgentSpawnFailure) obj;
    if (!Objects.equals(this.contextId, other.contextId))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.contextId);
    return result;
  }
  
  /**
   * Returns a String representation of the AgentSpawnFailure event's attributes only.
   */
  @SyntheticMember
  @Pure
  protected void toString(final ToStringBuilder builder) {
    super.toString(builder);
    builder.add("contextId", this.contextId);
    builder.add("agentType", this.agentType);
  }
  
  @SyntheticMember
  private static final long serialVersionUID = -1964798794L;
}
