/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.PrivateAPI;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Agent;
import io.sarl.lang.core.SRESpecificDataContainer;
import java.util.Objects;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Description of a task an agent has schedule to be performed at a later time.
 * This class is not thread-safe.
 * 
 * @author <a href="http://www.sebastianrodriguez.com.ar/">Sebastian Rodriguez</a>
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 */
@SarlSpecification("0.12")
@SarlElementType(10)
@SuppressWarnings("all")
public class AgentTask extends SRESpecificDataContainer {
  /**
   * A guard that is always true.
   */
  public static final Function1<? super Agent, ? extends Boolean> TRUE_GUARD = ((Function1<Agent, Boolean>) (Agent it) -> {
    return Boolean.valueOf(true);
  });
  
  /**
   * A guard that is always false.
   */
  public static final Function1<? super Agent, ? extends Boolean> FALSE_GUARD = ((Function1<Agent, Boolean>) (Agent it) -> {
    return Boolean.valueOf(false);
  });
  
  private String name;
  
  private Function1<? super Agent, ? extends Boolean> guard;
  
  private Procedure1<? super Agent> procedure;
  
  private final Object initiator;
  
  /**
   * Construct an AgentTask with the given initiator.
   * 
   * @param name the name of the task.
   * @param initiator the object that has initiated the execution of this task.
   *     If the value is {@code null} (the default value of the argument), the agent is assumed to
   *     be the initiator.
   */
  @DefaultValueSource
  public AgentTask(final String name, @DefaultValue("io.sarl.core.AgentTask#NEW_0") final Object initiator) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (name != null);
      }
    }
    assert new $AssertEvaluator$().$$result;
    this.name = name;
    this.initiator = initiator;
  }
  
  /**
   * Replies the procedure that is associated to this task.
   * 
   * @return the procedure to run.
   */
  @Pure
  public Procedure1<? super Agent> getProcedure() {
    return this.procedure;
  }
  
  /**
   * Change the procedure associated to this task.
   * 
   * @param procedure code of the procedure to associate to the task.
   */
  public void setProcedure(final Procedure1<? super Agent> procedure) {
    this.procedure = procedure;
  }
  
  /**
   * Replies the guard of this task.
   * 
   * @return the guard.
   */
  @Pure
  public Function1<? super Agent, ? extends Boolean> getGuard() {
    return this.guard;
  }
  
  /**
   * Change the guard of this task.
   * 
   * @param guard the code of the function that is the predicate associated to the guard.
   * @see #unless
   * @see #ifTrue
   */
  public void setGuard(final Function1<? super Agent, ? extends Boolean> guard) {
    this.guard = guard;
  }
  
  /**
   * Replies the name of this task.
   * 
   * @return the name.
   */
  @Pure
  public String getName() {
    return this.name;
  }
  
  /**
   * Change the name of this task.
   * 
   * @param name name of the task.
   * @since 0.7, it is part of the private API.
   * @privateapi
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @PrivateAPI
  public void setTaskName(final String name) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (name != null);
      }
    }
    assert new $AssertEvaluator$().$$result;
    this.name = name;
  }
  
  /**
   * Change the guard of this that with the negation of the given predicate.
   * 
   * @param predicate the code of the function that is the predicate to test.
   * @return <code>this</code>.
   * @see #setGuard
   */
  public AgentTask unless(final Function1<? super Agent, ? extends Boolean> predicate) {
    if ((predicate == null)) {
      this.guard = AgentTask.FALSE_GUARD;
    } else {
      final Function1<Agent, Boolean> _function = (Agent it) -> {
        Boolean _apply = predicate.apply(it);
        return Boolean.valueOf((!((_apply) == null ? false : (_apply).booleanValue())));
      };
      this.guard = _function;
    }
    return this;
  }
  
  /**
   * Change the guard to the given predicate.
   * 
   * @param predicate the code of the function that is the predicate to test.
   * @return <code>this</code>.
   * @see #setGuard
   */
  public AgentTask ifTrue(final Function1<? super Agent, ? extends Boolean> predicate) {
    if ((predicate == null)) {
      this.guard = null;
    } else {
      this.guard = predicate;
    }
    return this;
  }
  
  @Override
  @Pure
  public String toString() {
    return ("AgentTask: " + this.name);
  }
  
  /**
   * Replies the object that has initiated the execution of this task.
   * If the value replied by this function is {@code null}, the agent is assumed to be the initiator.
   * 
   * @return the initiator, or {@code null} if the initiator is the agent itself.
   * @since 0.5
   */
  @Pure
  public Object getInitiator() {
    return this.initiator;
  }
  
  /**
   * Default value for the parameter initiator
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static Object $DEFAULT_VALUE$NEW_0() {
    return null;
  }
  
  /**
   * Construct an AgentTask with the given initiator.
   * 
   * @optionalparam name the name of the task.
   * @optionalparam initiator the object that has initiated the execution of this task.
   *     If the value is {@code null} (the default value of the argument), the agent is assumed to
   *     be the initiator.
   */
  @DefaultValueUse("java.lang.String,java.lang.Object")
  @SyntheticMember
  public AgentTask(final String name) {
    this(name, $DEFAULT_VALUE$NEW_0());
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    AgentTask other = (AgentTask) obj;
    if (!Objects.equals(this.name, other.name))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.name);
    return result;
  }
}
