/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.SarlAsynchronousExecution;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Behavior;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.core.Event;
import io.sarl.lang.core.EventListener;
import io.sarl.lang.core.Scope;
import io.sarl.lang.util.ConcurrentCollection;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Agent behaviors management.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface Behaviors extends Capacity {
  /**
   * Replies <code>true</code> if at least a behavior is registered.
   * 
   * <p>A registered behavior is not necessary running. It may wait for events.
   * 
   * @return <code>true</code> if a behavior is registered; otherwise <code>false</code>.
   * @since 0.5
   */
  @Pure
  boolean hasRegisteredBehavior();
  
  /**
   * Replies the registered behaviors.
   * 
   * <p>A registered behavior is not necessary running. It may wait for events.
   * 
   * @return the unmodifiable collection of the registered behavior.
   * @since 0.5
   */
  @Pure
  ConcurrentCollection<Behavior> getRegisteredBehaviors();
  
  /**
   * Register a Behavior for the owner agent.
   * The new Behavior will react to all declared events received by the agent.
   * 
   * <p>If the filter is provided, it will be used for determining if the given behavior accepts a specific event.
   * If the filter function replies {@code true} for a specific event as argument, the event is fired in the
   * behavior context. If the filter function replies {@code false}, the event is not fired in the behavior context.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param attitude the behavior to add to the agent.
   * @param filter the filter that enables to indicates if an event is accepted by a behavior.
   * @param initializationParameters the set of parameters to pass to the behavior within the {@code Initialize} event.
   * @return the given behavior.
   * @since 0.5 for the {@code filter} parameter.
   */
  @DefaultValueSource
  Behavior registerBehavior(final Behavior attitude, @DefaultValue("io.sarl.core.Behaviors#REGISTERBEHAVIOR_0") final Function1<? super Event, ? extends Boolean> filter, final Object... initializationParameters);
  
  /**
   * Unregisters a behavior for the owner agent.
   * 
   * @param attitude the behavior to remove to the agent.
   * @return the given behavior.
   */
  Behavior unregisterBehavior(final Behavior attitude);
  
  /**
   * Wake the agent's behaviors reacting to the Event evt.
   * 
   * <p>Note that the scope parameter could be used only for filtering the agents (the current receiving agent)
   * in order to determine the ones that will receive the event.
   * The behaviors of the current agent (registered with
   * {@link #registerBehavior(Behavior, Function1, Object[])}
   * will always
   * receive the event if the current agent is not discarded. Because the behaviors have no associated address,
   * they cannot be filtered individually.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param event the event to emit to the agent's behaviors and in the internal context.
   * @param scope the definition of the scope that will be used for selected the receivers of the events.
   */
  @DefaultValueSource
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void wake(final Event event, @DefaultValue("io.sarl.core.Behaviors#WAKE_0") final Scope<Address> scope);
  
  /**
   * Wake a specific agent's behavior reacting to the Event evt.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param behavior the agent's behavior to wake up.
   * @param event the event to emit to the agent's behaviors and in the internal context.
   * @since 0.12
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void wake(final Behavior behavior, final Event event);
  
  /**
   * Wake a set of specific agent's behaviors reacting to the Event evt.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param behaviors the agent's behaviors to wake up.
   * @param event the event to emit to the agent's behaviors and in the internal context.
   * @since 0.12
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void wake(final Iterable<Behavior> behaviors, final Event event);
  
  /**
   * Replies the interface to dispatch an event to agent's Behaviors.
   * 
   * @return the event listener.
   */
  @Pure
  EventListener asEventListener();
  
  /**
   * Default value for the parameter filter
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Function1 $DEFAULT_VALUE$REGISTERBEHAVIOR_0() {
    return null;
  }
  
  /**
   * Default value for the parameter scope
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Scope $DEFAULT_VALUE$WAKE_0() {
    return null;
  }
  
  /**
   * Register a Behavior for the owner agent.
   * The new Behavior will react to all declared events received by the agent.
   * 
   * <p>If the filter is provided, it will be used for determining if the given behavior accepts a specific event.
   * If the filter function replies {@code true} for a specific event as argument, the event is fired in the
   * behavior context. If the filter function replies {@code false}, the event is not fired in the behavior context.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param attitude the behavior to add to the agent.
   * @optionalparam filter the filter that enables to indicates if an event is accepted by a behavior.
   * @param initializationParameters the set of parameters to pass to the behavior within the {@code Initialize} event.
   * @return the given behavior.
   * @since 0.5 for the {@code filter} parameter.
   */
  @DefaultValueUse("io.sarl.lang.core.Behavior,(io.sarl.lang.core.Event)=>boolean,java.lang.Object*")
  @SyntheticMember
  default Behavior registerBehavior(final Behavior attitude, final Object... initializationParameters) {
    return registerBehavior(attitude, $DEFAULT_VALUE$REGISTERBEHAVIOR_0(), initializationParameters);
  }
  
  /**
   * Wake the agent's behaviors reacting to the Event evt.
   * 
   * <p>Note that the scope parameter could be used only for filtering the agents (the current receiving agent)
   * in order to determine the ones that will receive the event.
   * The behaviors of the current agent (registered with
   * {@link #registerBehavior(Behavior, Function1, Object[])}
   * will always
   * receive the event if the current agent is not discarded. Because the behaviors have no associated address,
   * they cannot be filtered individually.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param event the event to emit to the agent's behaviors and in the internal context.
   * @optionalparam scope the definition of the scope that will be used for selected the receivers of the events.
   */
  @DefaultValueUse("io.sarl.lang.core.Event,io.sarl.lang.core.Scope")
  @SyntheticMember
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  default void wake(final Event event) {
    wake(event, $DEFAULT_VALUE$WAKE_0());
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends Behaviors> extends Capacity.ContextAwareCapacityWrapper<C> implements Behaviors {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public boolean hasRegisteredBehavior() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.hasRegisteredBehavior();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public ConcurrentCollection<Behavior> getRegisteredBehaviors() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getRegisteredBehaviors();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public Behavior registerBehavior(final Behavior attitude, final Function1<? super Event, ? extends Boolean> filter, final Object... initializationParameters) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.registerBehavior(attitude, filter, initializationParameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public Behavior unregisterBehavior(final Behavior attitude) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.unregisterBehavior(attitude);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void wake(final Event event, final Scope<Address> scope) {
      try {
        ensureCallerInLocalThread();
        this.capacity.wake(event, scope);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void wake(final Behavior behavior, final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.wake(behavior, event);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void wake(final Iterable<Behavior> behaviors, final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.wake(behaviors, event);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public EventListener asEventListener() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.asEventListener();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public Behavior registerBehavior(final Behavior attitude, final Object... initializationParameters) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.registerBehavior(attitude, initializationParameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void wake(final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.wake(event);
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
