/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Event;
import java.util.Objects;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Notifies the behaviors of the agent that it has joined a new super-holon.
 * The event is fired within the internal bus of the agent that have joined a super-holon.
 * The source of the event is the agent itself.
 * 
 * @author <a href="http://www.sebastianrodriguez.com.ar/">Sebastian Rodriguez</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 */
@SarlSpecification("0.12")
@SarlElementType(15)
@SuppressWarnings("all")
public final class ContextJoined extends Event {
  /**
   * The identifier of the context in the super-holon.
   */
  public final UUID holonContextID;
  
  /**
   * The identifier of the default space in the super-holon context.
   */
  public final UUID defaultSpaceID;
  
  /**
   * Construct the event.
   * 
   * @param contextID the identifier of the context.
   * @param defaultSpaceID the identifier of the default space.
   */
  public ContextJoined(final UUID contextID, final UUID defaultSpaceID) {
    this.holonContextID = contextID;
    this.defaultSpaceID = defaultSpaceID;
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    ContextJoined other = (ContextJoined) obj;
    if (!Objects.equals(this.holonContextID, other.holonContextID))
      return false;
    if (!Objects.equals(this.defaultSpaceID, other.defaultSpaceID))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.holonContextID);
    result = prime * result + Objects.hashCode(this.defaultSpaceID);
    return result;
  }
  
  /**
   * Returns a String representation of the ContextJoined event's attributes only.
   */
  @SyntheticMember
  @Pure
  protected void toString(final ToStringBuilder builder) {
    super.toString(builder);
    builder.add("holonContextID", this.holonContextID);
    builder.add("defaultSpaceID", this.defaultSpaceID);
  }
  
  @SyntheticMember
  private static final long serialVersionUID = 3200207935L;
}
