/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.core.AgentSpawnFailure;
import io.sarl.core.AgentSpawned;
import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.FiredEvent;
import io.sarl.lang.annotation.SarlAsynchronousExecution;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Agent;
import io.sarl.lang.core.AgentContext;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.core.Event;
import io.sarl.lang.core.EventSpace;
import io.sarl.lang.core.Scope;
import io.sarl.lang.core.Space;
import io.sarl.lang.core.SpaceID;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Gives access to the DefaultContext of the agent and common actions on its default space.
 * Defines a set of commonly accessed action performed on the Default Space
 * of the Default Context of the agent.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface DefaultContextInteractions extends Capacity {
  /**
   * Returns the Default context fo the agent.
   * 
   * @return the default context.
   */
  @Pure
  AgentContext getDefaultContext();
  
  /**
   * Returns the Default Space of the Default Context.
   * Equivalent to <code>defaultContext.defaultSpace</code>.
   * 
   * @return the default event space.
   */
  @Pure
  EventSpace getDefaultSpace();
  
  /**
   * Return the Address of the agent in the Default Space of the Default Context.
   * Equivalent to <code>defaultContext.defaultSpace.getAddress(this.ID)</code>
   * 
   * @return the address of the agent in the default space.
   */
  @Pure
  Address getDefaultAddress();
  
  /**
   * Emits a given event with the provided scope in the DefaultSpace of the DefaultContext.
   * Equivalent to <code>defaultContext.defaultSpace.emit(e,scope)</code>
   * 
   * @param event the event to emit.
   * @param scope the definition of the scope that will be used for selected the receivers of the events. If {@code null}, all the agents in the space will receive the event.
   */
  @DefaultValueSource
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void emit(final Event event, @DefaultValue("io.sarl.core.DefaultContextInteractions#EMIT_0") final Scope<Address> scope);
  
  /**
   * Replies the identifier of the default parent.
   * 
   * <p>The default parent agent has the same identifier as the default context.
   * 
   * @return the identifier of the default parent of the current agent.
   * @since 0.12
   */
  @Pure
  UUID getDefaultParentID();
  
  /**
   * Replies the event scope that corresponds to the default parent.
   * 
   * @return the identifier of the default parent of the current agent.
   * @since 0.12
   */
  @Pure
  Scope<Address> getDefaultParentScope();
  
  /**
   * Emits or forward the given event to the parent agent into the default context of the calling agent.
   * 
   * @param event the event to emit.
   * @param scope the definition of the scope that will be used for selected the receivers of the events. If {@code null}, all the agents in the space will receive the event.
   * @since 0.12
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void emitToParent(final Event event);
  
  /**
   * Inverse signature of send. Useful to send events using the agent's UUID.
   * 
   * @param receiver the identifier of the expected receiver.
   * @param event the event to emit.
   * @since 0.4
   * @deprecated since 0.11, see {@link #emit(Event,Scope)}
   */
  @Deprecated
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void willReceive(final UUID receiver, final Event event);
  
  /**
   * Replies if the given space is the default space of the default context.
   * 
   * @param space the space to test.
   * @return <code>true</code> if the given space is the default space of
   *     the default context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isDefaultSpace(final Space space);
  
  /**
   * Replies if the given identifier is the identifier of the default space of
   * the default context.
   * 
   * @param space the space to test.
   * @return <code>true</code> if the given identifier is the identifier
   *     of the default space of the default context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isDefaultSpace(final SpaceID space);
  
  /**
   * Replies if the given identifier is the identifier of the default space of
   * the default context.
   * 
   * @param space the space to test.
   * @return <code>true</code> if the given identifier is the identifier
   *     of the default space of the default context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isDefaultSpace(final UUID space);
  
  /**
   * Replies if the given event was emitted in the default space of
   * the default context.
   * 
   * @param event the event to test.
   * @return <code>true</code> if the given event is emitted in the default
   *     space of the default context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInDefaultSpace(final Event event);
  
  /**
   * Replies if the given context is the default context.
   * 
   * @param context the agent context to test.
   * @return <code>true</code> if the given context is the default context.
   *     Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isDefaultContext(final AgentContext context);
  
  /**
   * Replies if the given identifier is the identifier of the default context.
   * 
   * @param contextID the identifier of the context.
   * @return <code>true</code> if the given identifier is the identifier
   *     of the default context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isDefaultContext(final UUID contextID);
  
  /**
   * Spawns a new Agent inside the default context of this agent.
   * This action must automatically register the newly created agent
   * within the default space of the context.
   * 
   * @param agentType the type of the agent to spawn.
   * @param params the arguments to pass in the initialization event to the spawned agent.
   * @return the identifier of the spawned agent.
   * @fires AgentSpawned in DefaultSpace
   * @fires AgentSpawnFailure if the cannot be spawn.
   * @deprecated See the {@link Lifecycle} capacity.
   */
  @Deprecated
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  UUID spawn(final Class<? extends Agent> agentType, final Object... params);
  
  /**
   * Default value for the parameter scope
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Scope $DEFAULT_VALUE$EMIT_0() {
    return null;
  }
  
  /**
   * Emits a given event with the provided scope in the DefaultSpace of the DefaultContext.
   * Equivalent to <code>defaultContext.defaultSpace.emit(e,scope)</code>
   * 
   * @param event the event to emit.
   * @optionalparam scope the definition of the scope that will be used for selected the receivers of the events. If {@code null}, all the agents in the space will receive the event.
   */
  @DefaultValueUse("io.sarl.lang.core.Event,io.sarl.lang.core.Scope")
  @SyntheticMember
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  default void emit(final Event event) {
    emit(event, $DEFAULT_VALUE$EMIT_0());
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends DefaultContextInteractions> extends Capacity.ContextAwareCapacityWrapper<C> implements DefaultContextInteractions {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public AgentContext getDefaultContext() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefaultContext();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public EventSpace getDefaultSpace() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefaultSpace();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public Address getDefaultAddress() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefaultAddress();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void emit(final Event event, final Scope<Address> scope) {
      try {
        ensureCallerInLocalThread();
        this.capacity.emit(event, scope);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public UUID getDefaultParentID() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefaultParentID();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public Scope<Address> getDefaultParentScope() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefaultParentScope();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void emitToParent(final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.emitToParent(event);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void willReceive(final UUID receiver, final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.willReceive(receiver, event);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isDefaultSpace(final Space space) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDefaultSpace(space);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isDefaultSpace(final SpaceID space) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDefaultSpace(space);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isDefaultSpace(final UUID space) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDefaultSpace(space);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInDefaultSpace(final Event event) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInDefaultSpace(event);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isDefaultContext(final AgentContext context) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDefaultContext(context);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isDefaultContext(final UUID contextID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDefaultContext(contextID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public UUID spawn(final Class<? extends Agent> agentType, final Object... params) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.spawn(agentType, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void emit(final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.emit(event);
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
