/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.core.ContextJoined;
import io.sarl.core.ContextLeft;
import io.sarl.core.MemberJoined;
import io.sarl.core.MemberLeft;
import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.FiredEvent;
import io.sarl.lang.annotation.SarlAsynchronousExecution;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Agent;
import io.sarl.lang.core.AgentContext;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.core.Event;
import io.sarl.lang.core.EventSpace;
import io.sarl.lang.core.Scope;
import io.sarl.lang.core.Space;
import io.sarl.lang.core.SpaceID;
import io.sarl.lang.util.ConcurrentCollection;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Provides functions for accessing and managing the external contexts of an agent.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface ExternalContextAccess extends Capacity {
  /**
   * Replies all contexts this agent is a member of, including the default context.
   * 
   * @return the list of the known agent contexts.
   */
  @Pure
  ConcurrentCollection<AgentContext> getAllContexts();
  
  /**
   * Replies the AgentContext for the given ID.
   * The agent must have joined the context before calling this action or use its parentContextID
   * 
   * @param contextID the ID of the context to get.
   * @return the agent context instance.
   * @throws UnknownContextException if the context specified context is not known by the agent.
   * @see Agent#getParentID
   * @see #join
   */
  @Pure
  AgentContext getContext(final UUID contextID);
  
  /**
   * Replies the AgentContext that is the root of all the contexts.
   * Usually, the Universe context is managed by the SARL run-time environment.
   * The agent may be a member of this context, or not.
   * 
   * @return the context that is at the root of all the contexts.
   * @since 0.7
   */
  @Pure
  AgentContext getUniverseContext();
  
  /**
   * Joins a new parent context (a new super holon).
   * <p>
   * The agent will be involved in the context with the ID given by <var>contextID</var>.
   * The parameter <var>expectedDefaultSpaceID</var> is only used to check if
   * the caller of this function knows the ID of the default space in the context to
   * be involved in. Note that the context must already exists, and the default space
   * inside this context must have the same ID as <var>expectedDefaultSpaceID</var>.
   * If the given <var>expectedDefaultSpaceID</var> does not match the ID of the
   * default space in the context <var>contextID</var>, then the access to the context
   * is forbidden.
   * 
   * <p>This actions registers the agent in the default Space of the context with the
   * ID <var>contextID</var>.
   * 
   * <p>This function replies {@code false} if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive;</li>
   * <li>the given context identifier does not corresponds to a known context;</li>
   * <li>the given space identifier is not the one of the context's default space;</li>
   * <li>the given context identifier corresponds to a context in which the agent is already member.</li>
   * </ul>
   * 
   * @param contextID the identifier of the context to join.
   * @param expectedDefaultSpaceID the known identifier of the default space in the agent context with the given identifier.
   * @return the joined context. {@code null} if the context cannot be joined.
   * @fires ContextJoined in its inner Context default space (Behaviors#wake).
   * @fires MemberJoined in its parent Context default Space
   * @since 0.12 for the returned context.
   */
  @FiredEvent({ ContextJoined.class, MemberJoined.class })
  AgentContext join(final UUID contextID, final UUID expectedDefaultSpaceID);
  
  /**
   * Leaves the parent's context.
   * 
   * <p>Because an agent must be always into a context, this function fails (and replies {@code false}) when
   * the context to be leaved is the current default context of the agent, and there is no other context or
   * more than 1 other context that could be elected as the new default context.
   * 
   * <p>This function replies {@code false} if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive;</li>
   * <li>the given context identifier does not corresponds to a known context;</li>
   * <li>the given context identifier corresponds to a context in which the agent is not member.</li>
   * <li>the given context identifier corresponds to a known external context, but there is no other
   * context to be elected as the new default context for the agent.</li>
   * </ul>
   * 
   * @param contextID the identifier of the context to leave.
   * @return {@code true} if the agent has leaved the context. {@code false} if one of the conditions described in
   *     the function's comment evaluates to true.
   * @fires ContextLeft in its inner Context default space (Behaviors#wake).
   * @fires MemberLeft in its parent Context default Space
   */
  @FiredEvent({ ContextLeft.class, MemberLeft.class })
  boolean leave(final UUID contextID);
  
  /**
   * Replies if the given event was emitted in the given space.
   * 
   * @param event the event to test.
   * @param space the space in which the event may be.
   * @return <code>true</code> if the given event was emitted in the
   *     given space. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInSpace(final Event event, final Space space);
  
  /**
   * Replies if the given event was emitted in the space with
   * the given identifier..
   * 
   * @param event the event to test.
   * @param spaceID the identifier of the space in which the event may be.
   * @return <code>true</code> if the given event was emitted in the
   *     space with the given identifier. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInSpace(final Event event, final SpaceID spaceID);
  
  /**
   * Replies if the given event was emitted in the space with
   * the given identifier..
   * 
   * @param event the event to test.
   * @param spaceID the identifier of the space in which the event may be.
   * @return <code>true</code> if the given event was emitted in the
   *     space with the given identifier. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInSpace(final Event event, final UUID spaceID);
  
  /**
   * Emits a given event with the provided scope in the given space.
   * Equivalent to <code>space.emit(getID,event,scope)</code>
   * 
   * @param space the space in which the event should be fired.
   * @param event the event to emit.
   * @param scope the definition of the scope that will be used for selected the receivers of the events. If {@code null}, all the agents in the space will receive the event.
   * @since 0.6
   */
  @DefaultValueSource
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  void emit(final EventSpace space, final Event event, @DefaultValue("io.sarl.core.ExternalContextAccess#EMIT_0") final Scope<Address> scope);
  
  /**
   * Default value for the parameter scope
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Scope $DEFAULT_VALUE$EMIT_0() {
    return null;
  }
  
  /**
   * Emits a given event with the provided scope in the given space.
   * Equivalent to <code>space.emit(getID,event,scope)</code>
   * 
   * @param space the space in which the event should be fired.
   * @param event the event to emit.
   * @optionalparam scope the definition of the scope that will be used for selected the receivers of the events. If {@code null}, all the agents in the space will receive the event.
   * @since 0.6
   */
  @DefaultValueUse("io.sarl.lang.core.EventSpace,io.sarl.lang.core.Event,io.sarl.lang.core.Scope")
  @SyntheticMember
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  default void emit(final EventSpace space, final Event event) {
    emit(space, event, $DEFAULT_VALUE$EMIT_0());
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends ExternalContextAccess> extends Capacity.ContextAwareCapacityWrapper<C> implements ExternalContextAccess {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public ConcurrentCollection<AgentContext> getAllContexts() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getAllContexts();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentContext getContext(final UUID contextID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getContext(contextID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentContext getUniverseContext() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getUniverseContext();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentContext join(final UUID contextID, final UUID expectedDefaultSpaceID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.join(contextID, expectedDefaultSpaceID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean leave(final UUID contextID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.leave(contextID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInSpace(final Event event, final Space space) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInSpace(event, space);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInSpace(final Event event, final SpaceID spaceID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInSpace(event, spaceID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInSpace(final Event event, final UUID spaceID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInSpace(event, spaceID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void emit(final EventSpace space, final Event event, final Scope<Address> scope) {
      try {
        ensureCallerInLocalThread();
        this.capacity.emit(space, event, scope);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void emit(final EventSpace space, final Event event) {
      try {
        ensureCallerInLocalThread();
        this.capacity.emit(space, event);
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
