/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Event;
import java.util.Objects;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Initialization event.
 * The source of this event is the agent itself.
 * 
 * <h4>For an Agent</h4>
 * This event is received by the agent's native behavior when the agent is spawned.
 * 
 * <h4>For a Behavior</h4>
 * Each Behavior instance within the agent receives this event when it is registered for the first time.
 * 
 * <h4>For a Skill</h4>
 * No skill receives this event.
 * 
 * <h4>For Agent Members</h4>
 * The member agents do not receive this event from the parent agent because they are not yet created when the agent is spawned.
 * 
 * @author <a href="http://www.sebastianrodriguez.com.ar/">Sebastian Rodriguez</a>
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 */
@SarlSpecification("0.12")
@SarlElementType(15)
@SuppressWarnings("all")
public final class Initialize extends Event {
  /**
   * The initialization parameters.
   * Usually, they are the arguments passed to the agent spawning function.
   */
  public final Object[] parameters;
  
  /**
   * Identifier of the agent, which has spawned the under-initialization agent.
   * If the under-initialization agent was spawned by the SARL run-time environment
   * (and not by another agent), the identifier is {@code null}.
   * 
   * @since 0.5
   */
  public final UUID spawner;
  
  /**
   * Constructor.
   * 
   * @param spawner the identifier of the spawner of the under-initialization agent.
   * @param params the initialization parameters to give to the under-initialization agent.
   * @since 0.5
   */
  public Initialize(final UUID spawner, final Object... params) {
    this(null, spawner, params);
  }
  
  /**
   * Constructor.
   * 
   * @param source address of the agent that is emitting this event.
   * @param spawner the identifier of the spawner of the under-initialization agent.
   * @param params the initialization parameters to give to the under-initialization agent.
   * @since 0.5
   */
  public Initialize(final Address source, final UUID spawner, final Object... params) {
    super(source);
    this.spawner = spawner;
    this.parameters = params;
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    Initialize other = (Initialize) obj;
    if (!Objects.equals(this.spawner, other.spawner))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.spawner);
    return result;
  }
  
  /**
   * Returns a String representation of the Initialize event's attributes only.
   */
  @SyntheticMember
  @Pure
  protected void toString(final ToStringBuilder builder) {
    super.toString(builder);
    builder.add("parameters", this.parameters);
    builder.add("spawner", this.spawner);
  }
  
  @SyntheticMember
  private static final long serialVersionUID = -2191895051L;
}
