/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.core.AgentContext;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.core.Event;
import io.sarl.lang.core.EventSpace;
import io.sarl.lang.core.Space;
import io.sarl.lang.core.SpaceID;
import io.sarl.lang.util.ConcurrentSet;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Provides functions for accessing and managing the internal contexts of an agent.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface InnerContextAccess extends Capacity {
  /**
   * Replies the inner context.
   * 
   * @return the instance of the internal agent context.
   */
  @Pure
  AgentContext getInnerContext();
  
  /**
   * Replies the inner context.
   * 
   * @return the instanceof the internal agent context.
   * @since 0.12
   */
  @Pure
  EventSpace getInnerDefaultSpace();
  
  /**
   * Replies if the calling agent has other agents
   * as members of its inner context.
   * A member agent is an agent which is not the
   * calling agent, and is a member of at least
   * one space of the inner context.
   * 
   * @return <code>true</code> if the current agent has other agents as member of its internal context; otherwise <code>false</code>
   */
  @Pure
  boolean hasMemberAgent();
  
  /**
   * Replies the number of agents that are members
   * of the inner context of the calling agent.
   * A member agent is an agent which is not the
   * calling agent, and is a member of at least
   * one space of the inner context.
   * 
   * <p>The replied number includes the current agent. Indeed, the current agent is member of its internal context with the role of super-holon.
   * 
   * @return the number of members in the internal context of the current agent.
   */
  @Pure
  int getMemberAgentCount();
  
  /**
   * Replies all the member agents in the inner context.
   * A member agent is an agent which is not the
   * calling agent, and is a member of at least
   * one space of the inner context.
   * 
   * @return the identifiers of the members of the internal context of the current agent.
   */
  @Pure
  ConcurrentSet<UUID> getMemberAgents();
  
  /**
   * Replies if the given space is the default space of the inner context.
   * 
   * @param space the space to test.
   * @return <code>true</code> if the given space is the default space of
   *     the inner context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInnerDefaultSpace(final Space space);
  
  /**
   * Replies if the given identifier is the identifier of the
   * default space of the inner context.
   * 
   * @param spaceID the identifier to test.
   * @return <code>true</code> if the given identifier is the identifier
   *     of the default space of the inner context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInnerDefaultSpace(final SpaceID spaceID);
  
  /**
   * Replies if the given identifier is the identifier of the
   * default space of the inner context.
   * 
   * @param spaceID the identifier to test.
   * @return <code>true</code> if the given identifier is the identifier
   *     of the default space of the inner context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInnerDefaultSpace(final UUID spaceID);
  
  /**
   * Replies if the given event was emitted in the default space
   * of the inner context.
   * 
   * @param event the event to test.
   * @return <code>true</code> if the given event was emitted
   *     in the default space of the inner context. Otherwise <code>false</code>.
   * @since 0.2
   */
  @Pure
  boolean isInInnerDefaultSpace(final Event event);
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends InnerContextAccess> extends Capacity.ContextAwareCapacityWrapper<C> implements InnerContextAccess {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public AgentContext getInnerContext() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getInnerContext();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public EventSpace getInnerDefaultSpace() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getInnerDefaultSpace();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean hasMemberAgent() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.hasMemberAgent();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public int getMemberAgentCount() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getMemberAgentCount();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public ConcurrentSet<UUID> getMemberAgents() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getMemberAgents();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInnerDefaultSpace(final Space space) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInnerDefaultSpace(space);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInnerDefaultSpace(final SpaceID spaceID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInnerDefaultSpace(spaceID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInnerDefaultSpace(final UUID spaceID) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInnerDefaultSpace(spaceID);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInInnerDefaultSpace(final Event event) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInInnerDefaultSpace(event);
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
