/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.core.AgentKillFailure;
import io.sarl.core.AgentKilled;
import io.sarl.core.AgentSpawnFailure;
import io.sarl.core.AgentSpawned;
import io.sarl.core.Destroy;
import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.EarlyExit;
import io.sarl.lang.annotation.FiredEvent;
import io.sarl.lang.annotation.SarlAsynchronousExecution;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Agent;
import io.sarl.lang.core.AgentContext;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Lifecycle related actions.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface Lifecycle extends Capacity {
  /**
   * Spawns a new Agent inside the default context of this agent.
   * This action must automatically register the newly created agent
   * within the default space of the context.
   * 
   * <p>Caution: when this function returns, there is no warranty that the spawned agent
   * is initialized at the time of the return.
   * Indeed, agent spawning is done in parallel to the calling thread.
   * This parallel task for spawning is not considered as an agent task, i.e. it cannot
   * be stopped or cancelled by the spawning agent, or if this agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param agentType the type of the agent to spawn.
   * @param params the arguments to pass in the initialization event to the spawned agent.
   * @fires AgentSpawned in DefaultSpace
   * @fires AgentSpawnFailure if the agent cannot be spawned.
   * @since 0.5
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  void spawn(final Class<? extends Agent> agentType, final Object... params);
  
  /**
   * Spawns a group of agents inside the default context of this agent.
   * This action must automatically register the newly created agents
   * within the default space of the context.
   * 
   * <p>Caution: when this function returns, there is no warranty that the spawned agents
   * are initialized at the time of the return.
   * Indeed, agent spawning is done in parallel to the calling thread.
   * This parallel task for spawning is not considered as an agent task, i.e. it cannot
   * be stopped or cancelled by the spawning agent, or if this agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param nbAgents the number of agents to spawn.
   * @param agentType the type of the agents to spawn.
   * @param params the arguments to pass in the initialization event to the spawned agents. All the agents
   *     will receive the same initialization parameters.
   * @fires AgentSpawned in DefaultSpace
   * @fires AgentSpawnFailure if the agent cannot be spawned.
   * @since 0.5
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  void spawn(final int nbAgents, final Class<? extends Agent> agentType, final Object... params);
  
  /**
   * Spawns a new member agent in the parent's context (parentID).
   * 
   * <p>Caution: when this function returns, there is no warranty that the spawned agent
   * is initialized at the time of the return.
   * Indeed, agent spawning is done in parallel to the calling thread.
   * This parallel task for spawning is not considered as an agent task, i.e. it cannot
   * be stopped or cancelled by the spawning agent, or if this agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param agentClass the type of the agent to spawn.
   * @param context the context in which the agent must be spawned.
   * @param params the arguments to pass to the initialization event that will be sent to the spawned agent.
   * @fires AgentSpawned inside the default context of the parent. The source of the event is this agent.
   * @fires AgentSpawnFailure if the agent cannot be spawned.
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  void spawnInContext(final Class<? extends Agent> agentClass, final AgentContext context, final Object... params);
  
  /**
   * Spawns a group of agents in the parent's context (parentID).
   * 
   * <p>Caution: when this function returns, there is no warranty that the spawned agents
   * are initialized at the time of the return.
   * Indeed, agent spawning is done in parallel to the calling thread.
   * This parallel task for spawning is not considered as an agent task, i.e. it cannot
   * be stopped or cancelled by the spawning agent, or if this agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param nbAgents the number of agents to spawn.
   * @param agentClass the type of the agents to spawn.
   * @param context the context in which the agents must be spawned.
   * @param params the arguments to pass to the initialization event that will be sent to the spawned agents. All the agents
   *     will receive the same initialization parameters.
   * @fires AgentSpawned inside the default context of the parent. The source of the event is this agent.
   * @fires AgentSpawnFailure if the agent cannot be spawned.
   * @since 0.5
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  void spawnInContext(final int nbAgents, final Class<? extends Agent> agentClass, final AgentContext context, final Object... params);
  
  /**
   * Spawns a new member agent in the given context.
   * 
   * <p>Caution: when this function returns, there is no warranty that the spawned agent
   * is initialized at the time of the return.
   * Indeed, agent spawning is done in parallel to the calling thread.
   * This parallel task for spawning is not considered as an agent task, i.e. it cannot
   * be stopped or cancelled by the spawning agent, or if this agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param agentClass the type of the agent to spawn.
   * @param agentID the identifier of the spawned agent.
   * @param context the context in which the agent must be spawned.
   * @param params the arguments to pass to the initialization event that will be sent to the spawned agent.
   * @fires AgentSpawned inside the default context of the parent. The source of the event is this agent.
   * @fires AgentSpawnFailure if the agent cannot be spawned.
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  void spawnInContextWithID(final Class<? extends Agent> agentClass, final UUID agentID, final AgentContext context, final Object... params);
  
  /**
   * Spawns a new member agent in the parent's context (parentID).
   * 
   * <p>Caution: when this function returns, there is no warranty that the spawned agent
   * is initialized at the time of the return.
   * Indeed, agent spawning is done in parallel to the calling thread.
   * This parallel task for spawning is not considered as an agent task, i.e. it cannot
   * be stopped or cancelled by the spawning agent, or if this agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param agentClass the type of the agent to spawn.
   * @param agentID the identifier of the spawned agent.
   * @param params the arguments to pass to the initialization event that will be sent to the spawned agent.
   * @fires AgentSpawned inside the default context of the parent. The source of the event is this agent.
   * @fires AgentSpawnFailure if the agent cannot be spawned.
   * @since 0.12
   */
  @SuppressWarnings("use_reserved_sarl_annotation")
  @SarlAsynchronousExecution
  @FiredEvent({ AgentSpawned.class, AgentSpawnFailure.class })
  void spawnWithID(final Class<? extends Agent> agentClass, final UUID agentID, final Object... params);
  
  /**
   * Kills this agent.
   * 
   * <p>This action must automatically unregister this agent from the default context
   * and therefore all its spaces including the DefaultSpace.
   * If this is a composable agent, it must not have any members before calling this action.
   * Otherwise a RuntimeException will be thrown.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param terminationCause indicates the cause of the termination of the agent.
   *      If this argument is {@code null}, the agent terminates without specific cause.
   * @fires AgentKilled in DefaultSpace of all Contexts to which this agent belongs
   * @fires Destroy inside the agent
   * @fires AgentKillFailure inside the agent
   * @since 0.12
   */
  @DefaultValueSource
  @EarlyExit
  @FiredEvent({ AgentKilled.class, Destroy.class, AgentKillFailure.class })
  void killMe(@DefaultValue("io.sarl.core.Lifecycle#KILLME_0") final Object terminationCause);
  
  /**
   * Default value for the parameter terminationCause
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Object $DEFAULT_VALUE$KILLME_0() {
    return null;
  }
  
  /**
   * Kills this agent.
   * 
   * <p>This action must automatically unregister this agent from the default context
   * and therefore all its spaces including the DefaultSpace.
   * If this is a composable agent, it must not have any members before calling this action.
   * Otherwise a RuntimeException will be thrown.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @optionalparam terminationCause indicates the cause of the termination of the agent.
   *      If this argument is {@code null}, the agent terminates without specific cause.
   * @fires AgentKilled in DefaultSpace of all Contexts to which this agent belongs
   * @fires Destroy inside the agent
   * @fires AgentKillFailure inside the agent
   * @since 0.12
   */
  @DefaultValueUse("java.lang.Object")
  @SyntheticMember
  @EarlyExit
  @FiredEvent({ AgentKilled.class, Destroy.class, AgentKillFailure.class })
  default void killMe() {
    killMe($DEFAULT_VALUE$KILLME_0());
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends Lifecycle> extends Capacity.ContextAwareCapacityWrapper<C> implements Lifecycle {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public void spawn(final Class<? extends Agent> agentType, final Object... params) {
      try {
        ensureCallerInLocalThread();
        this.capacity.spawn(agentType, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void spawn(final int nbAgents, final Class<? extends Agent> agentType, final Object... params) {
      try {
        ensureCallerInLocalThread();
        this.capacity.spawn(nbAgents, agentType, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void spawnInContext(final Class<? extends Agent> agentClass, final AgentContext context, final Object... params) {
      try {
        ensureCallerInLocalThread();
        this.capacity.spawnInContext(agentClass, context, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void spawnInContext(final int nbAgents, final Class<? extends Agent> agentClass, final AgentContext context, final Object... params) {
      try {
        ensureCallerInLocalThread();
        this.capacity.spawnInContext(nbAgents, agentClass, context, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void spawnInContextWithID(final Class<? extends Agent> agentClass, final UUID agentID, final AgentContext context, final Object... params) {
      try {
        ensureCallerInLocalThread();
        this.capacity.spawnInContextWithID(agentClass, agentID, context, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void spawnWithID(final Class<? extends Agent> agentClass, final UUID agentID, final Object... params) {
      try {
        ensureCallerInLocalThread();
        this.capacity.spawnWithID(agentClass, agentID, params);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void killMe(final Object terminationCause) {
      try {
        ensureCallerInLocalThread();
        this.capacity.killMe(terminationCause);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void killMe() {
      try {
        ensureCallerInLocalThread();
        this.capacity.killMe();
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
