/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import java.text.MessageFormat;
import java.util.function.Supplier;
import java.util.logging.Logger;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Gives access to the logging mechanism associated to the agent.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface Logging extends Capacity {
  /**
   * Change the name used for logging.
   * 
   * @param name the name used for logging.
   */
  void setLoggingName(final String name);
  
  /**
   * Print the given message at information logging level.
   * <p>
   * <strong>This function is deprecated.</strong> Please use
   * {@link #info(Object,Object...)} in place of <code>println</code>.
   * 
   * @param message the elements to display.
   * @deprecated Use {@link #info(Object,Object...)}.
   */
  @Deprecated
  void println(final Object message);
  
  /**
   * Print the given message at error logging level.
   * 
   * @param message the elements to display. The format of the message must follow {@link MessageFormat}.
   * @param exception the exception that is the cause of the error.
   * @param parameters the values to dynamically insert into the message.
   */
  @DefaultValueSource
  void error(final Object message, @DefaultValue("io.sarl.core.Logging#ERROR_0") final Throwable exception, final Object... parameters);
  
  /**
   * Print the given message at error logging level.
   * 
   * @param messageProvider provides the error message.
   * @since 0.7
   */
  void error(final Supplier<String> messageProvider);
  
  /**
   * Print the given message at warning logging level.
   * 
   * @param message the elements to display. The format of the message must follow {@link MessageFormat}.
   * @param exception the exception that is the cause of the error.
   * @param parameters the values to dynamically insert into the message.
   */
  @DefaultValueSource
  void warning(final Object message, @DefaultValue("io.sarl.core.Logging#WARNING_0") final Throwable exception, final Object... parameters);
  
  /**
   * Print the given message at warning logging level.
   * 
   * @param messageProvider provides the warning message.
   * @since 0.7
   */
  void warning(final Supplier<String> messageProvider);
  
  /**
   * Print the given message at information logging level.
   * 
   * @param message the elements to display. The format of the message must follow {@link MessageFormat}.
   * @param parameters the values to dynamically insert into the message.
   */
  void info(final Object message, final Object... parameters);
  
  /**
   * Print the given message at information logging level.
   * 
   * @param messageProvider provides the info message.
   * @since 0.7
   */
  void info(final Supplier<String> messageProvider);
  
  /**
   * Print the given message at debug logging level.
   * 
   * @param message the elements to display. The format of the message must follow {@link MessageFormat}.
   * @param parameters the values to dynamically insert into the message.
   */
  void debug(final Object message, final Object... parameters);
  
  /**
   * Print the given message at debug logging level.
   * 
   * @param messageProvider provides the debug message.
   * @since 0.7
   */
  void debug(final Supplier<String> messageProvider);
  
  /**
   * Replies if the logging system is displaying the errors.
   * 
   * @return <code>true</code> if the errors are logged,
   *         <code>false</code> if not.
   */
  @Pure
  boolean isErrorLogEnabled();
  
  /**
   * Replies if the logging system is displaying the warnings.
   * 
   * @return <code>true</code> if the warnings are logged,
   *         <code>false</code> if not.
   */
  @Pure
  boolean isWarningLogEnabled();
  
  /**
   * Replies if the logging system is displaying the information messages.
   * 
   * @return <code>true</code> if the information messages are logged,
   *         <code>false</code> if not.
   */
  @Pure
  boolean isInfoLogEnabled();
  
  /**
   * Replies if the logging system is displaying the debugging messages.
   * 
   * @return <code>true</code> if the debugging messages are logged,
   *         <code>false</code> if not.
   */
  @Pure
  boolean isDebugLogEnabled();
  
  /**
   * Replies the logging level.
   * 
   * @return <code>0</code> for no logging, <code>1</code> for error,
   *         <code>2</code> for warning, <code>3</code> for info,
   *         <code>5</code> for debug.
   */
  @Pure
  int getLogLevel();
  
  /**
   * Set the logging level.
   * 
   * @param level <code>0</code> or lower for no logging, <code>1</code> for error,
   *         <code>2</code> for warning, <code>3</code> for info,
   *         <code>4</code> or higher for debug.
   */
  void setLogLevel(final int level);
  
  /**
   * Replies the backend logger.
   * 
   * @return the backend logger.
   */
  @Pure
  Logger getLogger();
  
  /**
   * Default value for the parameter exception
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Throwable $DEFAULT_VALUE$ERROR_0() {
    return null;
  }
  
  /**
   * Default value for the parameter exception
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default Throwable $DEFAULT_VALUE$WARNING_0() {
    return null;
  }
  
  /**
   * Print the given message at error logging level.
   * 
   * @param message the elements to display. The format of the message must follow {@link MessageFormat}.
   * @optionalparam exception the exception that is the cause of the error.
   * @param parameters the values to dynamically insert into the message.
   */
  @DefaultValueUse("java.lang.Object,java.lang.Throwable,java.lang.Object*")
  @SyntheticMember
  default void error(final Object message, final Object... parameters) {
    error(message, $DEFAULT_VALUE$ERROR_0(), parameters);
  }
  
  /**
   * Print the given message at warning logging level.
   * 
   * @param message the elements to display. The format of the message must follow {@link MessageFormat}.
   * @optionalparam exception the exception that is the cause of the error.
   * @param parameters the values to dynamically insert into the message.
   */
  @DefaultValueUse("java.lang.Object,java.lang.Throwable,java.lang.Object*")
  @SyntheticMember
  default void warning(final Object message, final Object... parameters) {
    warning(message, $DEFAULT_VALUE$WARNING_0(), parameters);
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends Logging> extends Capacity.ContextAwareCapacityWrapper<C> implements Logging {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public void setLoggingName(final String name) {
      try {
        ensureCallerInLocalThread();
        this.capacity.setLoggingName(name);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void println(final Object message) {
      try {
        ensureCallerInLocalThread();
        this.capacity.println(message);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void error(final Object message, final Throwable exception, final Object... parameters) {
      try {
        ensureCallerInLocalThread();
        this.capacity.error(message, exception, parameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void error(final Supplier<String> messageProvider) {
      try {
        ensureCallerInLocalThread();
        this.capacity.error(messageProvider);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void warning(final Object message, final Throwable exception, final Object... parameters) {
      try {
        ensureCallerInLocalThread();
        this.capacity.warning(message, exception, parameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void warning(final Supplier<String> messageProvider) {
      try {
        ensureCallerInLocalThread();
        this.capacity.warning(messageProvider);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void info(final Object message, final Object... parameters) {
      try {
        ensureCallerInLocalThread();
        this.capacity.info(message, parameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void info(final Supplier<String> messageProvider) {
      try {
        ensureCallerInLocalThread();
        this.capacity.info(messageProvider);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void debug(final Object message, final Object... parameters) {
      try {
        ensureCallerInLocalThread();
        this.capacity.debug(message, parameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void debug(final Supplier<String> messageProvider) {
      try {
        ensureCallerInLocalThread();
        this.capacity.debug(messageProvider);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isErrorLogEnabled() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isErrorLogEnabled();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isWarningLogEnabled() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isWarningLogEnabled();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isInfoLogEnabled() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isInfoLogEnabled();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isDebugLogEnabled() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDebugLogEnabled();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public int getLogLevel() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getLogLevel();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void setLogLevel(final int level) {
      try {
        ensureCallerInLocalThread();
        this.capacity.setLogLevel(level);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public Logger getLogger() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getLogger();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void error(final Object message, final Object... parameters) {
      try {
        ensureCallerInLocalThread();
        this.capacity.error(message, parameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void warning(final Object message, final Object... parameters) {
      try {
        ensureCallerInLocalThread();
        this.capacity.warning(message, parameters);
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
