/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Event;
import java.util.Objects;
import java.util.UUID;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Notifies the other other members that the member with the identifier agentID was
 * excluded (forced to leave) the holon context parentContextID.
 * The event is fired within the default space of the context from which the agent was excluded.
 * The excluded agent does not receive this event.
 * The source of the event is the parent agent.
 * The context id can be retrieved via {@code ev.source.spaceID.contextID}.
 * 
 * @author <a href="http://www.sebastianrodriguez.com.ar/">Sebastian Rodriguez</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 */
@SarlSpecification("0.12")
@SarlElementType(15)
@SuppressWarnings("all")
public final class MemberExcluded extends Event {
  /**
   * The identifier of the agent that was excluded from the context.
   */
  public final UUID agentID;
  
  /**
   * The fully qualified name of the type of the agent.
   */
  public final String agentType;
  
  /**
   * Construct the event.
   * 
   * @param source the address of the emitter.
   * @param agentID the identifier of the agent.
   * @param agentType fully qualified name of the type of the agent.
   */
  public MemberExcluded(final Address source, final UUID agentID, final String agentType) {
    this.setSource(source);
    this.agentID = agentID;
    this.agentType = agentType;
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    MemberExcluded other = (MemberExcluded) obj;
    if (!Objects.equals(this.agentID, other.agentID))
      return false;
    if (!Objects.equals(this.agentType, other.agentType))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.agentID);
    result = prime * result + Objects.hashCode(this.agentType);
    return result;
  }
  
  /**
   * Returns a String representation of the MemberExcluded event's attributes only.
   */
  @SyntheticMember
  @Pure
  protected void toString(final ToStringBuilder builder) {
    super.toString(builder);
    builder.add("agentID", this.agentID);
    builder.add("agentType", this.agentType);
  }
  
  @SyntheticMember
  private static final long serialVersionUID = -1556686477L;
}
