/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.core.ParticipantJoined;
import io.sarl.core.ParticipantLeft;
import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.FiredEvent;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.annotation.WarningOnCall;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.EventListener;
import io.sarl.lang.core.EventSpace;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Event driven interaction space where participants (agents, etc) are free register and unregister themselves.
 * A participant should only register once in this type of space.
 * 
 * <p>A participant is a software entity that is registered in order to receive the events that are emitted into
 * the space. A participant is usually an agent, but not restricted to this type of software entity.
 * It could be also a user-interface module.
 * 
 * <p>A weak participant is participant that could be considered as not a first-order member of the space.
 * When a participant is weak, it is not considered by functions such as {@link #isPseudoEmpty(java.lang.UUID)}.
 * 
 * @author <a href="http://www.sebastianrodriguez.com.ar/">Sebastian Rodriguez</a>
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 */
@SarlSpecification("0.12")
@SarlElementType(11)
@SuppressWarnings("all")
public interface OpenEventSpace extends EventSpace {
  /**
   * Registers the participant inside this space.
   * After registering a new participant, the Space should emit a ParticipantJoined
   * event where the source is the address of the newly registered agent.
   * 
   * <p>If the agent is already registered the address is return, but the listener is not replaced.
   * 
   * @param participant the participant to register.
   * @return the entity's address in this space
   * @fires ParticipantJoined in its enclosing Context default space.
   * @deprecated since 0.12, see {@link #registerStrongParticipant(EventListener)} and
   *     {@link #registerStrongParticipant(EventListener).
   */
  @DefaultValueSource
  @SuppressWarnings("use_reserved_sarl_annotation")
  @WarningOnCall("Prefer calling registerStrongParticipant() or registerWeakParticipant()")
  @Deprecated
  @FiredEvent(ParticipantJoined.class)
  default Address register(final EventListener participant, @DefaultValue("io.sarl.core.OpenEventSpace#REGISTER_0") final boolean weakParticipant) {
    Address _xifexpression = null;
    if (weakParticipant) {
      _xifexpression = this.registerWeakParticipant(participant);
    } else {
      _xifexpression = this.registerStrongParticipant(participant);
    }
    return _xifexpression;
  }
  
  /**
   * Registers the strong participant inside this space.
   * A Weak participant will not prevent the space from begin destroyed if it is the only one staying in it, a Strong participant will prevent the space destruction.
   * A space containing only Weak participants will be destroyed by the SRE Kernel.
   * After registering a new participant, the Space should emit a ParticipantJoined
   * event where the source is the address of the newly registered agent.
   * 
   * <p>If the agent is already registered the address is return, but the listener is not replaced.
   * 
   * @param participant the participant to register.
   * @return the entity's address in this space
   * @fires ParticipantJoined in its enclosing Context default space.
   * @since 0.12
   */
  @FiredEvent(ParticipantJoined.class)
  Address registerStrongParticipant(final EventListener participant);
  
  /**
   * Registers the weak participant inside this space.
   * A Weak participant will not prevent the space from begin destroyed if it is the only one staying in it, a Strong participant will prevent the space destruction.
   * A space containing only Weak participants will be destroyed by the SRE Kernel.
   * After registering a new participant, the Space should emit a ParticipantJoined
   * event where the source is the address of the newly registered agent.
   * 
   * <p>If the agent is already registered the address is return, but the listener is not replaced.
   * 
   * @param participant the participant to register.
   * @return the entity's address in this space
   * @fires ParticipantJoined in its enclosing Context default space.
   * @since 0.12
   */
  @FiredEvent(ParticipantJoined.class)
  Address registerWeakParticipant(final EventListener participant);
  
  /**
   * Unregisters the entity inside this space.
   * After unregistering an agent, the Space should emit a ParticipantLeft
   * event where the source is the address of the unregistered agent.
   * 
   * @param participant the participant to unregister.
   * @return the former entity's address
   * @fires ParticipantJoined in its enclosing Context default space.
   */
  @FiredEvent(ParticipantLeft.class)
  Address unregister(final EventListener participant);
  
  /**
   * Default value for the parameter weakParticipant
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("false")
  default boolean $DEFAULT_VALUE$REGISTER_0() {
    return false;
  }
  
  /**
   * Registers the participant inside this space.
   * After registering a new participant, the Space should emit a ParticipantJoined
   * event where the source is the address of the newly registered agent.
   * 
   * <p>If the agent is already registered the address is return, but the listener is not replaced.
   * 
   * @param participant the participant to register.
   * @return the entity's address in this space
   * @fires ParticipantJoined in its enclosing Context default space.
   * @deprecated since 0.12, see {@link #registerStrongParticipant(EventListener)} and
   *     {@link #registerStrongParticipant(EventListener).
   */
  @DefaultValueUse("io.sarl.lang.core.EventListener,boolean")
  @SyntheticMember
  @FiredEvent(ParticipantJoined.class)
  @SuppressWarnings("use_reserved_sarl_annotation")
  @WarningOnCall("Prefer calling registerStrongParticipant() or registerWeakParticipant()")
  @Deprecated
  default Address register(final EventListener participant) {
    return register(participant, $DEFAULT_VALUE$REGISTER_0());
  }
}
