/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.core.AgentTask;
import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Agent;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.util.ConcurrentSet;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Schedules actions in time.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface Schedules extends Capacity {
  /**
   * Replies the names of the active tasks.
   * 
   * @return the names of the active tasks.
   * @since 0.5
   */
  @Pure
  ConcurrentSet<String> getActiveTasks();
  
  /**
   * Schedule a given task to be executed after the specified delay.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given delay is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>If this function is invoked from a {@code Behavior} and there is no provided task,
   * the created task will be associated to the behavior instance. It means that the task will
   * be automatically canceled when the behavior instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param task the task that will run the given closure. If {@code null}, a new task is created.
   * @param delay time in milliseconds to delay the procedure execution.
   * @param procedure the closure to execute.
   * @return the generated task.
   */
  @DefaultValueSource
  AgentTask in(@DefaultValue("io.sarl.core.Schedules#IN_0") final AgentTask task, final long delay, final Procedure1<? super Agent> procedure);
  
  /**
   * Create a named task that can be retrieved and schedule later.
   * If a task with the specified name already exists, this task is returned.
   * 
   * <p>If this function is invoked from a {@code Behavior}, the created task will be associated
   * to the behavior instance. It means that the task will be automatically canceled when the behavior
   * instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param name the name of the task. If {@code null} or empty, a random name will be given to the task.
   * @return the task instance, or {@code null} if one of the conditions in the function's comment
   *      evaluates to true.
   */
  AgentTask task(final String name);
  
  /**
   * Change the name of the given task.
   * 
   * <p>This function should not be directly invoked from a regular SARL code.
   * It is provided for enabling a SRE to change the name of a task.
   * 
   * @param task the task to change.
   * @param name the new name of the task.
   * @since 0.7
   */
  void setName(final AgentTask task, final String name);
  
  /**
   * Attempts to cancel execution of this task.  This attempt will
   * fail if the task has already completed, has already been canceled,
   * or could not be canceled for some other reason. If successful,
   * and this task has not started when <code>cancel</code> is called,
   * this task should never run.  If the task has already started,
   * then the <code>mayInterruptIfRunning</code> parameter determines
   * whether the thread executing this task should be interrupted in
   * an attempt to stop the task.
   * 
   * @param task the task to cancel.
   * @param mayInterruptIfRunning <code>true</code> if the thread executing this
   *     task should be interrupted; otherwise, in-progress tasks are allowed
   *     to complete
   * @return <code>false</code> if the task could not be canceled,
   *     typically because it has already completed normally;
   *     <code>true</code> otherwise
   */
  @DefaultValueSource
  boolean cancel(final AgentTask task, @DefaultValue("io.sarl.core.Schedules#CANCEL_0") final boolean mayInterruptIfRunning);
  
  /**
   * Replies if the given task was canceled.
   * 
   * @param task the task.
   * @return <code>true</code> if the task was canceled with {@link #cancel(AgentTask,boolean)}
   *     or {@link #cancel(AgentTask)}; <code>false</code> otherwise.
   * @since 0.5
   */
  @Pure
  boolean isCanceled(final AgentTask task);
  
  /**
   * Schedule a periodic execution of the given task.
   * 
   * <p>If the duration of the task is greater to the given period length, then
   * multiple task's instances will be run in parallel, in opposite to the
   * {@code atFixedDelay()} function.
   * For example, consider the following code:
   * <pre><code>
   * every(500) [ sleep(2000) ]
   * </code></pre>
   * At a given time, 4 instances (A, B, C, D) of the task may be run in parallel:
   * <pre><code>
   * t=0   0500   1000   1500   2000   2500   3000   3500   4000   4500
   *   |    |      |      |      |      |      |      |      |      |
   *   [-A-----------------------]
   *        [-B-------------------------]
   *               [-C-------------------------]
   *                      [-D-------------------------]
   *                             [-E-------------------------]
   *                                    [-F-------------------------]
   * </code></pre>
   * For executing a task with a fixed delay between the runs, and not at a fixed rate,
   * you should use the {@code atFixedDelay()} function.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given period is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>If this function is invoked from a {@code Behavior} and there is no provided task,
   * the created task will be associated to the behavior instance. It means that the task will
   * be automatically canceled when the behavior instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param task the task to associate to the procedure. If {@code null} a new task is created.
   * @param period the number of milliseconds between two launches of the given procedure.
   * @param procedure the procedure to launch. The parameter of the procedure is the agent.
   * @return the given task.
   */
  @DefaultValueSource
  AgentTask every(@DefaultValue("io.sarl.core.Schedules#EVERY_0") final AgentTask task, final long period, final Procedure1<? super Agent> procedure);
  
  /**
   * Schedule a single-execution task.
   * 
   * <p>If the duration of the task is greater to the given delay length, then
   * no multiple task's instance are run in parallel, in opposite to the
   * {@code every()} function.
   * For example, consider the following code:
   * <pre><code>
   * atFixedDelay(500) [ sleep(2000) ]
   * </code></pre>
   * At a given time, 3 instances (A, B, C) of the task are run in sequence, and
   * each run is separated by 500 milliseconds:
   * <pre><code>
   * t=0   0500   1000   1500   2000   2500   3000   3500   4000   4500   5000
   *   |    |      |      |      |      |      |      |      |      |      |
   *   [-A-----------------------]
   *                                    [-B-------------------------]
   *                                                                       [-C-------------------------]
   * </code></pre>
   * For executing a task with a fixed rate, and not with a fixed delay between the task runs,
   * you should use the {@code every()} function.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given delay is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>It is recommended that the SARL run-time environment, which is providing the implementation
   * of this function to provide an efficient implementation in the case the argument
   * {@code delay} is equal to zero. Indeed, if the {@code delay} is equal to zero, the task
   * should be run in an infinite loop until it is canceled, or the owning agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param task the task to associate to the procedure. If {@code null} a new task is created.
   * @param delay the delay in milliseconds.
   * @param procedure the procedure to launch. The parameter of the procedure is the agent.
   * @return the given task.
   * @since 0.5
   */
  @DefaultValueSource
  AgentTask atFixedDelay(@DefaultValue("io.sarl.core.Schedules#ATFIXEDDELAY_0") final AgentTask task, final long delay, final Procedure1<? super Agent> procedure);
  
  /**
   * Schedule a single-execution task.
   * 
   * <p>This function is the easy to use version and efficient implementation of the code
   * {@code in(0) [statements]}.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>This function is supposed to execute the given procedure, even if the agent is not alive.
   * 
   * @param task the task to associate to the procedure. If {@code null} a new task is created.
   * @param procedure the procedure to launch. The parameter of the procedure is the agent.
   * @return the given task.
   * @since 0.5
   */
  @DefaultValueSource
  AgentTask execute(@DefaultValue("io.sarl.core.Schedules#EXECUTE_0") final AgentTask task, final Procedure1<? super Agent> procedure);
  
  /**
   * Schedule a given task to be executed at the given time.
   * 
   * <p>If the given time is passed, according to {@link Time} then the task is not executed.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given time is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>If this function is invoked from a {@code Behavior} and there is no provided task,
   * the created task will be associated to the behavior instance. It means that the task will
   * be automatically canceled when the behavior instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @param task the task that will run the given closure. If {@code null}, a new task is created.
   * @param time the time in milliseconds at which to start the procedure execution.
   * @param procedure the closure to execute.
   * @return the task is given, or a new task if the procedure is schedule, or
   *     {@code null} if the procedure is not scheduled.
   * @since 0.9
   */
  @DefaultValueSource
  AgentTask at(@DefaultValue("io.sarl.core.Schedules#AT_0") final AgentTask task, final long time, final Procedure1<? super Agent> procedure);
  
  /**
   * Default value for the parameter task
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default AgentTask $DEFAULT_VALUE$IN_0() {
    return null;
  }
  
  /**
   * Default value for the parameter mayInterruptIfRunning
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("true")
  default boolean $DEFAULT_VALUE$CANCEL_0() {
    return true;
  }
  
  /**
   * Default value for the parameter task
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default AgentTask $DEFAULT_VALUE$EVERY_0() {
    return null;
  }
  
  /**
   * Default value for the parameter task
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default AgentTask $DEFAULT_VALUE$ATFIXEDDELAY_0() {
    return null;
  }
  
  /**
   * Default value for the parameter task
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default AgentTask $DEFAULT_VALUE$EXECUTE_0() {
    return null;
  }
  
  /**
   * Default value for the parameter task
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default AgentTask $DEFAULT_VALUE$AT_0() {
    return null;
  }
  
  /**
   * Schedule a given task to be executed after the specified delay.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given delay is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>If this function is invoked from a {@code Behavior} and there is no provided task,
   * the created task will be associated to the behavior instance. It means that the task will
   * be automatically canceled when the behavior instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @optionalparam task the task that will run the given closure. If {@code null}, a new task is created.
   * @param delay time in milliseconds to delay the procedure execution.
   * @param procedure the closure to execute.
   * @return the generated task.
   */
  @DefaultValueUse("io.sarl.core.AgentTask,long,(io.sarl.lang.core.Agent)=>void")
  @SyntheticMember
  default AgentTask in(final long delay, final Procedure1<? super Agent> procedure) {
    return in($DEFAULT_VALUE$IN_0(), delay, procedure);
  }
  
  /**
   * Attempts to cancel execution of this task.  This attempt will
   * fail if the task has already completed, has already been canceled,
   * or could not be canceled for some other reason. If successful,
   * and this task has not started when <code>cancel</code> is called,
   * this task should never run.  If the task has already started,
   * then the <code>mayInterruptIfRunning</code> parameter determines
   * whether the thread executing this task should be interrupted in
   * an attempt to stop the task.
   * 
   * @param task the task to cancel.
   * @optionalparam mayInterruptIfRunning <code>true</code> if the thread executing this
   *     task should be interrupted; otherwise, in-progress tasks are allowed
   *     to complete
   * @return <code>false</code> if the task could not be canceled,
   *     typically because it has already completed normally;
   *     <code>true</code> otherwise
   */
  @DefaultValueUse("io.sarl.core.AgentTask,boolean")
  @SyntheticMember
  default boolean cancel(final AgentTask task) {
    return cancel(task, $DEFAULT_VALUE$CANCEL_0());
  }
  
  /**
   * Schedule a periodic execution of the given task.
   * 
   * <p>If the duration of the task is greater to the given period length, then
   * multiple task's instances will be run in parallel, in opposite to the
   * {@code atFixedDelay()} function.
   * For example, consider the following code:
   * <pre><code>
   * every(500) [ sleep(2000) ]
   * </code></pre>
   * At a given time, 4 instances (A, B, C, D) of the task may be run in parallel:
   * <pre><code>
   * t=0   0500   1000   1500   2000   2500   3000   3500   4000   4500
   *   |    |      |      |      |      |      |      |      |      |
   *   [-A-----------------------]
   *        [-B-------------------------]
   *               [-C-------------------------]
   *                      [-D-------------------------]
   *                             [-E-------------------------]
   *                                    [-F-------------------------]
   * </code></pre>
   * For executing a task with a fixed delay between the runs, and not at a fixed rate,
   * you should use the {@code atFixedDelay()} function.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given period is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>If this function is invoked from a {@code Behavior} and there is no provided task,
   * the created task will be associated to the behavior instance. It means that the task will
   * be automatically canceled when the behavior instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @optionalparam task the task to associate to the procedure. If {@code null} a new task is created.
   * @param period the number of milliseconds between two launches of the given procedure.
   * @param procedure the procedure to launch. The parameter of the procedure is the agent.
   * @return the given task.
   */
  @DefaultValueUse("io.sarl.core.AgentTask,long,(io.sarl.lang.core.Agent)=>void")
  @SyntheticMember
  default AgentTask every(final long period, final Procedure1<? super Agent> procedure) {
    return every($DEFAULT_VALUE$EVERY_0(), period, procedure);
  }
  
  /**
   * Schedule a single-execution task.
   * 
   * <p>If the duration of the task is greater to the given delay length, then
   * no multiple task's instance are run in parallel, in opposite to the
   * {@code every()} function.
   * For example, consider the following code:
   * <pre><code>
   * atFixedDelay(500) [ sleep(2000) ]
   * </code></pre>
   * At a given time, 3 instances (A, B, C) of the task are run in sequence, and
   * each run is separated by 500 milliseconds:
   * <pre><code>
   * t=0   0500   1000   1500   2000   2500   3000   3500   4000   4500   5000
   *   |    |      |      |      |      |      |      |      |      |      |
   *   [-A-----------------------]
   *                                    [-B-------------------------]
   *                                                                       [-C-------------------------]
   * </code></pre>
   * For executing a task with a fixed rate, and not with a fixed delay between the task runs,
   * you should use the {@code every()} function.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given delay is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>It is recommended that the SARL run-time environment, which is providing the implementation
   * of this function to provide an efficient implementation in the case the argument
   * {@code delay} is equal to zero. Indeed, if the {@code delay} is equal to zero, the task
   * should be run in an infinite loop until it is canceled, or the owning agent is killed.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @optionalparam task the task to associate to the procedure. If {@code null} a new task is created.
   * @param delay the delay in milliseconds.
   * @param procedure the procedure to launch. The parameter of the procedure is the agent.
   * @return the given task.
   * @since 0.5
   */
  @DefaultValueUse("io.sarl.core.AgentTask,long,(io.sarl.lang.core.Agent)=>void")
  @SyntheticMember
  default AgentTask atFixedDelay(final long delay, final Procedure1<? super Agent> procedure) {
    return atFixedDelay($DEFAULT_VALUE$ATFIXEDDELAY_0(), delay, procedure);
  }
  
  /**
   * Schedule a single-execution task.
   * 
   * <p>This function is the easy to use version and efficient implementation of the code
   * {@code in(0) [statements]}.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>This function is supposed to execute the given procedure, even if the agent is not alive.
   * 
   * @optionalparam task the task to associate to the procedure. If {@code null} a new task is created.
   * @param procedure the procedure to launch. The parameter of the procedure is the agent.
   * @return the given task.
   * @since 0.5
   */
  @DefaultValueUse("io.sarl.core.AgentTask,(io.sarl.lang.core.Agent)=>void")
  @SyntheticMember
  default AgentTask execute(final Procedure1<? super Agent> procedure) {
    return execute($DEFAULT_VALUE$EXECUTE_0(), procedure);
  }
  
  /**
   * Schedule a given task to be executed at the given time.
   * 
   * <p>If the given time is passed, according to {@link Time} then the task is not executed.
   * 
   * <p>The given procedure takes one parameter: the agent associated to the task. It is name <code>it</code> by default.
   * 
   * <p>The given time is expressed in milliseconds according to the time scale of the SRE.
   * It means that the given number of milliseconds may be not real milliseconds, depending
   * on the definition of the builtin capacity {@link Time}.
   * 
   * <p>If this function is invoked from a {@code Behavior} and there is no provided task,
   * the created task will be associated to the behavior instance. It means that the task will
   * be automatically canceled when the behavior instance is unregistered from the the owning agent.
   * 
   * <p>This function does nothing if one of the following conditions evaluates to true:
   * <ul>
   * <li>the agent is not alive.</li>
   * </ul>
   * 
   * @optionalparam task the task that will run the given closure. If {@code null}, a new task is created.
   * @param time the time in milliseconds at which to start the procedure execution.
   * @param procedure the closure to execute.
   * @return the task is given, or a new task if the procedure is schedule, or
   *     {@code null} if the procedure is not scheduled.
   * @since 0.9
   */
  @DefaultValueUse("io.sarl.core.AgentTask,long,(io.sarl.lang.core.Agent)=>void")
  @SyntheticMember
  default AgentTask at(final long time, final Procedure1<? super Agent> procedure) {
    return at($DEFAULT_VALUE$AT_0(), time, procedure);
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends Schedules> extends Capacity.ContextAwareCapacityWrapper<C> implements Schedules {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public ConcurrentSet<String> getActiveTasks() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getActiveTasks();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask in(final AgentTask task, final long delay, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.in(task, delay, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask task(final String name) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.task(name);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public void setName(final AgentTask task, final String name) {
      try {
        ensureCallerInLocalThread();
        this.capacity.setName(task, name);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean cancel(final AgentTask task, final boolean mayInterruptIfRunning) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.cancel(task, mayInterruptIfRunning);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean isCanceled(final AgentTask task) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isCanceled(task);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask every(final AgentTask task, final long period, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.every(task, period, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask atFixedDelay(final AgentTask task, final long delay, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.atFixedDelay(task, delay, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask execute(final AgentTask task, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.execute(task, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask at(final AgentTask task, final long time, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.at(task, time, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask in(final long delay, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.in(delay, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public boolean cancel(final AgentTask task) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.cancel(task);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask every(final long period, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.every(period, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask atFixedDelay(final long delay, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.atFixedDelay(delay, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask execute(final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.execute(procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public AgentTask at(final long time, final Procedure1<? super Agent> procedure) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.at(time, procedure);
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
