/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.DefaultValue;
import io.sarl.lang.annotation.DefaultValueSource;
import io.sarl.lang.annotation.DefaultValueUse;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSourceCode;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import java.util.concurrent.TimeUnit;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Gives access to the time in the agent application.
 * Time management is application-dependent. This capacity does not make any assumption
 * on neither time evolution nor time refresh rate.
 */
@SarlSpecification("0.12")
@SarlElementType(20)
@SuppressWarnings("all")
public interface Time extends Capacity {
  /**
   * Replies the current time.
   * 
   * @param timeUnit if present, specifies the time unit of the value to reply. By default is it seconds.
   * @return the current time.
   */
  @DefaultValueSource
  @Pure
  double getTime(@DefaultValue("io.sarl.core.Time#GETTIME_0") final TimeUnit timeUnit);
  
  /**
   * Replies the scaling factor between the agent time and the operating system time.
   * <p>Consider time in the agent application. It may evolves at a different rate than
   * the operating system time. The value that is replied by this function permits to
   * determine the proportionally between the agent application time and the operating system time.
   * A typical equation for representing this relation is: os time = OSTimeFactor * agent time.
   * 
   * @return the factor between agent time and operating system time.
   */
  @Pure
  double getOSTimeFactor();
  
  /**
   * Convert the given time from the time scale of this builtin capacity to the operating
   * system time scale. It means that the "SRE" milliseconds are converted to
   * operating system milliseconds.
   * 
   * @param timeValue the value to convert.
   * @return the number of milleseconds in the OS that correspond to the given number in the SRE.
   * @since 0.9
   */
  @Pure
  double toOSTime(final double timeValue);
  
  /**
   * Convert the given time from the operating system time scale to the time scale of this
   * builtin capacity. It means that the operating system milliseconds are converted to
   * "SRE" milliseconds.
   * 
   * @param timeValue the value to convert.
   * @return the number of milleseconds in the SRE that correspond to the given number in the OS.
   * @since 0.9
   */
  double fromOSTime(final double timeValue);
  
  /**
   * Convert the given duration from the time scale of this builtin capacity to the operating
   * system time scale. It means that the "SRE" milliseconds are converted to
   * operating system milliseconds.
   * 
   * @param timeDuration the value to convert.
   * @return the number of milleseconds in the OS that correspond to the given number in the SRE.
   * @since 0.9
   */
  @Pure
  double toOSDuration(final double timeDuration);
  
  /**
   * Convert the given duration from the operating system time scale to the time scale of this
   * builtin capacity. It means that the operating system milliseconds are converted to
   * "SRE" milliseconds.
   * 
   * @param timeDuration the value to convert.
   * @return the number of milleseconds in the SRE that correspond to the given number in the OS.
   * @since 0.9
   */
  double fromOSDuration(final double timeDuration);
  
  /**
   * Default value for the parameter timeUnit
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  default TimeUnit $DEFAULT_VALUE$GETTIME_0() {
    return null;
  }
  
  /**
   * Replies the current time.
   * 
   * @optionalparam timeUnit if present, specifies the time unit of the value to reply. By default is it seconds.
   * @return the current time.
   */
  @DefaultValueUse("java.util.concurrent.TimeUnit")
  @SyntheticMember
  @Pure
  default double getTime() {
    return getTime($DEFAULT_VALUE$GETTIME_0());
  }
  
  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends Time> extends Capacity.ContextAwareCapacityWrapper<C> implements Time {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }
    
    public double getTime(final TimeUnit timeUnit) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getTime(timeUnit);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public double getOSTimeFactor() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getOSTimeFactor();
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public double toOSTime(final double timeValue) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.toOSTime(timeValue);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public double fromOSTime(final double timeValue) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.fromOSTime(timeValue);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public double toOSDuration(final double timeDuration) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.toOSDuration(timeDuration);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public double fromOSDuration(final double timeDuration) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.fromOSDuration(timeDuration);
      } finally {
        resetCallerInLocalThread();
      }
    }
    
    public double getTime() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getTime();
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
