/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.sarlspecification;

import com.google.common.base.Objects;
import io.sarl.lang.SARLVersion;
import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import org.eclipse.xtext.xbase.lib.Pure;
import org.osgi.framework.Version;

/**
 * Check if a given agent class follows a specific version of the SARL specifications.
 * 
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.util 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.util
 * @since 0.5
 */
@SarlSpecification("0.12")
@SarlElementType(11)
@SuppressWarnings("all")
public interface SarlSpecificationChecker {
  /**
   * Replies the supported SARL specification version of the given type.
   * 
   * @param type the type from which the SARL specification version should be extracted.
   * @return The SARL specification version. The version is a floating-point number, as for
   *     {@link SARLVersion#SPECIFICATION_RELEASE_VERSION}. The value {@link Float#NaN} is replied
   *     if the given type has no marker related to the SARL specification version.
   * @deprecated since 0.10, see #getSarlSpecificationVersionObject(Class)
   */
  @Deprecated
  @Pure
  default float getSarlSpecificationVersion(final Class<?> type) {
    throw new UnsupportedOperationException();
  }
  
  /**
   * Replies the supported SARL specification version of the given type.
   * 
   * @param type the type from which the SARL specification version should be extracted.
   * @return The SARL specification version. The version is a {@code Version}, as for
   *     {@link SARLVersion#SPECIFICATION_RELEASE_VERSION_STRING}. The value {@code null} is replied
   *     if the given type has no marker related to the SARL specification version.
   * @since 0.10
   */
  @Pure
  default Version getSarlSpecificationVersionObject(final Class<?> type) {
    if ((type != null)) {
      SarlSpecification annotationInstance = type.<SarlSpecification>getAnnotation(SarlSpecification.class);
      if ((annotationInstance != null)) {
        final String versionString = annotationInstance.value();
        final Version version = Version.parseVersion(versionString);
        boolean _notEquals = (!Objects.equal(Version.emptyVersion, version));
        if (_notEquals) {
          return version;
        }
      }
    }
    return null;
  }
  
  /**
   * Compare the SARL specification version associated to the given type to the version of the
   * current SARL.
   * 
   * @param type the type to test.
   * @return a negative integer value if the type's version is lower than the version of the current SARL,
   *     zero if the two versions are equal, a positive integer value if type's version is greater
   *     than the version of the current SARL.
   */
  default int compareToSarlSpecificationVersion(final Class<?> type) {
    Version v = this.getSarlSpecificationVersionObject(type);
    if ((v != null)) {
      Version _parseVersion = Version.parseVersion(SARLVersion.SPECIFICATION_RELEASE_VERSION_STRING);
      return (v.compareTo(_parseVersion));
    }
    return Integer.MAX_VALUE;
  }
  
  /**
   * Replies if the given type is a SARL element that is following the specification of the current SARL version.
   * 
   * @param type the type to test.
   * @return <code>true</code> if the given type follows the specification of the current version.
   */
  @Pure
  default boolean isValidSarlElement(final Class<?> type) {
    Version _sarlSpecificationVersionObject = this.getSarlSpecificationVersionObject(type);
    Version _parseVersion = Version.parseVersion(SARLVersion.SPECIFICATION_RELEASE_VERSION_STRING);
    return Objects.equal(_sarlSpecificationVersionObject, _parseVersion);
  }
}
