/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.util;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Utilities on collections.
 * 
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.util 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.util
 */
@SarlSpecification("0.12")
@SarlElementType(10)
@SuppressWarnings("all")
public final class Collections3 {
  /**
   * Unmodifiable synchronized iterable.
   * 
   * @param <E> the type of the elements in the collection.
   * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
   * @version io.sarl.util 0.12.0 20210527-171007
   * @mavengroupid io.sarl
   * @mavenartifactid io.sarl.util
   * @since 0.8.6
   */
  @SarlSpecification("0.12")
  @SarlElementType(10)
  private static class SerializableCollectionWrapper<E extends Object> extends AbstractCollection<E> implements Serializable {
    private transient Collection<E> collection;
    
    SerializableCollectionWrapper(final Collection<E> collection) {
      this.collection = collection;
    }
    
    @Override
    public Iterator<E> iterator() {
      return this.collection.iterator();
    }
    
    @Override
    public int size() {
      return this.collection.size();
    }
    
    @SuppressWarnings("unused_private_member")
    private void writeObject(final ObjectOutputStream stream) {
      try {
        stream.defaultWriteObject();
        stream.writeInt(this.collection.size());
        for (final E element : this.collection) {
          stream.writeObject(element);
        }
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @SuppressWarnings("unused_private_member")
    private void readObject(final ObjectInputStream stream) {
      try {
        stream.defaultReadObject();
        final int s = stream.readInt();
        ArrayList<E> _arrayList = new ArrayList<E>(s);
        this.collection = _arrayList;
        for (int i = 0; (i < s); i++) {
          Object _readObject = stream.readObject();
          this.collection.add(((E) _readObject));
        }
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @Override
    @Pure
    @SyntheticMember
    public boolean equals(final Object obj) {
      return super.equals(obj);
    }
    
    @Override
    @Pure
    @SyntheticMember
    public int hashCode() {
      int result = super.hashCode();
      return result;
    }
    
    @SyntheticMember
    private static final long serialVersionUID = -4813965186L;
  }
  
  /**
   * Replies a serializable collection.
   * If the wrapped collection is serializable, it is replied itself without wrapping.
   * If the wrapped collection is not serializable, it is wrapped into a collection
   * that supports serialization of the collection elements.
   * 
   * @param <T> - type of the elements in the collection.
   * @param elements the set to transform as unmodifiable collection.
   * @return an unmodifiable synchronized set.
   * @since 0.8.6
   */
  @Pure
  public static <T extends Object> Collection<T> serializableCollection(final Collection<T> elements) {
    if ((elements instanceof Serializable)) {
      return elements;
    }
    return new Collections3.SerializableCollectionWrapper<T>(elements);
  }
  
  @SyntheticMember
  public Collections3() {
    super();
  }
}
