/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.util;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Date;
import java.util.Objects;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * JUL formatter based on pattern.
 * 
 * <p>The format string may contains one of the following elements:<ul>
 * <li><code>%1</code>: the date,</li>
 * <li><code>%2</code>: the name of the calling function,</li>
 * <li><code>%3</code>: the name of the logger,</li>
 * <li><code>%4</code>: the logging level,</li>
 * <li><code>%5</code>: the message, and</li>
 * <li><code>%6</code>: the throwable.</li>
 * </ul>
 * 
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.util 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.util
 * @since 0.12
 */
@SarlSpecification("0.12")
@SarlElementType(10)
@SuppressWarnings("all")
public class JulPatternFormatter extends Formatter {
  private final String pattern;
  
  private final Date dat = new Date();
  
  /**
   * Constructor.
   * 
   * @param pattern the pattern that is compatible with {@link String#format(String, Object...)}.
   */
  public JulPatternFormatter(final String pattern) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(pattern);
        this.$$result = (!_isNullOrEmpty);
      }
    }
    assert new $AssertEvaluator$().$$result;
    this.pattern = pattern;
  }
  
  @Override
  public String format(final LogRecord record) {
    this.dat.setTime(record.getMillis());
    final StringBuilder source = new StringBuilder();
    final String scn = record.getSourceClassName();
    final String logName = record.getLoggerName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(scn);
    if ((!_isNullOrEmpty)) {
      source.append(scn);
      final String smn = record.getSourceMethodName();
      boolean _isNullOrEmpty_1 = StringExtensions.isNullOrEmpty(smn);
      if ((!_isNullOrEmpty_1)) {
        source.append(" ");
        source.append(smn);
      }
    } else {
      source.append(logName);
    }
    final String message = this.formatMessage(record);
    String throwable = null;
    Throwable _thrown = record.getThrown();
    if ((_thrown != null)) {
      final StringWriter sw = new StringWriter();
      try (PrintWriter pw = new PrintWriter(sw)) {
        pw.println();
        record.getThrown().printStackTrace(pw);
      }
      throwable = sw.toString();
    } else {
      throwable = "";
    }
    return String.format(
      this.pattern, 
      this.dat, source, 
      this.filterLogName(logName), 
      record.getLevel().getLocalizedName(), message, throwable);
  }
  
  /**
   * Filter the log name in order to computer the one that is displayed into the log.
   * 
   * @param logName the log name to filter.
   * @return the displayable log name.
   */
  @Pure
  protected String filterLogName(final String logName) {
    return logName;
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    JulPatternFormatter other = (JulPatternFormatter) obj;
    if (!Objects.equals(this.pattern, other.pattern))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.pattern);
    return result;
  }
}
