package io.sealights.onpremise.agents.plugin.surefire;

import lombok.Data;
/**
 * Contains parameters from surefire/falesafe plugin configuration,
 * which are relevant for the Sealights
 *  
 * @author AlaSchneider
 *
 */
@Data	
public class PluginConfiguration {
	
	public final static String SUREFIRE_ARTIFACT_ID = "maven-surefire-plugin";
	public final static String FAILSAFE_ARTIFACT_ID = "maven-failsafe-plugin";
	public final static String APACHE_MAVEN_PLUGIN_GROUP = "org.apache.maven.plugins";
	public final static String ARG_LINE = "argLine";
	public final static String SKIP = "skip";
	public final static String REUSE_FORKS = "reuseForks";
	public final static String FORK_COUNT = "forkCount";
	public final static String THREAD_COUNT = "threadCount";
	public final static String FORK_MODE = "forkMode";
	public final static String PARALLEL = "parallel";
	
	enum InvalidForkmode {
		perthread, never
	}

	public static final String ARG_LINE_PLACEHOLDER = "{" + ARG_LINE + "}";
	private final static String SL_ARG_LINE_PLACEHOLDER = "{sealightsArgLine}";
	private final static String AT = "@";
	
	private String artifactId;
	private String version;
	private String argLine;
	private Boolean skip;
	private Boolean reuseForks;
	private Integer forkCount;
	private Integer threadCount;
	private String forkMode;
	private String parallel;
	private String jacocoPropertyName;
	private String jacocoPropertyPlaceholder;
	
	public static PluginConfiguration createSurefireConfiguration(String jacocoPropertyName) {
		return new PluginConfiguration(SUREFIRE_ARTIFACT_ID, jacocoPropertyName);
	}
	
	static PluginConfiguration createSurefireConfiguration() {
		return createSurefireConfiguration(ARG_LINE);
	}
	
	public static PluginConfiguration createFailsafeConfiguration(String jacocoPropertyName) {
		return new PluginConfiguration(FAILSAFE_ARTIFACT_ID, jacocoPropertyName);
	}
	
	static PluginConfiguration createFailsafeConfiguration() {
		return createFailsafeConfiguration(ARG_LINE);
	}
	
	PluginConfiguration(String artifactId, String jacocoPropertyName) {
		this.artifactId = artifactId;
		//use setter to define jacocoPropertyPlaceHolder
		setJacocoPropertyName(jacocoPropertyName);
	}
	
	PluginConfiguration(String artifactId) {
		this(artifactId, ARG_LINE);
	}
	
	public boolean isEmpty() {
		return version == null;
	}
	
	/**
	 * argLine property is relevant in the following cases:
	 * <li> argLine configuration parameter is not defined
	 * <li> argLine configuration parameter is defined with argLine property
	 * placeholder, but without sealightsArgLine property folder
	 * @return
	 */
    public boolean isArgLinePropertyRelevant() {
    	return !hasArgLine() ||
				(!hasSealightsPlaceholder() && hasArgLinePlaceholder());
    }
    
	public boolean hasArgLine() {
		return argLine != null;
	}
	
	public boolean hasArgLinePlaceholder() {
		return hasPlaceholder(ARG_LINE_PLACEHOLDER);
	}

	public boolean hasJacocoPropertyPlaceholder(){
		return hasPlaceholder(jacocoPropertyPlaceholder);
	}

	public boolean hasSealightsPlaceholder() {
		return hasPlaceholder(SL_ARG_LINE_PLACEHOLDER);
	}
	
	public boolean isArgLinePlaceholderWithAtSymbol() {
		return isPlaceholderWithAtSymbol(ARG_LINE_PLACEHOLDER);
	}
	private boolean hasPlaceholder(String placeholder){
		return argLine != null && argLine.contains(placeholder);
	}
	
	public boolean isSealightsPlaceholderWithAtSymbol() {
		return isPlaceholderWithAtSymbol(SL_ARG_LINE_PLACEHOLDER);
	}
	
	private boolean isPlaceholderWithAtSymbol(String placeholder) {
		if (argLine == null || placeholder == null) {
			// Don't care
			return true;
		}		
		return argLine.contains(AT + placeholder);
	}

	public void setJacocoPropertyName(String jacocoPropertyName) {
		this.jacocoPropertyName = jacocoPropertyName;
		this.jacocoPropertyPlaceholder = "{" + jacocoPropertyName + "}";
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append(artifactId).append(" configuration: ");
		builder.append("[version=").append(version);
		if (argLine != null) builder.append(", argLine='").append(argLine).append("'");
		if (reuseForks != null) builder.append(", reuseForks=").append(reuseForks);
		if (forkCount != null) builder.append(", forkCount=").append(forkCount);
		if (threadCount != null) builder.append(", threadCount=").append(threadCount);
		if (forkMode != null) builder.append(", forkMode=").append(forkMode);
		if (parallel != null) builder.append(", parallel=").append(parallel);
		builder.append("]");
		return builder.toString();
	}

}