package io.sealights.onpremise.agents.plugin;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.HashMap;

import org.apache.maven.execution.BuildFailure;
import org.apache.maven.execution.BuildSuccess;
import org.apache.maven.execution.BuildSummary;
import org.apache.maven.execution.ExecutionEvent;
import org.apache.maven.execution.ExecutionListener;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.project.MavenProject;

import io.sealights.plugins.engine.lifecycle.BuildEndInfo;
import io.sealights.plugins.engine.lifecycle.BuildEndListener;
import lombok.Setter;

public enum MavenEventsListener implements InvocationHandler {
	INSTANCE;
	
	public static final String BUILD_SUBPROJECTS_FAILED = "Build failed due to sub-projects failure";
	public static final String DEFAULT_PROJECT_NAME = "project name cannot be resolved";

	private static final String METHOD_NAME_PROJECT_FAILED = "projectFailed";
	private static final String METHOD_NAME_SESSION_ENDED = "sessionEnded";
	private static final String METHOD_MOJO_FAILED = "mojoFailed";
	private static final String METHOD_FORK_FAILED = "forkFailed";
	
	@Setter // defined for tests
	private MavenSession session;
	
	private ExecutionListener mavenExecutionListener;
	
	private BuildEndListener buildEndListener;
	
	private volatile boolean initialized = false;
	
	private HashMap<String, FailedProjectInfo> projectsFailures = new HashMap<>();

	public synchronized boolean initialize(MavenSession session, BuildEndListener buildEndListener) {
		if (initialized) {
			return false;
		}

		if (buildEndListener != null) {
			this.session = session;
			this.buildEndListener = buildEndListener;
			doInitialization();
			return true;
		}
		
		return false;
	}
	
	public synchronized Object invoke(Object target, Method method, Object[] args) throws Throwable {
		ExecutionEvent event = args.length>0 ? (ExecutionEvent) args[0] : null;
		handleIfProjectFailure(method, event);
		handleIfEndOfBuild(method, event);
		informOnError(method, event);
		return method.invoke(mavenExecutionListener, args);
	}

	/**
	 * Method is public to allow easier testing
	 * @param currentProject
	 * @return
	 */
	public FailedProjectInfo extractBuildFailureReason(MavenProject currentProject, boolean addToProjectsFailures) {
		FailedProjectInfo failureInfo = null; 
		if (currentProject == null) {
			// When failure on parallel build, session returns 'null' current project
			MavenProject rootProject = session.getTopLevelProject();
			failureInfo = new FailedProjectInfo(rootProject != null ? rootProject.getName() : DEFAULT_PROJECT_NAME, BUILD_SUBPROJECTS_FAILED);
		}
		else {
			BuildSummary buildSummary = session.getResult().getBuildSummary(currentProject);
			if (buildSummary instanceof BuildSuccess && projectsFailures.isEmpty()) {
				// Current project build success and no subprojects failures
				return null;
			}
			
			if (buildSummary instanceof BuildFailure) {
				// Current project build fails
				BuildFailure failure = (BuildFailure) buildSummary;
				failureInfo = new FailedProjectInfo(currentProject.getName());
				failureInfo.reason = BuildFailureFormatter.format(failure.getCause().toString());			
			}
			else {
				// Current project build success, but there are failures of subprojects 
				failureInfo = new FailedProjectInfo(currentProject.getName(), BUILD_SUBPROJECTS_FAILED);
			}
		}
		
		if (!projectsFailures.containsKey(failureInfo.projectName)) {
			projectsFailures.put(failureInfo.projectName, failureInfo);
		}
		return failureInfo;

	}

	private void doInitialization() {
		mavenExecutionListener = session.getRequest().getExecutionListener();
		Object instance = Proxy.newProxyInstance(
				getClass().getClassLoader(), 
				new Class[]{ExecutionListener.class},
				this);
		session.getRequest().setExecutionListener((ExecutionListener) instance);
		buildEndListener.getPluginLogger().info("Initialized, session:'{}', buildEndListener:{}", session, buildEndListener);
		initialized = true;		
	}

	private void handleIfEndOfBuild(Method method, ExecutionEvent event) {
		if (!method.getName().equals(METHOD_NAME_SESSION_ENDED)) {
			return;
		}

		MavenProject project = session.getResult().getProject();
		logMethodInfo(method, project, null);
		BuildSummary buildSummary = session.getResult().getBuildSummary(project);
		BuildEndInfo buildInfo = new BuildEndInfo();
		buildInfo.setBuildDurationMSec(buildSummary != null ? buildSummary.getTime() : 0);

		FailedProjectInfo buildFailure = extractBuildFailureReason(session.getResult().getProject(), false);
		if (buildFailure == null) {
			buildInfo.setSuccess(true);
		} 
		else {
			buildInfo.setSuccess(false);
			if (projectsFailures.isEmpty()) {
				// Just put the last reason
				buildInfo.setFailureReason(buildFailure.reason);
			}
			else {
				buildInfo.setSuccess(false);
				buildInfo.setFailureReason(BuildFailureFormatter.collectFailures(buildFailure, projectsFailures));
			}
		}
		
		buildEndListener.notifyBuildEndInfo(buildInfo);
	}

	private void handleIfProjectFailure(Method method, ExecutionEvent event) {
		if (!method.getName().equals(METHOD_NAME_PROJECT_FAILED)) {
			return;
		}
		
		MavenProject currentPoject = session.getCurrentProject();
		logMethodInfo(method, currentPoject, null);
		extractBuildFailureReason(currentPoject, true);
	}
	
	private void informOnError(Method method, ExecutionEvent event) {
		if (method.getName().equals(METHOD_MOJO_FAILED) || method.getName().equals(METHOD_FORK_FAILED)) {
			logMethodInfo(method, session.getCurrentProject(), event);
		}
	}
	
	private void logMethodInfo(Method method, MavenProject project, ExecutionEvent event) {
		if (event != null && event.getException() != null) {
		buildEndListener.getPluginLogger().info("Handling '{}', exception:'{}', project:'{}', session:{}", 
				method.getName(),
				event.getException().getMessage().replaceAll("\n", ""),
				project != null ? project.getName() : project, 
				session);
		}
		else {
			buildEndListener.getPluginLogger().info("Handling '{}', project:'{}', session:{}", 
					method.getName(), 
					project != null ? project.getName() : project, 
					session);
		}
	}
	
	static class FailedProjectInfo {
		private static final String UNKNOWN_FAILURE = "Undefined failure reason";
		String projectName;
		String reason = UNKNOWN_FAILURE;
		
		FailedProjectInfo(String projectName) {
			this.projectName = projectName;
		}
		
		FailedProjectInfo(String projectName, String reason) {
			this(projectName);
			this.reason = reason;
		}
	}

}
