package io.sealights.onpremise.agents.plugin.surefire;

import org.apache.maven.artifact.versioning.ComparableVersion;
import org.apache.maven.model.Plugin;

import io.sealights.onpremise.agents.infra.configuration.validation.ValidationResult;
import io.sealights.onpremise.agents.infra.utils.StringUtils;
import lombok.Getter;
import lombok.Setter;

/**
 * Implements validation of surefire/failsafe plugins configuration
 * 
 * @author AlaSchneider
 *
 */
public class PluginCfgValidator {
	public final static String SUREFIRE_OR_FAILSAFE_NOT_FOUND = "neither surefire or failsafe plugin is set, SeaLights Test Listener will not run.";
	public final static String MIN_SUREFIRE_VERSION = "2.9";
	public static final String USE_VERSIONS = "please use the version '"+ MIN_SUREFIRE_VERSION + "' or later.";
	public static final String UNDEFINED_VERSION = "maven-surefire-plugin version is undefined; " + USE_VERSIONS;
	public static final String UNSUPPORTED_MIN_VERSION_FMT = 
			"maven-surefire-plugin version '%s' is not supported; " + USE_VERSIONS;
	//TODO vhrebinnyi - review links usage, try avoid hard coded values
	public static final String WIKI_LINK_SUREFIRE_INTEGRATION = "https://sealights.atlassian.net/wiki/spaces/SUP/pages/2873655305/Java+-+How+to+configure+Surefire+integration";
	public static final String INVALID_ARG_LINE = 
			"Incorrect 'argLine' value - the Sealights test-listener cannot be defined:"
			+ " @{argLine} or @{sealightsArgLine} should be added "
			+ WIKI_LINK_SUREFIRE_INTEGRATION;
	public static final String ARG_LINE_PLACEHOLDER_WITH_$ = 
			"Inaccurate 'argLine' value - the Sealights additions may be ignored; "
			+ "${argLine} should be replaced by @{argLine} or @{sealightsArgLine} should be added "
			+ WIKI_LINK_SUREFIRE_INTEGRATION;
	public static final String SL_ARG_LINE_PLACEHOLDER_WITH_$ = 
			"Inaccurate 'sealightsArgLine' value - the Sealights additions may be ignored;"
			+ " configuration ${sealightsArgLine} should be replaced by  @{sealightsArgLine} "
			+ WIKI_LINK_SUREFIRE_INTEGRATION;
	
	public static final String INVALID_FORK_MODE_NEVER = 
			"Unsupported 'forkMode' value 'never'";
	public static final String INVALID_FORK_MODE_PERTHREAD = 
			"Unsupported 'forkMode' value 'perthread', it cannot be used in combination with 'threadCount=0'";
	public static final String INVALID_FORK_COUNT = "Unsupported 'forkCount' value '0'";
	public static final String REUSE_FORKS_FALSE = "The parameter 'reuseForks=false' may affect the performance";
	public static final String PARALLEL_EXISTS = "The parameter 'parallel' may affects the coloring ";

	@Getter
	private final PluginConfiguration configuration;
	@Getter
	private ValidationResult validationResult = new ValidationResult();
	@Getter @Setter
	private boolean pluginExists = false;
	private boolean validateVersion;
	
	public static PluginCfgValidator createSurefireCfgValidator() {
		return new PluginCfgValidator(new PluginConfiguration(PluginConfiguration.SUREFIRE_ARTIFACT_ID), true);
	}
	
	public static PluginCfgValidator createFailsafeCfgValidator() {
		return new PluginCfgValidator(new PluginConfiguration(PluginConfiguration.FAILSAFE_ARTIFACT_ID), false);
	}
	
	PluginCfgValidator(PluginConfiguration configuration, boolean validateVersion) {
		this.configuration = configuration;
		this.validateVersion = validateVersion;
	}
	
	public void fillAndValidateConfiguration(Plugin plugin) {
		configuration.setArtifactId(plugin.getArtifactId());
		configuration.setVersion(plugin.getVersion());
    	setPluginExists(true);
    	if (validateVersion && !isSupportedVersion()) {
     		return;
    	}

    	try {
    		PluginCfgParser.parse(plugin, configuration);
    	}
    	catch (Exception e) {
    		validationResult.addWarning("Partial configuration was collected, error:" + e.toString());
    	}
    	
    	validate();
	}
	
	public boolean isArgLinePropertyRelevant() {
		return isPluginExists() && configuration.isArgLinePropertyRelevant();
	}

	public boolean isValidConfiguration() {
		return validationResult.isValid();
	}
	
	boolean isSupportedVersion() {
		if (!validateVersion) {
			return true;
		}
		if (configuration.getVersion() == null) {
			validationResult.addWarning(UNDEFINED_VERSION);
			return false;
		}
		ComparableVersion current = new ComparableVersion(configuration.getVersion());
		boolean supportedVersion = current.compareTo(new ComparableVersion(MIN_SUREFIRE_VERSION)) >= 0;
		if (!supportedVersion) {
			validationResult.addWarning(String.format(UNSUPPORTED_MIN_VERSION_FMT, configuration.getVersion()));
		}
		return supportedVersion;
	}
	
	boolean validate() {
		validateArgLine();
		validateForkCount();
		validateForkMode();
		validateParallel();
		validateReuseForks();
		return validationResult.isValid();
	}
	
	private void validateArgLine() {
		if (!configuration.hasArgLine()) {
			return;
		}
		
		boolean isArgLineDefined = configuration.hasArgLinePlaceholder();
		boolean isSealightsArgLineDefined = configuration.hasSealightsPlaceholder();
		
		if (!isArgLineDefined && !isSealightsArgLineDefined) {
			validationResult.addError(INVALID_ARG_LINE);
			return;
		}
		
		// add version consideration
		if (!isSealightsArgLineDefined) {
			validateArgLinePlaceholder();
		}
		else {
			validateSealightsArgLinePlaceholder();
		}
	}
	
	private void validateArgLinePlaceholder() {
		if (!configuration.isArgLinePlacehodlerWithAtSymbol()) {
			validationResult.addWarning(ARG_LINE_PLACEHOLDER_WITH_$);
		}		
	}

	private void validateSealightsArgLinePlaceholder() {
		if (!configuration.isSealightsPlacehodlerWithAtSymbol()) {
			validationResult.addWarning(SL_ARG_LINE_PLACEHOLDER_WITH_$);
		}		
	}

	protected void validateForkMode() {
		String forkMode = configuration.getForkMode();
		if (forkMode == null) {
			return;
		}		
		
		if (PluginConfiguration.InvalidForkmode.never.name().equals(forkMode)) {
			validationResult.addError(INVALID_FORK_MODE_NEVER);
			
		}
		
		if (PluginConfiguration.InvalidForkmode.perthread.name().equals(forkMode)) {
			if (isValueOfZero(configuration.getThreadCount())) {
				validationResult.addError(INVALID_FORK_MODE_PERTHREAD);
			}
		}		
	}
	
	protected void validateForkCount() {
		if (isValueOfZero(configuration.getForkCount())) {
			validationResult.addError(INVALID_FORK_COUNT);
		}
	}
	
	protected void validateReuseForks() {
		if (configuration.getReuseForks() != null && !configuration.getReuseForks()) {
			validationResult.addWarning(REUSE_FORKS_FALSE);
		}
	}
	
	protected void validateParallel() {
		if (StringUtils.isNotEmpty(configuration.getParallel())) {
			validationResult.addWarning(PARALLEL_EXISTS);
		}
	}
	
	private boolean isValueOfZero(Integer value) {
		return value != null && value == 0;
	}
	
}
