package io.sentry.android.gradle.tasks

import io.sentry.android.gradle.extensions.SentryPluginExtension
import io.sentry.android.gradle.telemetry.SentryTelemetryService
import io.sentry.android.gradle.telemetry.withSentryTelemetry
import io.sentry.android.gradle.util.PropertiesUtil
import java.util.Properties
import org.gradle.api.Project
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.Directory
import org.gradle.api.file.RegularFile
import org.gradle.api.provider.Provider
import org.gradle.api.tasks.CacheableTask
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity.NONE
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.TaskProvider

@CacheableTask
abstract class SentryGenerateDebugMetaPropertiesTask : DirectoryOutputTask() {

  init {
    description = "Combines multiple properties files into sentry-debug-meta.properties"
  }

  @get:Internal
  val outputFile: Provider<RegularFile>
    get() = output.file(SENTRY_DEBUG_META_PROPERTIES_OUTPUT)

  // we only care about file contents
  @get:PathSensitive(NONE) @get:InputFiles abstract val inputFiles: ConfigurableFileCollection

  @TaskAction
  fun generateProperties() {
    val outputDir = output.get().asFile
    outputDir.mkdirs()

    val debugMetaPropertiesFile = outputFile.get().asFile
    val props = Properties()
    props.setProperty("io.sentry.build-tool", "gradle")
    inputFiles.forEach { inputFile ->
      PropertiesUtil.loadMaybe(inputFile)?.let { props.putAll(it) }
    }
    debugMetaPropertiesFile.writer().use {
      props.store(it, "Generated by sentry-android-gradle-plugin")
    }
  }

  companion object {
    internal const val SENTRY_DEBUG_META_PROPERTIES_OUTPUT = "sentry-debug-meta.properties"

    fun register(
      project: Project,
      extension: SentryPluginExtension,
      sentryTelemetryProvider: Provider<SentryTelemetryService>?,
      tasksGeneratingProperties: List<TaskProvider<out PropertiesFileOutputTask>>,
      output: Provider<Directory>? = null,
      taskSuffix: String = "",
    ): TaskProvider<SentryGenerateDebugMetaPropertiesTask> {
      val inputFiles: List<Provider<RegularFile>> =
        tasksGeneratingProperties.mapNotNull { it.flatMap { task -> task.outputFile } }
      return project.tasks.register(
        "generateSentryDebugMetaProperties$taskSuffix",
        SentryGenerateDebugMetaPropertiesTask::class.java,
      ) { task ->
        task.inputFiles.setFrom(inputFiles)
        output?.let { task.output.set(it) }
        task.withSentryTelemetry(extension, sentryTelemetryProvider)
      }
    }
  }
}
