/*
 * Decompiled with CFR 0.152.
 */
package io.sirix.access;

import com.github.benmanes.caffeine.cache.AsyncCache;
import com.google.common.base.MoreObjects;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.Maps;
import com.google.crypto.tink.CleartextKeysetHandle;
import com.google.crypto.tink.JsonKeysetWriter;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.KeysetWriter;
import com.google.crypto.tink.proto.KeyTemplate;
import com.google.crypto.tink.streamingaead.StreamingAeadKeyTemplates;
import io.sirix.access.DatabaseConfiguration;
import io.sirix.access.Databases;
import io.sirix.access.PathBasedPool;
import io.sirix.access.ResourceConfiguration;
import io.sirix.access.ResourceStore;
import io.sirix.access.WriteLocksRegistry;
import io.sirix.access.trx.node.AfterCommitState;
import io.sirix.api.Database;
import io.sirix.api.NodeCursor;
import io.sirix.api.NodeReadOnlyTrx;
import io.sirix.api.NodeTrx;
import io.sirix.api.ResourceSession;
import io.sirix.api.Transaction;
import io.sirix.api.TransactionManager;
import io.sirix.cache.BufferManager;
import io.sirix.cache.BufferManagerImpl;
import io.sirix.exception.SirixException;
import io.sirix.exception.SirixIOException;
import io.sirix.exception.SirixUsageException;
import io.sirix.io.StorageType;
import io.sirix.io.bytepipe.Encryptor;
import io.sirix.utils.SirixFiles;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.GeneralSecurityException;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.checkerframework.checker.index.qual.NonNegative;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LocalDatabase<T extends ResourceSession<? extends NodeReadOnlyTrx, W>, W extends NodeTrx & NodeCursor>
implements Database<T> {
    private static final Logger logger = LoggerFactory.getLogger(LocalDatabase.class);
    private final AtomicLong resourceID = new AtomicLong();
    private final TransactionManager transactionManager;
    private volatile boolean isClosed;
    private final BiMap<Long, String> resourceIDsToResourceNames;
    private final DatabaseConfiguration dbConfig;
    private final PathBasedPool<Database<?>> sessions;
    private final ResourceStore<T> resourceStore;
    private final PathBasedPool<ResourceSession<?, ?>> resourceManagers;
    private final WriteLocksRegistry writeLocks;
    private final ConcurrentMap<Path, BufferManager> bufferManagers;

    public LocalDatabase(TransactionManager transactionManager, DatabaseConfiguration dbConfig, PathBasedPool<Database<?>> sessions, ResourceStore<T> resourceStore, WriteLocksRegistry writeLocks, PathBasedPool<ResourceSession<?, ?>> resourceManagers) {
        this.transactionManager = transactionManager;
        this.dbConfig = Objects.requireNonNull(dbConfig);
        this.sessions = sessions;
        this.resourceStore = resourceStore;
        this.resourceManagers = resourceManagers;
        this.writeLocks = writeLocks;
        this.resourceIDsToResourceNames = Maps.synchronizedBiMap((BiMap)HashBiMap.create());
        this.sessions.putObject(dbConfig.getDatabaseFile(), this);
        this.bufferManagers = Databases.getBufferManager(dbConfig.getDatabaseFile());
    }

    private void addResourceToBufferManagerMapping(Path resourceFile, ResourceConfiguration resourceConfig) {
        if (resourceConfig.getStorageType() == StorageType.MEMORY_MAPPED) {
            this.bufferManagers.put(resourceFile, new BufferManagerImpl(100, 1000, 5000, 50000, 500, 20));
        } else {
            this.bufferManagers.put(resourceFile, new BufferManagerImpl(500, 1000, 5000, 50000, 500, 20));
        }
    }

    @Override
    public @NonNull T beginResourceSession(String resourceName) {
        this.assertNotClosed();
        Path resourcePath = this.dbConfig.getDatabaseFile().resolve(DatabaseConfiguration.DatabasePaths.DATA.getFile()).resolve(resourceName);
        if (!Files.exists(resourcePath, new LinkOption[0])) {
            throw new SirixUsageException("Resource could not be opened (since it was not created?) at location", resourcePath.toString());
        }
        if (this.resourceStore.hasOpenResourceSession(resourcePath)) {
            return this.resourceStore.getOpenResourceSession(resourcePath);
        }
        ResourceConfiguration resourceConfig = ResourceConfiguration.deserialize(resourcePath);
        assert (resourceConfig.resourcePath.getParent().getParent().equals(this.dbConfig.getDatabaseFile()));
        this.resourceIDsToResourceNames.forcePut((Object)resourceConfig.getID(), (Object)resourceConfig.getResource().getFileName().toString());
        if (!this.bufferManagers.containsKey(resourcePath)) {
            this.addResourceToBufferManagerMapping(resourcePath, resourceConfig);
        }
        return this.resourceStore.beginResourceSession(resourceConfig, (BufferManager)this.bufferManagers.get(resourcePath), resourcePath);
    }

    @Override
    public String getName() {
        return this.dbConfig.getDatabaseName();
    }

    @Override
    public synchronized boolean createResource(ResourceConfiguration resourceConfig) {
        this.assertNotClosed();
        boolean returnVal = true;
        resourceConfig.setDatabaseConfiguration(this.dbConfig);
        Path path = this.dbConfig.getDatabaseFile().resolve(DatabaseConfiguration.DatabasePaths.DATA.getFile()).resolve(resourceConfig.resourcePath);
        if (Files.exists(path, new LinkOption[0])) {
            return false;
        }
        try {
            Files.createDirectory(path, new FileAttribute[0]);
        }
        catch (IOException | SecurityException | UnsupportedOperationException e) {
            returnVal = false;
        }
        if (returnVal) {
            for (ResourceConfiguration.ResourcePaths resourcePath : ResourceConfiguration.ResourcePaths.values()) {
                Path toCreate = path.resolve(resourcePath.getPath());
                try {
                    if (resourcePath.isFolder()) {
                        Files.createDirectory(toCreate, new FileAttribute[0]);
                        if (resourcePath == ResourceConfiguration.ResourcePaths.ENCRYPTION_KEY) {
                            this.createAndStoreKeysetIfNeeded(resourceConfig, toCreate);
                        }
                    } else {
                        Files.createFile(toCreate, new FileAttribute[0]);
                    }
                }
                catch (IOException | SecurityException | UnsupportedOperationException e) {
                    returnVal = false;
                }
                if (!returnVal) break;
            }
        }
        if (returnVal) {
            this.resourceID.set(this.dbConfig.getMaxResourceID());
            ResourceConfiguration.serialize(resourceConfig.setID(this.resourceID.getAndIncrement()));
            this.dbConfig.setMaximumResourceID(this.resourceID.get());
            this.resourceIDsToResourceNames.forcePut((Object)this.resourceID.get(), (Object)resourceConfig.getResource().getFileName().toString());
            returnVal = this.bootstrapResource(resourceConfig);
        }
        if (!returnVal) {
            SirixFiles.recursiveRemove(resourceConfig.resourcePath);
        }
        if (!this.bufferManagers.containsKey(path)) {
            this.addResourceToBufferManagerMapping(path, resourceConfig);
        }
        return returnVal;
    }

    void createAndStoreKeysetIfNeeded(ResourceConfiguration resConfig, Path createdPath) {
        Path encryptionKeyPath = createdPath.resolve("encryptionKey.json");
        if (resConfig.byteHandlePipeline.getComponents().contains(new Encryptor(createdPath.getParent()))) {
            try {
                Files.createFile(encryptionKeyPath, new FileAttribute[0]);
                KeysetHandle handle = KeysetHandle.generateNew((KeyTemplate)StreamingAeadKeyTemplates.AES256_CTR_HMAC_SHA256_4KB);
                CleartextKeysetHandle.write((KeysetHandle)handle, (KeysetWriter)JsonKeysetWriter.withPath((Path)encryptionKeyPath));
            }
            catch (IOException | GeneralSecurityException e) {
                throw new IllegalStateException(e);
            }
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private boolean bootstrapResource(ResourceConfiguration resConfig) {
        try (T resourceTrxManager = this.beginResourceSession(resConfig.getResource().getFileName().toString());){
            Object wtx = resourceTrxManager.beginNodeTrx(AfterCommitState.CLOSE);
            try {
                boolean useCustomCommitTimestamps = resConfig.customCommitTimestamps();
                if (useCustomCommitTimestamps) {
                    wtx.commit(null, Instant.ofEpochMilli(0L));
                } else {
                    wtx.commit();
                }
                boolean bl = true;
                if (wtx != null) {
                    wtx.close();
                }
                return bl;
            }
            catch (Throwable throwable) {
                if (wtx != null) {
                    try {
                        wtx.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
        }
        catch (SirixException e) {
            logger.error(e.getMessage(), (Throwable)e);
            return false;
        }
    }

    @Override
    public boolean isOpen() {
        return !this.isClosed;
    }

    @Override
    public synchronized Database<T> removeResource(String name) {
        this.assertNotClosed();
        Objects.requireNonNull(name);
        Path resourceFile = this.dbConfig.getDatabaseFile().resolve(DatabaseConfiguration.DatabasePaths.DATA.getFile()).resolve(name);
        if (this.resourceManagers.containsAnyEntry(resourceFile)) {
            throw new IllegalStateException("Open resource managers found, must be closed first: " + String.valueOf(this.resourceManagers));
        }
        if (Files.exists(resourceFile, new LinkOption[0]) && ResourceConfiguration.ResourcePaths.compareStructure(resourceFile) == 0) {
            AsyncCache cache;
            SirixFiles.recursiveRemove(resourceFile);
            this.writeLocks.removeWriteLock(resourceFile);
            BufferManager bufferManager = (BufferManager)this.bufferManagers.remove(resourceFile);
            if (bufferManager != null) {
                try {
                    bufferManager.clearAllCaches();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
            if ((cache = (AsyncCache)StorageType.CACHE_REPOSITORY.remove(resourceFile)) != null) {
                cache.synchronous().invalidateAll();
            }
        }
        return this;
    }

    @Override
    public synchronized String getResourceName(@NonNegative long id) {
        this.assertNotClosed();
        return (String)this.resourceIDsToResourceNames.get((Object)id);
    }

    @Override
    public synchronized long getResourceID(String name) {
        this.assertNotClosed();
        return (Long)this.resourceIDsToResourceNames.inverse().get((Object)Objects.requireNonNull(name));
    }

    private void assertNotClosed() {
        if (this.isClosed) {
            throw new IllegalStateException("Database is already closed.");
        }
    }

    @Override
    public DatabaseConfiguration getDatabaseConfig() {
        this.assertNotClosed();
        return this.dbConfig;
    }

    @Override
    public synchronized boolean existsResource(String resourceName) {
        this.assertNotClosed();
        Path resourceFile = this.dbConfig.getDatabaseFile().resolve(DatabaseConfiguration.DatabasePaths.DATA.getFile()).resolve(resourceName);
        return Files.exists(resourceFile, new LinkOption[0]) && ResourceConfiguration.ResourcePaths.compareStructure(resourceFile) == 0;
    }

    @Override
    public List<Path> listResources() {
        List<Path> list;
        block8: {
            this.assertNotClosed();
            Stream<Path> stream = Files.list(this.dbConfig.getDatabaseFile().resolve(DatabaseConfiguration.DatabasePaths.DATA.getFile()));
            try {
                list = stream.toList();
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new SirixIOException(e);
                }
            }
            stream.close();
        }
        return list;
    }

    @Override
    public Transaction beginTransaction() {
        return null;
    }

    @Override
    public synchronized void close() {
        if (this.isClosed) {
            return;
        }
        logger.trace("Close local database instance.");
        this.isClosed = true;
        this.resourceStore.close();
        this.transactionManager.close();
        this.sessions.removeObject(this.dbConfig.getDatabaseFile(), this);
        SirixFiles.recursiveRemove(this.dbConfig.getDatabaseFile().resolve(DatabaseConfiguration.DatabasePaths.LOCK.getFile()));
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("dbConfig", (Object)this.dbConfig).toString();
    }
}

