/*
 * Decompiled with CFR 0.152.
 */
package io.skodjob.testframe.executor;

import io.fabric8.kubernetes.api.model.EnvVar;
import io.skodjob.testframe.clients.KubeClusterException;
import io.skodjob.testframe.executor.ExecBuilder;
import io.skodjob.testframe.executor.ExecResult;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Exec {
    private static final Logger LOGGER = LoggerFactory.getLogger(Exec.class);
    private static final Pattern ERROR_PATTERN = Pattern.compile("Error from server \\(([a-zA-Z0-9]+)\\):");
    private static final Pattern INVALID_PATTERN = Pattern.compile("The ([a-zA-Z0-9]+) \"([a-z0-9.-]+)\" is invalid:");
    private static final Pattern PATH_SPLITTER = Pattern.compile(File.pathSeparator);
    private static final int MAXIMUM_EXEC_LOG_CHARACTER_SIZE = 2000;
    private static final Object LOCK = new Object();
    private Process process;
    private String stdOut;
    private String stdErr;
    private StreamGobbler stdOutReader;
    private StreamGobbler stdErrReader;
    private Path logPath;
    private final boolean appendLineSeparator;

    public Exec() {
        this.appendLineSeparator = true;
    }

    public Exec(Path logPath) {
        this.appendLineSeparator = true;
        this.logPath = logPath;
    }

    public Exec(boolean appendLineSeparator) {
        this.appendLineSeparator = appendLineSeparator;
    }

    public static ExecBuilder builder() {
        return new ExecBuilder();
    }

    public String out() {
        return this.stdOut;
    }

    public String err() {
        return this.stdErr;
    }

    public static ExecResult exec(String ... command) {
        return Exec.exec(Arrays.asList(command));
    }

    public static ExecResult exec(boolean logToOutput, String ... command) {
        ArrayList<String> commands = new ArrayList<String>(Arrays.asList(command));
        return Exec.exec(null, commands, 0, logToOutput);
    }

    public static ExecResult exec(List<String> command) {
        return Exec.exec(null, command, 0, false);
    }

    public static ExecResult exec(String input, List<String> command) {
        return Exec.exec(input, command, 0, false);
    }

    public static ExecResult exec(String input, List<String> command, int timeout, boolean logToOutput) {
        return Exec.exec(input, command, Collections.emptySet(), timeout, logToOutput, true);
    }

    public static ExecResult exec(String input, List<String> command, int timeout, boolean logToOutput, boolean throwErrors) {
        return Exec.exec(input, command, Collections.emptySet(), timeout, logToOutput, throwErrors);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static ExecResult exec(String input, List<String> command, Set<EnvVar> envVars, int timeout, boolean logToOutput, boolean throwErrors) {
        try {
            Exec executor = new Exec();
            LOGGER.info("Command: {}", (Object)String.join((CharSequence)" ", command));
            int ret = executor.execute(input, command, envVars, timeout);
            Object object = LOCK;
            synchronized (object) {
                if (logToOutput) {
                    LOGGER.info("RETURN code: {}", (Object)ret);
                    if (!executor.out().isEmpty()) {
                        LOGGER.info("======STDOUT START=======");
                        LOGGER.info("{}", (Object)Exec.cutExecutorLog(executor.out()));
                        LOGGER.info("======STDOUT END======");
                    }
                    if (!executor.err().isEmpty()) {
                        LOGGER.info("======STDERR START=======");
                        LOGGER.info("{}", (Object)Exec.cutExecutorLog(executor.err()));
                        LOGGER.info("======STDERR END======");
                    }
                }
            }
            ExecResult execResult = new ExecResult(ret, executor.out(), executor.err());
            if (throwErrors && ret != 0) {
                String msg = "`" + String.join((CharSequence)" ", command) + "` got status code " + ret + " and stderr:\n------\n" + executor.stdErr + "\n------\nand stdout:\n------\n" + executor.stdOut + "\n------";
                Matcher matcher = ERROR_PATTERN.matcher(executor.err());
                KubeClusterException t = null;
                if (matcher.find()) {
                    switch (matcher.group(1)) {
                        case "NotFound": {
                            t = new KubeClusterException.NotFound(execResult, msg);
                            break;
                        }
                        case "AlreadyExists": {
                            t = new KubeClusterException.AlreadyExists(execResult, msg);
                            break;
                        }
                    }
                }
                if ((matcher = INVALID_PATTERN.matcher(executor.err())).find()) {
                    t = new KubeClusterException.InvalidResource(execResult, msg);
                }
                if (t == null) {
                    t = new KubeClusterException(execResult, msg);
                }
                throw t;
            }
            return new ExecResult(ret, executor.out(), executor.err());
        }
        catch (IOException | ExecutionException e) {
            throw new KubeClusterException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new KubeClusterException(e);
        }
    }

    public int execute(String input, List<String> commands, Set<EnvVar> envVars, long timeoutMs) throws IOException, InterruptedException, ExecutionException {
        LOGGER.trace("Running command - {}", (Object)String.join((CharSequence)" ", commands.toArray(new String[0])));
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        builder.command(commands);
        if (envVars != null) {
            envVars.forEach(e -> builder.environment().put(e.getName(), e.getValue()));
        }
        builder.directory(new File(System.getProperty("user.dir")));
        this.process = builder.start();
        try (OutputStream outputStream = this.process.getOutputStream();){
            if (input != null) {
                LOGGER.trace("With stdin {}", (Object)input);
                outputStream.write(input.getBytes(Charset.defaultCharset()));
            }
        }
        Future<String> output = this.readStdOutput();
        Future<String> error = this.readStdError();
        int retCode = 1;
        if (timeoutMs > 0L) {
            if (this.process.waitFor(timeoutMs, TimeUnit.MILLISECONDS)) {
                retCode = this.process.exitValue();
            } else {
                this.process.destroyForcibly();
            }
        } else {
            retCode = this.process.waitFor();
        }
        try {
            this.stdOut = output.get(500L, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException ex) {
            output.cancel(true);
            this.stdOut = this.stdOutReader.getData();
        }
        try {
            this.stdErr = error.get(500L, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException ex) {
            error.cancel(true);
            this.stdErr = this.stdErrReader.getData();
        }
        this.storeOutputsToFile();
        return retCode;
    }

    public void stop() {
        this.process.destroyForcibly();
        this.stdOut = this.stdOutReader.getData();
        this.stdErr = this.stdErrReader.getData();
    }

    private Future<String> readStdOutput() {
        this.stdOutReader = new StreamGobbler(this.process.getInputStream());
        return this.stdOutReader.read();
    }

    private Future<String> readStdError() {
        this.stdErrReader = new StreamGobbler(this.process.getErrorStream());
        return this.stdErrReader.read();
    }

    private void storeOutputsToFile() {
        if (this.logPath != null) {
            try {
                Files.createDirectories(this.logPath, new FileAttribute[0]);
                Files.writeString(Paths.get(this.logPath.toString(), "stdOutput.log"), (CharSequence)this.stdOut, Charset.defaultCharset(), new OpenOption[0]);
                Files.writeString(Paths.get(this.logPath.toString(), "stdError.log"), (CharSequence)this.stdErr, Charset.defaultCharset(), new OpenOption[0]);
            }
            catch (Exception ex) {
                LOGGER.warn("Cannot save output of execution: " + ex.getMessage());
            }
        }
    }

    public static boolean isExecutableOnPath(String cmd) {
        for (String dir : PATH_SPLITTER.split(System.getenv("PATH"))) {
            if (!new File(dir, cmd).canExecute()) continue;
            return true;
        }
        return false;
    }

    public static String cutExecutorLog(String log) {
        if (log.length() > 2000) {
            LOGGER.warn("Executor log is too long. Going to strip it and print only first {} characters", (Object)2000);
            return log.substring(0, 2000);
        }
        return log;
    }

    class StreamGobbler {
        private final InputStream is;
        private final StringBuffer data = new StringBuffer();

        StreamGobbler(InputStream is) {
            this.is = is;
        }

        public String getData() {
            return this.data.toString();
        }

        public Future<String> read() {
            return CompletableFuture.supplyAsync(() -> {
                Scanner scanner = new Scanner(this.is, StandardCharsets.UTF_8);
                try {
                    while (scanner.hasNextLine()) {
                        this.data.append(scanner.nextLine());
                        if (!Exec.this.appendLineSeparator) continue;
                        this.data.append(System.lineSeparator());
                    }
                    scanner.close();
                    String string = this.data.toString();
                    scanner.close();
                    return string;
                }
                catch (Throwable throwable) {
                    try {
                        try {
                            scanner.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                    catch (Exception e) {
                        throw new CompletionException(e);
                    }
                }
            }, runnable -> new Thread(runnable).start());
        }
    }
}

