package io.smallrye.reactive.messaging.aws.sqs;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code smallrye-sqs} connector.
*/
 public class SqsConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new SqsConnectorCommonConfiguration.
   */
  public SqsConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the queue value from the configuration.
  * Attribute Name: queue
  * Description: The name of the SQS queue, defaults to channel name if not provided
  * @return the queue
  */
  public Optional<String> getQueue() {
    return config.getOptionalValue("queue", String.class);
  }

  /**
  * Gets the queue.url value from the configuration.
  * Attribute Name: queue.url
  * Description: The url of the SQS queue
  * @return the queue.url
  */
  public Optional<String> getQueueUrl() {
    return config.getOptionalValue("queue.url", String.class);
  }

  /**
  * Gets the region value from the configuration.
  * Attribute Name: region
  * Description: The name of the SQS region
  * @return the region
  */
  public Optional<String> getRegion() {
    return config.getOptionalValue("region", String.class);
  }

  /**
  * Gets the endpoint-override value from the configuration.
  * Attribute Name: endpoint-override
  * Description: The endpoint override
  * @return the endpoint-override
  */
  public Optional<String> getEndpointOverride() {
    return config.getOptionalValue("endpoint-override", String.class);
  }

  /**
  * Gets the credentials-provider value from the configuration.
  * Attribute Name: credentials-provider
  * Description: The credential provider to be used in the client
  * @return the credentials-provider
  */
  public Optional<String> getCredentialsProvider() {
    return config.getOptionalValue("credentials-provider", String.class);
  }

  /**
  * Gets the health-enabled value from the configuration.
  * Attribute Name: health-enabled
  * Description: Whether health reporting is enabled (default) or disabled
  * Default Value: true
  * @return the health-enabled
  */
  public Boolean getHealthEnabled() {
    return config.getOptionalValue("health-enabled", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  public void validate() {
  }
}
