package io.smallrye.reactive.messaging.jms;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code smallrye-jms} connector.
*/
 public class JmsConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new JmsConnectorCommonConfiguration.
   */
  public JmsConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the connection-factory-name value from the configuration.
  * Attribute Name: connection-factory-name
  * Description: The name of the JMS connection factory  (`javax.jms.ConnectionFactory`) to be used. If not set, it uses any exposed JMS connection factory
  * @return the connection-factory-name
  */
  public Optional<String> getConnectionFactoryName() {
    return config.getOptionalValue("connection-factory-name", String.class);
  }

  /**
  * Gets the username value from the configuration.
  * Attribute Name: username
  * Description: The username to connect to to the JMS server
  * @return the username
  */
  public Optional<String> getUsername() {
    return config.getOptionalValue("username", String.class);
  }

  /**
  * Gets the password value from the configuration.
  * Attribute Name: password
  * Description: The password to connect to to the JMS server
  * @return the password
  */
  public Optional<String> getPassword() {
    return config.getOptionalValue("password", String.class);
  }

  /**
  * Gets the session-mode value from the configuration.
  * Attribute Name: session-mode
  * Description: The session mode. Accepted values are AUTO_ACKNOWLEDGE, SESSION_TRANSACTED, CLIENT_ACKNOWLEDGE, DUPS_OK_ACKNOWLEDGE
  * Default Value: AUTO_ACKNOWLEDGE
  * @return the session-mode
  */
  public String getSessionMode() {
    return config.getOptionalValue("session-mode", String.class)
     .orElse("AUTO_ACKNOWLEDGE");
  }

  /**
  * Gets the client-id value from the configuration.
  * Attribute Name: client-id
  * Description: The client id
  * @return the client-id
  */
  public Optional<String> getClientId() {
    return config.getOptionalValue("client-id", String.class);
  }

  /**
  * Gets the destination value from the configuration.
  * Attribute Name: destination
  * Description: The name of the JMS destination. If not set the name of the channel is used
  * @return the destination
  */
  public Optional<String> getDestination() {
    return config.getOptionalValue("destination", String.class);
  }

  /**
  * Gets the destination-type value from the configuration.
  * Attribute Name: destination-type
  * Description: The type of destination. It can be either `queue` or `topic`
  * Default Value: queue
  * @return the destination-type
  */
  public String getDestinationType() {
    return config.getOptionalValue("destination-type", String.class)
     .orElse("queue");
  }

  public void validate() {
  }
}
