package io.smallrye.reactive.messaging.jms;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code smallrye-jms} connector.
*/
 public class JmsConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new JmsConnectorCommonConfiguration.
   */
  public JmsConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the connection-factory-name value from the configuration.
  * Attribute Name: connection-factory-name
  * Description: The name of the JMS connection factory  (`jakarta.jms.ConnectionFactory`) to be used. If not set, it uses any exposed JMS connection factory
  * @return the connection-factory-name
  */
  public Optional<String> getConnectionFactoryName() {
    return config.getOptionalValue("connection-factory-name", String.class);
  }

  /**
  * Gets the username value from the configuration.
  * Attribute Name: username
  * Description: The username to connect to to the JMS server
  * @return the username
  */
  public Optional<String> getUsername() {
    return config.getOptionalValue("username", String.class);
  }

  /**
  * Gets the password value from the configuration.
  * Attribute Name: password
  * Description: The password to connect to to the JMS server
  * @return the password
  */
  public Optional<String> getPassword() {
    return config.getOptionalValue("password", String.class);
  }

  /**
  * Gets the session-mode value from the configuration.
  * Attribute Name: session-mode
  * Description: The session mode. Accepted values are AUTO_ACKNOWLEDGE, SESSION_TRANSACTED, CLIENT_ACKNOWLEDGE, DUPS_OK_ACKNOWLEDGE
  * Default Value: AUTO_ACKNOWLEDGE
  * @return the session-mode
  */
  public String getSessionMode() {
    return config.getOptionalValue("session-mode", String.class)
     .orElse("AUTO_ACKNOWLEDGE");
  }

  /**
  * Gets the client-id value from the configuration.
  * Attribute Name: client-id
  * Description: The client id
  * @return the client-id
  */
  public Optional<String> getClientId() {
    return config.getOptionalValue("client-id", String.class);
  }

  /**
  * Gets the destination value from the configuration.
  * Attribute Name: destination
  * Description: The name of the JMS destination. If not set the name of the channel is used
  * @return the destination
  */
  public Optional<String> getDestination() {
    return config.getOptionalValue("destination", String.class);
  }

  /**
  * Gets the destination-type value from the configuration.
  * Attribute Name: destination-type
  * Description: The type of destination. It can be either `queue` or `topic`
  * Default Value: queue
  * @return the destination-type
  */
  public String getDestinationType() {
    return config.getOptionalValue("destination-type", String.class)
     .orElse("queue");
  }

  /**
  * Gets the tracing-enabled value from the configuration.
  * Attribute Name: tracing-enabled
  * Description: Whether tracing is enabled (default) or disabled
  * Default Value: true
  * @return the tracing-enabled
  */
  public Boolean getTracingEnabled() {
    return config.getOptionalValue("tracing-enabled", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the retry value from the configuration.
  * Attribute Name: retry
  * Description: Whether to retry on terminal stream errors.
  * Default Value: true
  * @return the retry
  */
  public Boolean getRetry() {
    return config.getOptionalValue("retry", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the retry.max-retries value from the configuration.
  * Attribute Name: retry.max-retries
  * Description: Maximum number of retries for terminal stream errors.
  * Default Value: 3
  * @return the retry.max-retries
  */
  public Integer getRetryMaxRetries() {
    return config.getOptionalValue("retry.max-retries", Integer.class)
     .orElse(Integer.valueOf("3"));
  }

  /**
  * Gets the retry.initial-delay value from the configuration.
  * Attribute Name: retry.initial-delay
  * Description: The initial delay for the retry.
  * Default Value: PT1S
  * @return the retry.initial-delay
  */
  public String getRetryInitialDelay() {
    return config.getOptionalValue("retry.initial-delay", String.class)
     .orElse("PT1S");
  }

  /**
  * Gets the retry.max-delay value from the configuration.
  * Attribute Name: retry.max-delay
  * Description: The maximum delay
  * Default Value: PT10S
  * @return the retry.max-delay
  */
  public String getRetryMaxDelay() {
    return config.getOptionalValue("retry.max-delay", String.class)
     .orElse("PT10S");
  }

  /**
  * Gets the retry.jitter value from the configuration.
  * Attribute Name: retry.jitter
  * Description: How much the delay jitters as a multiplier between 0 and 1. The formula is current delay * jitter. For example, with a current delay of 2H, a jitter of 0.5 will result in an actual delay somewhere between 1H and 3H.
  * Default Value: 0.5
  * @return the retry.jitter
  */
  public Double getRetryJitter() {
    return config.getOptionalValue("retry.jitter", Double.class)
     .orElse(Double.valueOf("0.5"));
  }

  public void validate() {
  }
}
