package io.smallrye.reactive.messaging.jms;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;

/**
 * Extract the outgoing configuration for the {@code smallrye-jms} connector.
*/
public class JmsConnectorOutgoingConfiguration extends JmsConnectorCommonConfiguration {

  /**
   * Creates a new JmsConnectorOutgoingConfiguration.
   */
  public JmsConnectorOutgoingConfiguration(Config config) {
    super(config);
    validate();
  }

  /**
  * Gets the disable-message-id value from the configuration.
  * Attribute Name: disable-message-id
  * Description: Omit the message id in the outbound JMS message
  * @return the disable-message-id
  */
  public Optional<Boolean> getDisableMessageId() {
    return config.getOptionalValue("disable-message-id", Boolean.class);
  }

  /**
  * Gets the disable-message-timestamp value from the configuration.
  * Attribute Name: disable-message-timestamp
  * Description: Omit the message timestamp in the outbound JMS message
  * @return the disable-message-timestamp
  */
  public Optional<Boolean> getDisableMessageTimestamp() {
    return config.getOptionalValue("disable-message-timestamp", Boolean.class);
  }

  /**
  * Gets the delivery-mode value from the configuration.
  * Attribute Name: delivery-mode
  * Description: The delivery mode. Either `persistent` or `non_persistent`
  * @return the delivery-mode
  */
  public Optional<String> getDeliveryMode() {
    return config.getOptionalValue("delivery-mode", String.class);
  }

  /**
  * Gets the delivery-delay value from the configuration.
  * Attribute Name: delivery-delay
  * Description: The delivery delay
  * @return the delivery-delay
  */
  public Optional<Long> getDeliveryDelay() {
    return config.getOptionalValue("delivery-delay", Long.class);
  }

  /**
  * Gets the ttl value from the configuration.
  * Attribute Name: ttl
  * Description: The JMS Message time-to-live
  * @return the ttl
  */
  public Optional<Long> getTtl() {
    return config.getOptionalValue("ttl", Long.class);
  }

  /**
  * Gets the correlation-id value from the configuration.
  * Attribute Name: correlation-id
  * Description: The JMS Message correlation id
  * @return the correlation-id
  */
  public Optional<String> getCorrelationId() {
    return config.getOptionalValue("correlation-id", String.class);
  }

  /**
  * Gets the priority value from the configuration.
  * Attribute Name: priority
  * Description: The JMS Message priority
  * @return the priority
  */
  public Optional<Integer> getPriority() {
    return config.getOptionalValue("priority", Integer.class);
  }

  /**
  * Gets the reply-to value from the configuration.
  * Attribute Name: reply-to
  * Description: The reply to destination if any
  * @return the reply-to
  */
  public Optional<String> getReplyTo() {
    return config.getOptionalValue("reply-to", String.class);
  }

  /**
  * Gets the reply-to-destination-type value from the configuration.
  * Attribute Name: reply-to-destination-type
  * Description: The type of destination for the response. It can be either `queue` or `topic`
  * Default Value: queue
  * @return the reply-to-destination-type
  */
  public String getReplyToDestinationType() {
    return config.getOptionalValue("reply-to-destination-type", String.class)
     .orElse("queue");
  }

  /**
  * Gets the merge value from the configuration.
  * Attribute Name: merge
  * Description: Whether the connector should allow multiple upstreams
  * Default Value: false
  * @return the merge
  */
  public Boolean getMerge() {
    return config.getOptionalValue("merge", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  public void validate() {
    super.validate();
  }
}
