/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.reactive.messaging.kafka.impl;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanBuilder;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.context.Context;
import io.opentelemetry.semconv.trace.attributes.SemanticAttributes;
import io.smallrye.mutiny.Multi;
import io.smallrye.reactive.messaging.TracingMetadata;
import io.smallrye.reactive.messaging.health.HealthReport;
import io.smallrye.reactive.messaging.kafka.DeserializationFailureHandler;
import io.smallrye.reactive.messaging.kafka.IncomingKafkaRecord;
import io.smallrye.reactive.messaging.kafka.KafkaCDIEvents;
import io.smallrye.reactive.messaging.kafka.KafkaConnector;
import io.smallrye.reactive.messaging.kafka.KafkaConnectorIncomingConfiguration;
import io.smallrye.reactive.messaging.kafka.KafkaConsumerRebalanceListener;
import io.smallrye.reactive.messaging.kafka.commit.ContextHolder;
import io.smallrye.reactive.messaging.kafka.commit.KafkaCommitHandler;
import io.smallrye.reactive.messaging.kafka.commit.KafkaIgnoreCommit;
import io.smallrye.reactive.messaging.kafka.commit.KafkaLatestCommit;
import io.smallrye.reactive.messaging.kafka.commit.KafkaThrottledLatestProcessedCommit;
import io.smallrye.reactive.messaging.kafka.fault.KafkaDeadLetterQueue;
import io.smallrye.reactive.messaging.kafka.fault.KafkaFailStop;
import io.smallrye.reactive.messaging.kafka.fault.KafkaFailureHandler;
import io.smallrye.reactive.messaging.kafka.fault.KafkaIgnoreFailure;
import io.smallrye.reactive.messaging.kafka.health.KafkaSourceHealth;
import io.smallrye.reactive.messaging.kafka.i18n.KafkaExceptions;
import io.smallrye.reactive.messaging.kafka.i18n.KafkaLogging;
import io.smallrye.reactive.messaging.kafka.impl.ReactiveKafkaConsumer;
import io.vertx.core.impl.EventLoopContext;
import io.vertx.core.impl.VertxInternal;
import io.vertx.mutiny.core.Vertx;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.enterprise.inject.Instance;
import org.apache.kafka.common.errors.RebalanceInProgressException;

public class KafkaSource<K, V> {
    private final Multi<IncomingKafkaRecord<K, V>> stream;
    private final KafkaFailureHandler failureHandler;
    private final KafkaCommitHandler commitHandler;
    private final KafkaConnectorIncomingConfiguration configuration;
    private final List<Throwable> failures = new ArrayList<Throwable>();
    private final Set<String> topics;
    private final boolean isTracingEnabled;
    private final boolean isHealthEnabled;
    private final boolean isCloudEventEnabled;
    private final String channel;
    private volatile boolean subscribed;
    private final KafkaSourceHealth health;
    private final String group;
    private final int index;
    private final Instance<DeserializationFailureHandler<?>> deserializationFailureHandlers;
    private final Instance<KafkaConsumerRebalanceListener> consumerRebalanceListeners;
    private final ReactiveKafkaConsumer<K, V> client;
    private final EventLoopContext context;

    public KafkaSource(Vertx vertx, String consumerGroup, KafkaConnectorIncomingConfiguration config, Instance<KafkaConsumerRebalanceListener> consumerRebalanceListeners, KafkaCDIEvents kafkaCDIEvents, Instance<DeserializationFailureHandler<?>> deserializationFailureHandlers, int index) {
        Pattern pattern;
        this.group = consumerGroup;
        this.index = index;
        this.deserializationFailureHandlers = deserializationFailureHandlers;
        this.consumerRebalanceListeners = consumerRebalanceListeners;
        this.topics = this.getTopics(config);
        if (config.getPattern().booleanValue()) {
            pattern = Pattern.compile(config.getTopic().orElseThrow(() -> new IllegalArgumentException("Invalid Kafka incoming configuration for channel `" + config.getChannel() + "`, `pattern` must be used with the `topic` attribute")));
            KafkaLogging.log.configuredPattern(config.getChannel(), pattern.toString());
        } else {
            KafkaLogging.log.configuredTopics(config.getChannel(), this.topics);
            pattern = null;
        }
        this.configuration = config;
        this.context = ((VertxInternal)vertx.getDelegate()).createEventLoopContext();
        this.client = new ReactiveKafkaConsumer(config, this);
        String commitStrategy = config.getCommitStrategy().orElse(Boolean.parseBoolean(this.client.get("enable.auto.commit")) ? KafkaCommitHandler.Strategy.IGNORE.name() : KafkaCommitHandler.Strategy.THROTTLED.name());
        this.commitHandler = this.createCommitHandler(vertx, this.client, consumerGroup, config, commitStrategy);
        this.failureHandler = this.createFailureHandler(config, this.client.configuration(), kafkaCDIEvents);
        this.health = this.configuration.getHealthEnabled() != false ? new KafkaSourceHealth(this, this.configuration, this.client) : null;
        this.isTracingEnabled = this.configuration.getTracingEnabled();
        this.isHealthEnabled = this.configuration.getHealthEnabled();
        this.isCloudEventEnabled = this.configuration.getCloudEvents();
        this.channel = this.configuration.getChannel();
        kafkaCDIEvents.consumer().fire(this.client.unwrap());
        if (this.commitHandler instanceof ContextHolder) {
            ((ContextHolder)((Object)this.commitHandler)).capture((io.vertx.core.Context)this.context);
        }
        this.client.setRebalanceListener();
        Multi multi = pattern != null ? this.client.subscribe(pattern).onSubscription().invoke(() -> {
            this.subscribed = true;
        }) : this.client.subscribe(this.topics).onSubscription().invoke(() -> {
            this.subscribed = true;
        });
        multi = multi.onFailure().invoke(t -> {
            KafkaLogging.log.unableToReadRecord(this.topics, (Throwable)t);
            this.reportFailure((Throwable)t, false);
        });
        Multi incomingMulti = multi.map(rec -> this.commitHandler.received(new IncomingKafkaRecord(rec, this.commitHandler, this.failureHandler, this.isCloudEventEnabled, this.isTracingEnabled)));
        if (config.getTracingEnabled().booleanValue()) {
            incomingMulti = incomingMulti.onItem().invoke(this::incomingTrace);
        }
        this.stream = incomingMulti.onFailure().invoke(t -> this.reportFailure((Throwable)t, false));
    }

    private Set<String> getTopics(KafkaConnectorIncomingConfiguration config) {
        String list = config.getTopics().orElse(null);
        String top = config.getTopic().orElse(null);
        String channel = config.getChannel();
        boolean isPattern = config.getPattern();
        if (list != null && top != null) {
            throw new IllegalArgumentException("The Kafka incoming configuration for channel `" + channel + "` cannot use `topics` and `topic` at the same time");
        }
        if (list != null && isPattern) {
            throw new IllegalArgumentException("The Kafka incoming configuration for channel `" + channel + "` cannot use `topics` and `pattern` at the same time");
        }
        if (list != null) {
            String[] strings = list.split(",");
            return Arrays.stream(strings).map(String::trim).collect(Collectors.toSet());
        }
        if (top != null) {
            return Collections.singleton(top);
        }
        return Collections.singleton(channel);
    }

    public synchronized void reportFailure(Throwable failure, boolean fatal) {
        if (failure instanceof RebalanceInProgressException) {
            KafkaLogging.log.failureReportedDuringRebalance(this.topics, failure);
            return;
        }
        KafkaLogging.log.failureReported(this.topics, failure);
        if (this.failures.size() == 10) {
            this.failures.remove(0);
        }
        this.failures.add(failure);
        if (fatal && this.client != null) {
            this.client.close();
        }
    }

    public void incomingTrace(IncomingKafkaRecord<K, V> kafkaRecord) {
        if (this.isTracingEnabled && KafkaConnector.TRACER != null) {
            TracingMetadata tracingMetadata = TracingMetadata.fromMessage(kafkaRecord).orElse(TracingMetadata.empty());
            SpanBuilder spanBuilder = KafkaConnector.TRACER.spanBuilder(kafkaRecord.getTopic() + " receive").setSpanKind(SpanKind.CONSUMER);
            Context parentSpanContext = tracingMetadata.getPreviousContext();
            if (parentSpanContext != null) {
                spanBuilder.setParent(parentSpanContext);
            } else {
                spanBuilder.setNoParent();
            }
            Span span = spanBuilder.startSpan();
            span.setAttribute(SemanticAttributes.MESSAGING_KAFKA_PARTITION, kafkaRecord.getPartition());
            span.setAttribute("offset", kafkaRecord.getOffset());
            span.setAttribute(SemanticAttributes.MESSAGING_SYSTEM, (Object)"kafka");
            span.setAttribute(SemanticAttributes.MESSAGING_DESTINATION, (Object)kafkaRecord.getTopic());
            span.setAttribute(SemanticAttributes.MESSAGING_DESTINATION_KIND, (Object)"topic");
            String groupId = this.client.get("group.id");
            String clientId = this.client.get("client.id");
            span.setAttribute("messaging.consumer_id", this.constructConsumerId(groupId, clientId));
            span.setAttribute(SemanticAttributes.MESSAGING_KAFKA_CONSUMER_GROUP, (Object)groupId);
            if (!clientId.isEmpty()) {
                span.setAttribute(SemanticAttributes.MESSAGING_KAFKA_CLIENT_ID, (Object)clientId);
            }
            span.makeCurrent();
            kafkaRecord.injectTracingMetadata(tracingMetadata.withSpan(span));
            span.end();
        }
    }

    private String constructConsumerId(String groupId, String clientId) {
        String consumerId = groupId;
        if (!clientId.isEmpty()) {
            consumerId = consumerId + " - " + clientId;
        }
        return consumerId;
    }

    private KafkaFailureHandler createFailureHandler(KafkaConnectorIncomingConfiguration config, Map<String, ?> kafkaConfiguration, KafkaCDIEvents kafkaCDIEvents) {
        String strategy = config.getFailureStrategy();
        KafkaFailureHandler.Strategy actualStrategy = KafkaFailureHandler.Strategy.from(strategy);
        switch (actualStrategy) {
            case FAIL: {
                return new KafkaFailStop(config.getChannel(), this);
            }
            case IGNORE: {
                return new KafkaIgnoreFailure(config.getChannel());
            }
            case DEAD_LETTER_QUEUE: {
                return KafkaDeadLetterQueue.create(kafkaConfiguration, config, this, kafkaCDIEvents);
            }
        }
        throw KafkaExceptions.ex.illegalArgumentInvalidFailureStrategy(strategy);
    }

    private KafkaCommitHandler createCommitHandler(Vertx vertx, ReactiveKafkaConsumer<K, V> consumer, String group, KafkaConnectorIncomingConfiguration config, String strategy) {
        KafkaCommitHandler.Strategy actualStrategy = KafkaCommitHandler.Strategy.from(strategy);
        switch (actualStrategy) {
            case LATEST: {
                KafkaLogging.log.commitStrategyForChannel("latest", config.getChannel());
                return new KafkaLatestCommit(vertx, this.configuration, consumer);
            }
            case IGNORE: {
                KafkaLogging.log.commitStrategyForChannel("ignore", config.getChannel());
                return new KafkaIgnoreCommit();
            }
            case THROTTLED: {
                KafkaLogging.log.commitStrategyForChannel("throttled", config.getChannel());
                return KafkaThrottledLatestProcessedCommit.create(vertx, consumer, group, config, this);
            }
        }
        throw KafkaExceptions.ex.illegalArgumentInvalidCommitStrategy(strategy);
    }

    public Multi<IncomingKafkaRecord<K, V>> getStream() {
        return this.stream;
    }

    public void closeQuietly() {
        try {
            if (this.configuration.getGracefulShutdown().booleanValue()) {
                Duration pollTimeoutTwice = Duration.ofMillis((long)this.configuration.getPollTimeout().intValue() * 2L);
                if (((Boolean)this.client.runOnPollingThread(c -> {
                    Set partitions = c.assignment();
                    if (!partitions.isEmpty()) {
                        KafkaLogging.log.pauseAllPartitionOnTermination();
                        c.pause((Collection)partitions);
                        return true;
                    }
                    return false;
                }).await().atMost(pollTimeoutTwice)).booleanValue()) {
                    this.grace(pollTimeoutTwice);
                }
            }
            this.commitHandler.terminate(this.configuration.getGracefulShutdown());
            this.failureHandler.terminate();
        }
        catch (Throwable e) {
            KafkaLogging.log.exceptionOnClose(e);
        }
        try {
            this.client.close();
        }
        catch (Throwable e) {
            KafkaLogging.log.exceptionOnClose(e);
        }
        if (this.health != null) {
            this.health.close();
        }
    }

    private void grace(Duration duration) {
        try {
            Thread.sleep(duration.toMillis());
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void isAlive(HealthReport.HealthReportBuilder builder) {
        if (this.isHealthEnabled) {
            ArrayList<Throwable> actualFailures;
            KafkaSource kafkaSource = this;
            synchronized (kafkaSource) {
                actualFailures = new ArrayList<Throwable>(this.failures);
            }
            if (!actualFailures.isEmpty()) {
                builder.add(this.channel, false, actualFailures.stream().map(Throwable::getMessage).collect(Collectors.joining()));
            } else {
                builder.add(this.channel, true);
            }
        }
    }

    public void isReady(HealthReport.HealthReportBuilder builder) {
        if (this.health != null && this.configuration.getHealthReadinessEnabled().booleanValue()) {
            this.health.isReady(builder);
        }
    }

    public void isStarted(HealthReport.HealthReportBuilder builder) {
        if (this.health != null) {
            this.health.isStarted(builder);
        }
    }

    public ReactiveKafkaConsumer<K, V> getConsumer() {
        return this.client;
    }

    String getConsumerGroup() {
        return this.group;
    }

    int getConsumerIndex() {
        return this.index;
    }

    Instance<DeserializationFailureHandler<?>> getDeserializationFailureHandlers() {
        return this.deserializationFailureHandlers;
    }

    Instance<KafkaConsumerRebalanceListener> getConsumerRebalanceListeners() {
        return this.consumerRebalanceListeners;
    }

    public KafkaCommitHandler getCommitHandler() {
        return this.commitHandler;
    }

    io.vertx.mutiny.core.Context getContext() {
        return new io.vertx.mutiny.core.Context((io.vertx.core.Context)this.context);
    }

    public String getChannel() {
        return this.configuration.getChannel();
    }

    public boolean hasSubscribers() {
        return this.subscribed;
    }
}

