package io.smallrye.reactive.messaging.pulsar;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code smallrye-pulsar} connector.
*/
 public class PulsarConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new PulsarConnectorCommonConfiguration.
   */
  public PulsarConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the client-configuration value from the configuration.
  * Attribute Name: client-configuration
  * Description: Identifier of a CDI bean that provides the default Pulsar client configuration for this channel. The channel configuration can still override any attribute. The bean must have a type of Map<String, Object> and must use the @io.smallrye.common.annotation.Identifier qualifier to set the identifier.
  * @return the client-configuration
  */
  public Optional<String> getClientConfiguration() {
    return config.getOptionalValue("client-configuration", String.class);
  }

  /**
  * Gets the serviceUrl value from the configuration.
  * Attribute Name: serviceUrl
  * Description: The service URL for the Pulsar service
  * Default Value: pulsar://localhost:6650
  * @return the serviceUrl
  */
  public String getServiceUrl() {
    return config.getOptionalValue("serviceUrl", String.class)
     .orElse("pulsar://localhost:6650");
  }

  /**
  * Gets the topic value from the configuration.
  * Attribute Name: topic
  * Description: The consumed / populated Pulsar topic. If not set, the channel name is used
  * @return the topic
  */
  public Optional<String> getTopic() {
    return config.getOptionalValue("topic", String.class);
  }

  /**
  * Gets the schema value from the configuration.
  * Attribute Name: schema
  * Description: The Pulsar schema type of this channel. When configured a schema is built with the given SchemaType and used for the channel. When absent, the schema is resolved searching for a CDI bean typed `Schema` qualified with `@Identifier` and the channel name. As a fallback AUTO_CONSUME or AUTO_PRODUCE are used.
  * @return the schema
  */
  public Optional<String> getSchema() {
    return config.getOptionalValue("schema", String.class);
  }

  /**
  * Gets the health-enabled value from the configuration.
  * Attribute Name: health-enabled
  * Description: Whether health reporting is enabled (default) or disabled
  * Default Value: true
  * @return the health-enabled
  */
  public Boolean getHealthEnabled() {
    return config.getOptionalValue("health-enabled", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the tracing-enabled value from the configuration.
  * Attribute Name: tracing-enabled
  * Description: Whether tracing is enabled (default) or disabled
  * Default Value: true
  * @return the tracing-enabled
  */
  public Boolean getTracingEnabled() {
    return config.getOptionalValue("tracing-enabled", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  public void validate() {
  }
}
