package io.smallrye.reactive.messaging.pulsar;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;

/**
 * Extract the outgoing configuration for the {@code smallrye-pulsar} connector.
*/
public class PulsarConnectorOutgoingConfiguration extends PulsarConnectorCommonConfiguration {

  /**
   * Creates a new PulsarConnectorOutgoingConfiguration.
   */
  public PulsarConnectorOutgoingConfiguration(Config config) {
    super(config);
    validate();
  }

  /**
  * Gets the producer-configuration value from the configuration.
  * Attribute Name: producer-configuration
  * Description: Identifier of a CDI bean that provides the default Pulsar producer configuration for this channel. The channel configuration can still override any attribute. The bean must have a type of Map<String, Object> and must use the @io.smallrye.common.annotation.Identifier qualifier to set the identifier.
  * @return the producer-configuration
  */
  public Optional<String> getProducerConfiguration() {
    return config.getOptionalValue("producer-configuration", String.class);
  }

  /**
  * Gets the maxPendingMessages value from the configuration.
  * Attribute Name: maxPendingMessages
  * Description: The maximum size of a queue holding pending messages, i.e messages waiting to receive an acknowledgment from a broker
  * Default Value: 1000
  * @return the maxPendingMessages
  */
  public Integer getMaxPendingMessages() {
    return config.getOptionalValue("maxPendingMessages", Integer.class)
     .orElse(Integer.valueOf("1000"));
  }

  /**
  * Gets the waitForWriteCompletion value from the configuration.
  * Attribute Name: waitForWriteCompletion
  * Description: Whether the client waits for the broker to acknowledge the written record before acknowledging the message
  * Default Value: true
  * @return the waitForWriteCompletion
  */
  public Boolean getWaitForWriteCompletion() {
    return config.getOptionalValue("waitForWriteCompletion", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  public void validate() {
    super.validate();
  }
}
