package io.smallrye.reactive.messaging.rabbitmq;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code smallrye-rabbitmq} connector.
*/
 public class RabbitMQConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new RabbitMQConnectorCommonConfiguration.
   */
  public RabbitMQConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the username value from the configuration.
  * Attribute Name: username
  * Description: The username used to authenticate to the broker
  * MicroProfile Config Alias: rabbitmq-username
  * @return the username
  */
  public Optional<String> getUsername() {
    Optional<String> maybe = config.getOptionalValue("username", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-username", String.class);
  }

  /**
  * Gets the password value from the configuration.
  * Attribute Name: password
  * Description: The password used to authenticate to the broker
  * MicroProfile Config Alias: rabbitmq-password
  * @return the password
  */
  public Optional<String> getPassword() {
    Optional<String> maybe = config.getOptionalValue("password", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-password", String.class);
  }

  /**
  * Gets the host value from the configuration.
  * Attribute Name: host
  * Description: The broker hostname
  * MicroProfile Config Alias: rabbitmq-host
  * Default Value: localhost
  * @return the host
  */
  public String getHost() {
    return config.getOptionalValue("host", String.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-host", String.class, "localhost"));
  }

  /**
  * Gets the port value from the configuration.
  * Attribute Name: port
  * Description: The broker port
  * MicroProfile Config Alias: rabbitmq-port
  * Default Value: 5672
  * @return the port
  */
  public Integer getPort() {
    return config.getOptionalValue("port", Integer.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-port", Integer.class, Integer.valueOf("5672")));
  }

  /**
  * Gets the addresses value from the configuration.
  * Attribute Name: addresses
  * Description: The multiple addresses for cluster mode, when given overrides the host and port
  * MicroProfile Config Alias: rabbitmq-addresses
  * @return the addresses
  */
  public Optional<String> getAddresses() {
    Optional<String> maybe = config.getOptionalValue("addresses", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-addresses", String.class);
  }

  /**
  * Gets the ssl value from the configuration.
  * Attribute Name: ssl
  * Description: Whether or not the connection should use SSL
  * MicroProfile Config Alias: rabbitmq-ssl
  * Default Value: false
  * @return the ssl
  */
  public Boolean getSsl() {
    return config.getOptionalValue("ssl", Boolean.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-ssl", Boolean.class, Boolean.valueOf("false")));
  }

  /**
  * Gets the trust-all value from the configuration.
  * Attribute Name: trust-all
  * Description: Whether to skip trust certificate verification
  * MicroProfile Config Alias: rabbitmq-trust-all
  * Default Value: false
  * @return the trust-all
  */
  public Boolean getTrustAll() {
    return config.getOptionalValue("trust-all", Boolean.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-trust-all", Boolean.class, Boolean.valueOf("false")));
  }

  /**
  * Gets the trust-store-path value from the configuration.
  * Attribute Name: trust-store-path
  * Description: The path to a JKS trust store
  * MicroProfile Config Alias: rabbitmq-trust-store-path
  * @return the trust-store-path
  */
  public Optional<String> getTrustStorePath() {
    Optional<String> maybe = config.getOptionalValue("trust-store-path", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-trust-store-path", String.class);
  }

  /**
  * Gets the trust-store-password value from the configuration.
  * Attribute Name: trust-store-password
  * Description: The password of the JKS trust store
  * MicroProfile Config Alias: rabbitmq-trust-store-password
  * @return the trust-store-password
  */
  public Optional<String> getTrustStorePassword() {
    Optional<String> maybe = config.getOptionalValue("trust-store-password", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-trust-store-password", String.class);
  }

  /**
  * Gets the connection-timeout value from the configuration.
  * Attribute Name: connection-timeout
  * Description: The TCP connection timeout (ms); 0 is interpreted as no timeout
  * Default Value: 60000
  * @return the connection-timeout
  */
  public Integer getConnectionTimeout() {
    return config.getOptionalValue("connection-timeout", Integer.class)
     .orElse(Integer.valueOf("60000"));
  }

  /**
  * Gets the handshake-timeout value from the configuration.
  * Attribute Name: handshake-timeout
  * Description: The AMQP 0-9-1 protocol handshake timeout (ms)
  * Default Value: 10000
  * @return the handshake-timeout
  */
  public Integer getHandshakeTimeout() {
    return config.getOptionalValue("handshake-timeout", Integer.class)
     .orElse(Integer.valueOf("10000"));
  }

  /**
  * Gets the automatic-recovery-enabled value from the configuration.
  * Attribute Name: automatic-recovery-enabled
  * Description: Whether automatic connection recovery is enabled
  * Default Value: false
  * @return the automatic-recovery-enabled
  */
  public Boolean getAutomaticRecoveryEnabled() {
    return config.getOptionalValue("automatic-recovery-enabled", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the automatic-recovery-on-initial-connection value from the configuration.
  * Attribute Name: automatic-recovery-on-initial-connection
  * Description: Whether automatic recovery on initial connections is enabled
  * Default Value: true
  * @return the automatic-recovery-on-initial-connection
  */
  public Boolean getAutomaticRecoveryOnInitialConnection() {
    return config.getOptionalValue("automatic-recovery-on-initial-connection", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the reconnect-attempts value from the configuration.
  * Attribute Name: reconnect-attempts
  * Description: The number of reconnection attempts
  * MicroProfile Config Alias: rabbitmq-reconnect-attempts
  * Default Value: 100
  * @return the reconnect-attempts
  */
  public Integer getReconnectAttempts() {
    return config.getOptionalValue("reconnect-attempts", Integer.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-reconnect-attempts", Integer.class, Integer.valueOf("100")));
  }

  /**
  * Gets the reconnect-interval value from the configuration.
  * Attribute Name: reconnect-interval
  * Description: The interval (in seconds) between two reconnection attempts
  * MicroProfile Config Alias: rabbitmq-reconnect-interval
  * Default Value: 10
  * @return the reconnect-interval
  */
  public Integer getReconnectInterval() {
    return config.getOptionalValue("reconnect-interval", Integer.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-reconnect-interval", Integer.class, Integer.valueOf("10")));
  }

  /**
  * Gets the network-recovery-interval value from the configuration.
  * Attribute Name: network-recovery-interval
  * Description: How long (ms) will automatic recovery wait before attempting to reconnect
  * Default Value: 5000
  * @return the network-recovery-interval
  */
  public Integer getNetworkRecoveryInterval() {
    return config.getOptionalValue("network-recovery-interval", Integer.class)
     .orElse(Integer.valueOf("5000"));
  }

  /**
  * Gets the user value from the configuration.
  * Attribute Name: user
  * Description: The user name to use when connecting to the broker
  * Default Value: guest
  * @return the user
  */
  public String getUser() {
    return config.getOptionalValue("user", String.class)
     .orElse("guest");
  }

  /**
  * Gets the include-properties value from the configuration.
  * Attribute Name: include-properties
  * Description: Whether to include properties when a broker message is passed on the event bus
  * Default Value: false
  * @return the include-properties
  */
  public Boolean getIncludeProperties() {
    return config.getOptionalValue("include-properties", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the requested-channel-max value from the configuration.
  * Attribute Name: requested-channel-max
  * Description: The initially requested maximum channel number
  * Default Value: 2047
  * @return the requested-channel-max
  */
  public Integer getRequestedChannelMax() {
    return config.getOptionalValue("requested-channel-max", Integer.class)
     .orElse(Integer.valueOf("2047"));
  }

  /**
  * Gets the requested-heartbeat value from the configuration.
  * Attribute Name: requested-heartbeat
  * Description: The initially requested heartbeat interval (seconds), zero for none
  * Default Value: 60
  * @return the requested-heartbeat
  */
  public Integer getRequestedHeartbeat() {
    return config.getOptionalValue("requested-heartbeat", Integer.class)
     .orElse(Integer.valueOf("60"));
  }

  /**
  * Gets the use-nio value from the configuration.
  * Attribute Name: use-nio
  * Description: Whether usage of NIO Sockets is enabled
  * Default Value: false
  * @return the use-nio
  */
  public Boolean getUseNio() {
    return config.getOptionalValue("use-nio", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the virtual-host value from the configuration.
  * Attribute Name: virtual-host
  * Description: The virtual host to use when connecting to the broker
  * MicroProfile Config Alias: rabbitmq-virtual-host
  * Default Value: /
  * @return the virtual-host
  */
  public String getVirtualHost() {
    return config.getOptionalValue("virtual-host", String.class)
     .orElseGet(() -> getFromAliasWithDefaultValue("rabbitmq-virtual-host", String.class, "/"));
  }

  /**
  * Gets the client-options-name value from the configuration.
  * Attribute Name: client-options-name
  * Description: The name of the RabbitMQ Client Option bean used to customize the RabbitMQ client configuration
  * MicroProfile Config Alias: rabbitmq-client-options-name
  * @return the client-options-name
  */
  public Optional<String> getClientOptionsName() {
    Optional<String> maybe = config.getOptionalValue("client-options-name", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-client-options-name", String.class);
  }

  /**
  * Gets the credentials-provider-name value from the configuration.
  * Attribute Name: credentials-provider-name
  * Description: The name of the RabbitMQ Credentials Provider bean used to provide dynamic credentials to the RabbitMQ client
  * MicroProfile Config Alias: rabbitmq-credentials-provider-name
  * @return the credentials-provider-name
  */
  public Optional<String> getCredentialsProviderName() {
    Optional<String> maybe = config.getOptionalValue("credentials-provider-name", String.class);
    if (maybe.isPresent()) { return maybe; }
    return getFromAlias("rabbitmq-credentials-provider-name", String.class);
  }

  /**
  * Gets the exchange.name value from the configuration.
  * Attribute Name: exchange.name
  * Description: The exchange that messages are published to or consumed from. If not set, the channel name is used. If set to "", the default exchange is used.
  * @return the exchange.name
  */
  public Optional<String> getExchangeName() {
    return config.getOptionalValue("exchange.name", String.class);
  }

  /**
  * Gets the exchange.durable value from the configuration.
  * Attribute Name: exchange.durable
  * Description: Whether the exchange is durable
  * Default Value: true
  * @return the exchange.durable
  */
  public Boolean getExchangeDurable() {
    return config.getOptionalValue("exchange.durable", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the exchange.auto-delete value from the configuration.
  * Attribute Name: exchange.auto-delete
  * Description: Whether the exchange should be deleted after use
  * Default Value: false
  * @return the exchange.auto-delete
  */
  public Boolean getExchangeAutoDelete() {
    return config.getOptionalValue("exchange.auto-delete", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the exchange.type value from the configuration.
  * Attribute Name: exchange.type
  * Description: The exchange type: direct, fanout, headers or topic (default)
  * Default Value: topic
  * @return the exchange.type
  */
  public String getExchangeType() {
    return config.getOptionalValue("exchange.type", String.class)
     .orElse("topic");
  }

  /**
  * Gets the exchange.declare value from the configuration.
  * Attribute Name: exchange.declare
  * Description: Whether to declare the exchange; set to false if the exchange is expected to be set up independently
  * Default Value: true
  * @return the exchange.declare
  */
  public Boolean getExchangeDeclare() {
    return config.getOptionalValue("exchange.declare", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the tracing.enabled value from the configuration.
  * Attribute Name: tracing.enabled
  * Description: Whether tracing is enabled (default) or disabled
  * Default Value: true
  * @return the tracing.enabled
  */
  public Boolean getTracingEnabled() {
    return config.getOptionalValue("tracing.enabled", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the tracing.attribute-headers value from the configuration.
  * Attribute Name: tracing.attribute-headers
  * Description: A comma-separated list of headers that should be recorded as span attributes. Relevant only if tracing.enabled=true
  * Default Value: 
  * @return the tracing.attribute-headers
  */
  public String getTracingAttributeHeaders() {
    return config.getOptionalValue("tracing.attribute-headers", String.class)
     .orElse("");
  }

  public void validate() {
  }
}
