package io.smallrye.reactive.messaging.rabbitmq;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;

/**
 * Extract the incoming configuration for the {@code smallrye-rabbitmq} connector.
*/
public class RabbitMQConnectorIncomingConfiguration extends RabbitMQConnectorCommonConfiguration {

  /**
   * Creates a new RabbitMQConnectorIncomingConfiguration.
   */
  public RabbitMQConnectorIncomingConfiguration(Config config) {
    super(config);
    validate();
  }

  /**
  * Gets the queue.name value from the configuration.
  * Attribute Name: queue.name
  * Description: The queue from which messages are consumed. If not set, the channel name is used.
  * @return the queue.name
  */
  public Optional<String> getQueueName() {
    return config.getOptionalValue("queue.name", String.class);
  }

  /**
  * Gets the queue.durable value from the configuration.
  * Attribute Name: queue.durable
  * Description: Whether the queue is durable
  * Default Value: true
  * @return the queue.durable
  */
  public Boolean getQueueDurable() {
    return config.getOptionalValue("queue.durable", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the queue.exclusive value from the configuration.
  * Attribute Name: queue.exclusive
  * Description: Whether the queue is for exclusive use
  * Default Value: false
  * @return the queue.exclusive
  */
  public Boolean getQueueExclusive() {
    return config.getOptionalValue("queue.exclusive", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the queue.auto-delete value from the configuration.
  * Attribute Name: queue.auto-delete
  * Description: Whether the queue should be deleted after use
  * Default Value: false
  * @return the queue.auto-delete
  */
  public Boolean getQueueAutoDelete() {
    return config.getOptionalValue("queue.auto-delete", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the queue.declare value from the configuration.
  * Attribute Name: queue.declare
  * Description: Whether to declare the queue and binding; set to false if these are expected to be set up independently
  * Default Value: true
  * @return the queue.declare
  */
  public Boolean getQueueDeclare() {
    return config.getOptionalValue("queue.declare", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the queue.ttl value from the configuration.
  * Attribute Name: queue.ttl
  * Description: If specified, the time (ms) for which a message can remain in the queue undelivered before it is dead
  * @return the queue.ttl
  */
  public Optional<Long> getQueueTtl() {
    return config.getOptionalValue("queue.ttl", Long.class);
  }

  /**
  * Gets the queue.single-active-consumer value from the configuration.
  * Attribute Name: queue.single-active-consumer
  * Description: If set to true, only one consumer can actively consume messages
  * @return the queue.single-active-consumer
  */
  public Optional<Boolean> getQueueSingleActiveConsumer() {
    return config.getOptionalValue("queue.single-active-consumer", Boolean.class);
  }

  /**
  * Gets the queue.x-queue-type value from the configuration.
  * Attribute Name: queue.x-queue-type
  * Description: If automatically declare queue, we can choose different types of queue [quorum, classic, stream]
  * @return the queue.x-queue-type
  */
  public Optional<String> getQueueXQueueType() {
    return config.getOptionalValue("queue.x-queue-type", String.class);
  }

  /**
  * Gets the queue.x-queue-mode value from the configuration.
  * Attribute Name: queue.x-queue-mode
  * Description: If automatically declare queue, we can choose different modes of queue [lazy, default]
  * @return the queue.x-queue-mode
  */
  public Optional<String> getQueueXQueueMode() {
    return config.getOptionalValue("queue.x-queue-mode", String.class);
  }

  /**
  * Gets the max-incoming-internal-queue-size value from the configuration.
  * Attribute Name: max-incoming-internal-queue-size
  * Description: The maximum size of the incoming internal queue
  * Default Value: 500000
  * @return the max-incoming-internal-queue-size
  */
  public Integer getMaxIncomingInternalQueueSize() {
    return config.getOptionalValue("max-incoming-internal-queue-size", Integer.class)
     .orElse(Integer.valueOf("500000"));
  }

  /**
  * Gets the connection-count value from the configuration.
  * Attribute Name: connection-count
  * Description: The number of RabbitMQ connections to create for consuming from this queue. This might be necessary to consume from a sharded queue with a single client.
  * Default Value: 1
  * @return the connection-count
  */
  public Integer getConnectionCount() {
    return config.getOptionalValue("connection-count", Integer.class)
     .orElse(Integer.valueOf("1"));
  }

  /**
  * Gets the queue.x-max-priority value from the configuration.
  * Attribute Name: queue.x-max-priority
  * Description: Define priority level queue consumer
  * @return the queue.x-max-priority
  */
  public Optional<Integer> getQueueXMaxPriority() {
    return config.getOptionalValue("queue.x-max-priority", Integer.class);
  }

  /**
  * Gets the queue.x-delivery-limit value from the configuration.
  * Attribute Name: queue.x-delivery-limit
  * Description: If queue.x-queue-type is quorum, when a message has been returned more times than the limit the message will be dropped or dead-lettered
  * @return the queue.x-delivery-limit
  */
  public Optional<Long> getQueueXDeliveryLimit() {
    return config.getOptionalValue("queue.x-delivery-limit", Long.class);
  }

  /**
  * Gets the auto-bind-dlq value from the configuration.
  * Attribute Name: auto-bind-dlq
  * Description: Whether to automatically declare the DLQ and bind it to the binder DLX
  * Default Value: false
  * @return the auto-bind-dlq
  */
  public Boolean getAutoBindDlq() {
    return config.getOptionalValue("auto-bind-dlq", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the dead-letter-queue-name value from the configuration.
  * Attribute Name: dead-letter-queue-name
  * Description: The name of the DLQ; if not supplied will default to the queue name with '.dlq' appended
  * @return the dead-letter-queue-name
  */
  public Optional<String> getDeadLetterQueueName() {
    return config.getOptionalValue("dead-letter-queue-name", String.class);
  }

  /**
  * Gets the dead-letter-exchange value from the configuration.
  * Attribute Name: dead-letter-exchange
  * Description: A DLX to assign to the queue. Relevant only if auto-bind-dlq is true
  * Default Value: DLX
  * @return the dead-letter-exchange
  */
  public String getDeadLetterExchange() {
    return config.getOptionalValue("dead-letter-exchange", String.class)
     .orElse("DLX");
  }

  /**
  * Gets the dead-letter-exchange-type value from the configuration.
  * Attribute Name: dead-letter-exchange-type
  * Description: The type of the DLX to assign to the queue. Relevant only if auto-bind-dlq is true
  * Default Value: direct
  * @return the dead-letter-exchange-type
  */
  public String getDeadLetterExchangeType() {
    return config.getOptionalValue("dead-letter-exchange-type", String.class)
     .orElse("direct");
  }

  /**
  * Gets the dead-letter-routing-key value from the configuration.
  * Attribute Name: dead-letter-routing-key
  * Description: A dead letter routing key to assign to the queue; if not supplied will default to the queue name
  * @return the dead-letter-routing-key
  */
  public Optional<String> getDeadLetterRoutingKey() {
    return config.getOptionalValue("dead-letter-routing-key", String.class);
  }

  /**
  * Gets the dlx.declare value from the configuration.
  * Attribute Name: dlx.declare
  * Description: Whether to declare the dead letter exchange binding. Relevant only if auto-bind-dlq is true; set to false if these are expected to be set up independently
  * Default Value: false
  * @return the dlx.declare
  */
  public Boolean getDlxDeclare() {
    return config.getOptionalValue("dlx.declare", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the dead-letter-queue-type value from the configuration.
  * Attribute Name: dead-letter-queue-type
  * Description: If automatically declare DLQ, we can choose different types of DLQ [quorum, classic, stream]
  * @return the dead-letter-queue-type
  */
  public Optional<String> getDeadLetterQueueType() {
    return config.getOptionalValue("dead-letter-queue-type", String.class);
  }

  /**
  * Gets the dead-letter-queue-mode value from the configuration.
  * Attribute Name: dead-letter-queue-mode
  * Description: If automatically declare DLQ, we can choose different modes of DLQ [lazy, default]
  * @return the dead-letter-queue-mode
  */
  public Optional<String> getDeadLetterQueueMode() {
    return config.getOptionalValue("dead-letter-queue-mode", String.class);
  }

  /**
  * Gets the dead-letter-ttl value from the configuration.
  * Attribute Name: dead-letter-ttl
  * Description: If specified, the time (ms) for which a message can remain in DLQ undelivered before it is dead. Relevant only if auto-bind-dlq is true
  * @return the dead-letter-ttl
  */
  public Optional<Long> getDeadLetterTtl() {
    return config.getOptionalValue("dead-letter-ttl", Long.class);
  }

  /**
  * Gets the dead-letter-dlx value from the configuration.
  * Attribute Name: dead-letter-dlx
  * Description: If specified, a DLX to assign to the DLQ. Relevant only if auto-bind-dlq is true
  * @return the dead-letter-dlx
  */
  public Optional<String> getDeadLetterDlx() {
    return config.getOptionalValue("dead-letter-dlx", String.class);
  }

  /**
  * Gets the dead-letter-dlx-routing-key value from the configuration.
  * Attribute Name: dead-letter-dlx-routing-key
  * Description: If specified, a dead letter routing key to assign to the DLQ. Relevant only if auto-bind-dlq is true
  * @return the dead-letter-dlx-routing-key
  */
  public Optional<String> getDeadLetterDlxRoutingKey() {
    return config.getOptionalValue("dead-letter-dlx-routing-key", String.class);
  }

  /**
  * Gets the failure-strategy value from the configuration.
  * Attribute Name: failure-strategy
  * Description: The failure strategy to apply when a RabbitMQ message is nacked. Accepted values are `fail`, `accept`, `reject` (default), `requeue` or name of a bean
  * Default Value: reject
  * @return the failure-strategy
  */
  public String getFailureStrategy() {
    return config.getOptionalValue("failure-strategy", String.class)
     .orElse("reject");
  }

  /**
  * Gets the broadcast value from the configuration.
  * Attribute Name: broadcast
  * Description: Whether the received RabbitMQ messages must be dispatched to multiple _subscribers_
  * Default Value: false
  * @return the broadcast
  */
  public Boolean getBroadcast() {
    return config.getOptionalValue("broadcast", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the auto-acknowledgement value from the configuration.
  * Attribute Name: auto-acknowledgement
  * Description: Whether the received RabbitMQ messages must be acknowledged when received; if true then delivery constitutes acknowledgement
  * Default Value: false
  * @return the auto-acknowledgement
  */
  public Boolean getAutoAcknowledgement() {
    return config.getOptionalValue("auto-acknowledgement", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the keep-most-recent value from the configuration.
  * Attribute Name: keep-most-recent
  * Description: Whether to discard old messages instead of recent ones
  * Default Value: false
  * @return the keep-most-recent
  */
  public Boolean getKeepMostRecent() {
    return config.getOptionalValue("keep-most-recent", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the routing-keys value from the configuration.
  * Attribute Name: routing-keys
  * Description: A comma-separated list of routing keys to bind the queue to the exchange. Relevant only if 'exchange.type' is topic or direct
  * Default Value: #
  * @return the routing-keys
  */
  public String getRoutingKeys() {
    return config.getOptionalValue("routing-keys", String.class)
     .orElse("#");
  }

  /**
  * Gets the arguments value from the configuration.
  * Attribute Name: arguments
  * Description: A comma-separated list of arguments [key1:value1,key2:value2,...] to bind the queue to the exchange. Relevant only if 'exchange.type' is headers
  * @return the arguments
  */
  public Optional<String> getArguments() {
    return config.getOptionalValue("arguments", String.class);
  }

  /**
  * Gets the content-type-override value from the configuration.
  * Attribute Name: content-type-override
  * Description: Override the content_type attribute of the incoming message, should be a valid MINE type
  * @return the content-type-override
  */
  public Optional<String> getContentTypeOverride() {
    return config.getOptionalValue("content-type-override", String.class);
  }

  /**
  * Gets the max-outstanding-messages value from the configuration.
  * Attribute Name: max-outstanding-messages
  * Description: The maximum number of outstanding/unacknowledged messages being processed by the connector at a time; must be a positive number
  * @return the max-outstanding-messages
  */
  public Optional<Integer> getMaxOutstandingMessages() {
    return config.getOptionalValue("max-outstanding-messages", Integer.class);
  }

  public void validate() {
    super.validate();
  }
}
