package io.smallrye.reactive.messaging.rabbitmq;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;

/**
 * Extract the outgoing configuration for the {@code smallrye-rabbitmq} connector.
*/
public class RabbitMQConnectorOutgoingConfiguration extends RabbitMQConnectorCommonConfiguration {

  /**
   * Creates a new RabbitMQConnectorOutgoingConfiguration.
   */
  public RabbitMQConnectorOutgoingConfiguration(Config config) {
    super(config);
    validate();
  }

  /**
  * Gets the max-outgoing-internal-queue-size value from the configuration.
  * Attribute Name: max-outgoing-internal-queue-size
  * Description: The maximum size of the outgoing internal queue
  * @return the max-outgoing-internal-queue-size
  */
  public Optional<Integer> getMaxOutgoingInternalQueueSize() {
    return config.getOptionalValue("max-outgoing-internal-queue-size", Integer.class);
  }

  /**
  * Gets the max-inflight-messages value from the configuration.
  * Attribute Name: max-inflight-messages
  * Description: The maximum number of messages to be written to RabbitMQ concurrently; must be a positive number
  * Default Value: 1024
  * @return the max-inflight-messages
  */
  public Long getMaxInflightMessages() {
    return config.getOptionalValue("max-inflight-messages", Long.class)
     .orElse(Long.valueOf("1024"));
  }

  /**
  * Gets the default-routing-key value from the configuration.
  * Attribute Name: default-routing-key
  * Description: The default routing key to use when sending messages to the exchange
  * Default Value: 
  * @return the default-routing-key
  */
  public String getDefaultRoutingKey() {
    return config.getOptionalValue("default-routing-key", String.class)
     .orElse("");
  }

  /**
  * Gets the default-ttl value from the configuration.
  * Attribute Name: default-ttl
  * Description: If specified, the time (ms) sent messages can remain in queues undelivered before they are dead
  * @return the default-ttl
  */
  public Optional<Long> getDefaultTtl() {
    return config.getOptionalValue("default-ttl", Long.class);
  }

  /**
  * Gets the publish-confirms value from the configuration.
  * Attribute Name: publish-confirms
  * Description: If set to true, published messages are acknowledged when the publish confirm is received from the broker
  * Default Value: false
  * @return the publish-confirms
  */
  public Boolean getPublishConfirms() {
    return config.getOptionalValue("publish-confirms", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  public void validate() {
    super.validate();
  }
}
