/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance;

import com.netflix.hystrix.HystrixCircuitBreaker;
import io.smallrye.faulttolerance.config.CircuitBreakerConfig;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.jboss.logging.Logger;

public class SynchronousCircuitBreaker
implements HystrixCircuitBreaker {
    private static final Logger LOGGER = Logger.getLogger(SynchronousCircuitBreaker.class);
    private final AtomicReference<Status> status;
    private final AtomicLong circuitOpenedAt;
    private final CircuitBreakerConfig config;
    private final AtomicInteger successCount;
    private final AtomicInteger halfOpenAttempts;
    private final LinkedList<Boolean> rollingWindow;
    private final String id;
    private final AtomicLong lastStatusChangeAt;
    private final AtomicLong openTotal;
    private final AtomicLong halfOpenTotal;
    private final AtomicLong closedTotal;

    SynchronousCircuitBreaker(CircuitBreakerConfig config) {
        this.config = config;
        this.status = new AtomicReference<Status>(Status.CLOSED);
        this.circuitOpenedAt = new AtomicLong(-1L);
        this.successCount = new AtomicInteger(0);
        this.halfOpenAttempts = new AtomicInteger(0);
        this.id = config.getMethodInfo();
        this.openTotal = new AtomicLong();
        this.halfOpenTotal = new AtomicLong();
        this.closedTotal = new AtomicLong();
        this.lastStatusChangeAt = new AtomicLong(System.nanoTime());
        this.rollingWindow = new LinkedList();
    }

    public void markSuccess() {
    }

    public void markNonSuccess() {
    }

    public synchronized boolean isOpen() {
        return this.circuitOpenedAt.get() >= 0L;
    }

    public synchronized boolean allowRequest() {
        switch (this.status.get()) {
            case CLOSED: {
                return true;
            }
            case HALF_OPEN: {
                return this.isHalfOpenAttemptAllowed();
            }
            case OPEN: {
                return this.isAfterDelay();
            }
        }
        return false;
    }

    public synchronized boolean attemptExecution() {
        switch (this.status.get()) {
            case CLOSED: {
                return true;
            }
            case HALF_OPEN: {
                if (this.isHalfOpenAttemptAllowed()) {
                    this.halfOpenAttempts.incrementAndGet();
                    return true;
                }
                return false;
            }
            case OPEN: {
                if (this.isAfterDelay()) {
                    this.toHalfOpen();
                    return true;
                }
                return false;
            }
        }
        return false;
    }

    synchronized void executionSucceeded() {
        this.record(true);
        this.successCount.incrementAndGet();
        Status current = this.status.get();
        if (Status.HALF_OPEN == current && this.isSuccessThresholdReached()) {
            this.toClosed();
        } else if (Status.CLOSED == current && this.isFailureThresholdReached()) {
            this.toOpen(current);
        }
    }

    synchronized void executionFailed() {
        this.record(false);
        Status current = this.status.get();
        if (Status.HALF_OPEN == current || Status.CLOSED == current && this.isFailureThresholdReached()) {
            this.toOpen(current);
        }
    }

    public long getClosedTotal() {
        return this.getTotalVal(Status.CLOSED, this.closedTotal.get());
    }

    public long getOpenTotal() {
        return this.getTotalVal(Status.OPEN, this.openTotal.get());
    }

    public long getHalfOpenTotal() {
        return this.getTotalVal(Status.HALF_OPEN, this.halfOpenTotal.get());
    }

    private long getTotalVal(Status status, long sum) {
        return status.equals((Object)this.status.get()) ? sum + (System.nanoTime() - this.lastStatusChangeAt.get()) : sum;
    }

    private void toClosed() {
        LOGGER.debugf("HALF_OPEN >> CLOSED [id:%s]", (Object)this.id);
        this.status.set(Status.CLOSED);
        this.circuitOpenedAt.set(-1L);
        long currentTime = System.nanoTime();
        this.halfOpenTotal.addAndGet(currentTime - this.lastStatusChangeAt.getAndSet(currentTime));
        this.reset();
    }

    private void toOpen(Status current) {
        LOGGER.debugf("%s >> OPEN [id:%s]", (Object)current, (Object)this.id);
        this.status.set(Status.OPEN);
        this.circuitOpenedAt.set(System.currentTimeMillis());
        long currentTime = System.nanoTime();
        switch (current) {
            case CLOSED: {
                this.closedTotal.addAndGet(currentTime - this.lastStatusChangeAt.getAndSet(currentTime));
                break;
            }
            case HALF_OPEN: {
                this.halfOpenTotal.addAndGet(currentTime - this.lastStatusChangeAt.getAndSet(currentTime));
                break;
            }
        }
        this.reset();
    }

    private void toHalfOpen() {
        LOGGER.debugf("OPEN >> HALF_OPEN [id:%s]", (Object)this.id);
        this.status.set(Status.HALF_OPEN);
        this.halfOpenAttempts.set(1);
        long currentTime = System.nanoTime();
        this.openTotal.addAndGet(currentTime - this.lastStatusChangeAt.getAndSet(currentTime));
        this.reset();
    }

    private boolean isAfterDelay() {
        long elapsed;
        long openedAt = this.circuitOpenedAt.get();
        long delay = (Long)this.config.get("delay");
        if (delay == 0L) {
            return true;
        }
        ChronoUnit delayUnit = (ChronoUnit)this.config.get("delayUnit");
        if (delayUnit.equals(ChronoUnit.MILLIS)) {
            elapsed = System.currentTimeMillis() - openedAt;
        } else {
            Instant start = Instant.ofEpochMilli(openedAt);
            Instant now = Instant.now();
            elapsed = delayUnit.between(start, now);
        }
        return elapsed >= delay;
    }

    private boolean isFailureThresholdReached() {
        double failureRatio;
        int requestCount = this.rollingWindow.size();
        if (!this.isRequestVolumeThresholdReached(requestCount)) {
            return false;
        }
        double failureCheck = (double)this.getFailureCount() / (double)requestCount;
        return failureCheck >= (failureRatio = ((Double)this.config.get("failureRatio")).doubleValue()) || failureRatio <= 0.0 && failureCheck == 1.0;
    }

    private boolean isRequestVolumeThresholdReached(int requestCount) {
        return requestCount >= (Integer)this.config.get("requestVolumeThreshold");
    }

    private boolean isSuccessThresholdReached() {
        return this.successCount.get() >= this.config.get("successThreshold", Integer.class);
    }

    private boolean isHalfOpenAttemptAllowed() {
        return this.halfOpenAttempts.get() < this.config.get("successThreshold", Integer.class);
    }

    private void reset() {
        this.successCount.set(0);
        this.halfOpenAttempts.set(0);
        this.rollingWindow.clear();
    }

    private int getFailureCount() {
        int count = 0;
        for (Boolean result : this.rollingWindow) {
            if (!result.booleanValue()) continue;
            ++count;
        }
        return count;
    }

    private void record(boolean requestResult) {
        this.rollingWindow.addFirst(!requestResult);
        if (this.rollingWindow.size() > (Integer)this.config.get("requestVolumeThreshold")) {
            this.rollingWindow.removeLast();
        }
    }

    public boolean failsOn(Throwable throwable) {
        Class[] exceptions;
        for (Class exception : exceptions = (Class[])this.config.get("failOn")) {
            if (!exception.isAssignableFrom(throwable.getClass())) continue;
            return true;
        }
        return false;
    }

    static enum Status {
        CLOSED,
        OPEN,
        HALF_OPEN;

    }
}

