/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.jwt.auth.principal;

import io.smallrye.jwt.KeyUtils;
import io.smallrye.jwt.auth.principal.JWTAuthContextInfo;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URL;
import java.security.Key;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import org.jboss.logging.Logger;
import org.jose4j.json.JsonUtil;
import org.jose4j.jwk.HttpsJwks;
import org.jose4j.jwk.JsonWebKey;
import org.jose4j.jwk.PublicJsonWebKey;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.jwx.JsonWebStructure;
import org.jose4j.keys.resolvers.VerificationKeyResolver;
import org.jose4j.lang.JoseException;
import org.jose4j.lang.UnresolvableKeyException;

public class KeyLocationResolver
implements VerificationKeyResolver {
    private static final Logger LOGGER = Logger.getLogger(KeyLocationResolver.class);
    private static final String HTTPS_SCHEME = "https:";
    private static final String HTTP_BASED_SCHEME = "http";
    private static final String CLASSPATH_SCHEME = "classpath:";
    private static final String FILE_SCHEME = "file:";
    PublicKey verificationKey;
    private List<JsonWebKey> jsonWebKeys;
    private HttpsJwks httpsJwks;
    private JWTAuthContextInfo authContextInfo;

    public KeyLocationResolver(JWTAuthContextInfo authContextInfo) throws UnresolvableKeyException {
        this.authContextInfo = authContextInfo;
        try {
            this.initializeKeyContent();
        }
        catch (Exception e) {
            throw new UnresolvableKeyException("Failed to load a key from: " + authContextInfo.getPublicKeyLocation(), (Throwable)e);
        }
    }

    public Key resolveKey(JsonWebSignature jws, List<JsonWebStructure> nestingContext) throws UnresolvableKeyException {
        KeyLocationResolver.verifyKid(jws, this.authContextInfo.getTokenKeyId());
        if (this.verificationKey != null) {
            return this.verificationKey;
        }
        PublicKey key = this.tryAsJwk(jws);
        if (key == null) {
            throw new UnresolvableKeyException("Failed to resolve a key from: " + this.authContextInfo.getPublicKeyLocation());
        }
        return key;
    }

    private PublicKey tryAsJwk(JsonWebSignature jws) throws UnresolvableKeyException {
        String kid = KeyLocationResolver.getKid(jws);
        if (this.httpsJwks != null) {
            return this.getHttpsJwk(kid);
        }
        if (this.jsonWebKeys != null) {
            return this.getJsonWebKey(kid);
        }
        return null;
    }

    PublicKey getHttpsJwk(String kid) {
        LOGGER.debugf("Trying to create a key from the HTTPS JWK(S)...", new Object[0]);
        try {
            return KeyLocationResolver.getKeyFromJsonWebKeys(kid, this.httpsJwks.getJsonWebKeys());
        }
        catch (Exception e) {
            LOGGER.debug((Object)"Failed to create a key from the HTTPS JWK(S)", (Throwable)e);
            return null;
        }
    }

    PublicKey getJsonWebKey(String kid) {
        LOGGER.debugf("Trying the create a key from the JWK(S)...", new Object[0]);
        try {
            return KeyLocationResolver.getKeyFromJsonWebKeys(kid, this.jsonWebKeys);
        }
        catch (Exception e) {
            LOGGER.debug((Object)"Failed to create a key from the JWK(S)", (Throwable)e);
            return null;
        }
    }

    private static void verifyKid(JsonWebSignature jws, String expectedKid) throws UnresolvableKeyException {
        String kid;
        if (expectedKid != null && (kid = KeyLocationResolver.getKid(jws)) != null && !kid.equals(expectedKid)) {
            LOGGER.debugf("Invalid token 'kid' header: %s, expected: %s", (Object)kid, (Object)expectedKid);
            throw new UnresolvableKeyException("Invalid token 'kid' header");
        }
    }

    private static String getKid(JsonWebSignature jws) throws UnresolvableKeyException {
        return jws.getHeaders().getStringHeaderValue("kid");
    }

    protected void initializeKeyContent() throws Exception {
        if (this.authContextInfo.getPublicKeyLocation().startsWith(HTTPS_SCHEME)) {
            LOGGER.debugf("Trying to load the keys from the HTTPS JWK(S)...", new Object[0]);
            this.httpsJwks = this.initializeHttpsJwks();
            this.httpsJwks.setDefaultCacheDuration(this.authContextInfo.getJwksRefreshInterval().longValue() * 60L);
            try {
                this.httpsJwks.refresh();
                return;
            }
            catch (JoseException joseException) {
                // empty catch block
            }
        }
        String content = this.readKeyContent(this.authContextInfo.getPublicKeyLocation());
        this.verificationKey = KeyLocationResolver.tryAsPEMPublicKey(content);
        if (this.verificationKey == null) {
            this.verificationKey = KeyLocationResolver.tryAsPEMCertificate(content);
        }
        if (this.verificationKey == null) {
            this.jsonWebKeys = KeyLocationResolver.loadJsonWebKeys(content);
            if (this.jsonWebKeys != null && this.authContextInfo.getTokenKeyId() != null) {
                this.verificationKey = this.getJsonWebKey(this.authContextInfo.getTokenKeyId());
            }
        }
    }

    protected HttpsJwks initializeHttpsJwks() {
        return new HttpsJwks(this.authContextInfo.getPublicKeyLocation());
    }

    protected String readKeyContent(String keyLocation) throws IOException {
        InputStream is = null;
        if (keyLocation.startsWith(HTTP_BASED_SCHEME)) {
            is = this.getUrlResolver().resolve(keyLocation);
        } else if (keyLocation.startsWith(FILE_SCHEME)) {
            is = KeyLocationResolver.getAsFileSystemResource(keyLocation.substring(FILE_SCHEME.length()));
        } else if (keyLocation.startsWith(CLASSPATH_SCHEME)) {
            is = KeyLocationResolver.getAsClasspathResource(keyLocation.substring(CLASSPATH_SCHEME.length()));
        } else {
            is = KeyLocationResolver.getAsFileSystemResource(keyLocation);
            if (is == null) {
                is = KeyLocationResolver.getAsClasspathResource(keyLocation);
            }
            if (is == null) {
                is = this.getUrlResolver().resolve(keyLocation);
            }
        }
        if (is == null) {
            throw new IOException("No resource with the named " + keyLocation + " location exists");
        }
        StringWriter contents = new StringWriter();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(is));){
            String line = null;
            while ((line = reader.readLine()) != null) {
                contents.write(line);
            }
        }
        return contents.toString();
    }

    protected UrlStreamResolver getUrlResolver() {
        return new UrlStreamResolver();
    }

    static PublicKey tryAsPEMPublicKey(String content) {
        LOGGER.debugf("Trying to create a key from the encoded PEM key...", new Object[0]);
        try {
            return KeyUtils.decodePublicKey(content);
        }
        catch (Exception e) {
            LOGGER.debug((Object)"Failed to create a key from the encoded PEM key", (Throwable)e);
            return null;
        }
    }

    static PublicKey tryAsPEMCertificate(String content) {
        LOGGER.debugf("Trying to create a key from the encoded PEM certificate...", new Object[0]);
        try {
            return KeyUtils.decodeCertificate(content);
        }
        catch (Exception e) {
            LOGGER.debug((Object)"Failed to to create a key from the encoded PEM certificate", (Throwable)e);
            return null;
        }
    }

    static List<JsonWebKey> loadJsonWebKeys(String content) {
        LOGGER.debugf("Trying to load the local JWK(S)...", new Object[0]);
        JsonObject jwks = null;
        try (JsonReader reader = Json.createReader((Reader)new StringReader(content));){
            jwks = reader.readObject();
        }
        catch (Exception ex) {
            LOGGER.debug((Object)"Failed to load the JWK(S)");
            return null;
        }
        List<JsonWebKey> localKeys = null;
        JsonArray keys = jwks.getJsonArray("keys");
        try {
            if (keys != null) {
                localKeys = new ArrayList<JsonWebKey>(keys.size());
                for (int i = 0; i < keys.size(); ++i) {
                    localKeys.add(KeyLocationResolver.createJsonWebKey(keys.getJsonObject(i)));
                }
            } else {
                localKeys = Collections.singletonList(KeyLocationResolver.createJsonWebKey(jwks));
            }
        }
        catch (Exception ex) {
            LOGGER.debug((Object)"Failed to parse the JWK JSON representation");
            return null;
        }
        return localKeys;
    }

    static PublicKey getKeyFromJsonWebKeys(String kid, List<JsonWebKey> keys) {
        if (kid != null) {
            for (JsonWebKey currentJwk : keys) {
                if (!kid.equals(currentJwk.getKeyId())) continue;
                return ((PublicJsonWebKey)PublicJsonWebKey.class.cast(currentJwk)).getPublicKey();
            }
        }
        if (keys.size() == 1 && (kid == null || keys.get(0).getKeyId() == null)) {
            return ((PublicJsonWebKey)PublicJsonWebKey.class.cast(keys.get(0))).getPublicKey();
        }
        return null;
    }

    static JsonWebKey createJsonWebKey(JsonObject jsonObject) throws Exception {
        return JsonWebKey.Factory.newJwk((Map)JsonUtil.parseJson((String)jsonObject.toString()));
    }

    static InputStream getAsFileSystemResource(String publicKeyLocation) throws IOException {
        try {
            return new FileInputStream(publicKeyLocation);
        }
        catch (FileNotFoundException e) {
            return null;
        }
    }

    static InputStream getAsClasspathResource(String location) {
        return Thread.currentThread().getContextClassLoader().getResourceAsStream(location);
    }

    static class UrlStreamResolver {
        UrlStreamResolver() {
        }

        public InputStream resolve(String keyLocation) throws IOException {
            return new URL(keyLocation).openStream();
        }
    }
}

