package io.smallrye.stork.loadbalancer.leastresponsetime;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import io.smallrye.stork.api.config.LoadBalancerConfig;

/**
 *  Configuration for the {@code LeastResponseTimeLoadBalancerProvider} LoadBalancer.
 */
 public class LeastResponseTimeConfiguration implements io.smallrye.stork.api.config.LoadBalancerConfig{
   private final Map<String, String> parameters;

   /**
    * Creates a new LeastResponseTimeConfiguration
    *
    * @param params the parameters, must not be {@code null}
    */
   public LeastResponseTimeConfiguration(Map<String, String> params) {
      parameters = Collections.unmodifiableMap(params);
   }

   /**
    * Creates a new LeastResponseTimeConfiguration
    */
   public LeastResponseTimeConfiguration() {
      parameters = Collections.emptyMap();
   }


  /**
   * @return the type
   */
   @Override
   public String type() {
      return "least-response-time";
   }


   /**
    * @return the parameters
    */
   @Override
   public Map<String, String> parameters() {
      return parameters;
   }

   private LeastResponseTimeConfiguration extend(String key, String value) {
      Map<String, String> copy = new HashMap<>(parameters);
      copy.put(key, value);
      return new LeastResponseTimeConfiguration(copy);
   }

   /**
    * How much *score* should decline in time, see Score calculation in the docs for details. By default: 0.9
    *
    * @return the configured declining-factor, {@code 0.9} if not set
    */
   public String getDecliningFactor() {
      String result = parameters.get("declining-factor");
      return result == null ? "0.9" : result;
   }

   /**
    * Set the 'declining-factor' attribute. Default is 0.9.
    * 
    * @param value the value for declining-factor
    * @return the current LeastResponseTimeConfiguration to chain calls
    */
   public LeastResponseTimeConfiguration withDecliningFactor(String value) {
      return extend("declining-factor", value);
   }

   /**
    * This load balancer treats an erroneous response as a response after this time. By default: 60s
    *
    * @return the configured error-penalty, {@code 60s} if not set
    */
   public String getErrorPenalty() {
      String result = parameters.get("error-penalty");
      return result == null ? "60s" : result;
   }

   /**
    * Set the 'error-penalty' attribute. Default is 60s.
    * 
    * @param value the value for error-penalty
    * @return the current LeastResponseTimeConfiguration to chain calls
    */
   public LeastResponseTimeConfiguration withErrorPenalty(String value) {
      return extend("error-penalty", value);
   }

   /**
    * Whether the load balancer should use a SecureRandom instead of a Random (default). Check [this page](https://stackoverflow.com/questions/11051205/difference-between-java-util-random-and-java-security-securerandom) to understand the difference By default: false
    *
    * @return the configured use-secure-random, {@code false} if not set
    */
   public String getUseSecureRandom() {
      String result = parameters.get("use-secure-random");
      return result == null ? "false" : result;
   }

   /**
    * Set the 'use-secure-random' attribute. Default is false.
    * 
    * @param value the value for use-secure-random
    * @return the current LeastResponseTimeConfiguration to chain calls
    */
   public LeastResponseTimeConfiguration withUseSecureRandom(String value) {
      return extend("use-secure-random", value);
   }
}
