package io.smallrye.stork.loadbalancer.random;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import io.smallrye.stork.api.config.ConfigWithType;

/**
 *  Configuration for the {@code RandomLoadBalancerProvider} LoadBalancer.
 */
 public class RandomConfiguration implements io.smallrye.stork.api.config.ConfigWithType{
   private final Map<String, String> parameters;

   /**
    * Creates a new RandomConfiguration
    *
    * @param params the parameters, must not be {@code null}
    */
   public RandomConfiguration(Map<String, String> params) {
      parameters = Collections.unmodifiableMap(params);
   }

   /**
    * Creates a new RandomConfiguration
    */
   public RandomConfiguration() {
      parameters = Collections.emptyMap();
   }


  /**
   * @return the type
   */
   @Override
   public String type() {
      return "random";
   }


   /**
    * @return the parameters
    */
   @Override
   public Map<String, String> parameters() {
      return parameters;
   }

   private RandomConfiguration extend(String key, String value) {
      Map<String, String> copy = new HashMap<>(parameters);
      copy.put(key, value);
      return new RandomConfiguration(copy);
   }

   /**
    * Whether the load balancer should use a SecureRandom instead of a Random (default). Check [this page](https://stackoverflow.com/questions/11051205/difference-between-java-util-random-and-java-security-securerandom) to understand the difference By default: false
    *
    * @return the configured use-secure-random, {@code false} if not set
    */
   public String getUseSecureRandom() {
      String result = parameters.get("use-secure-random");
      return result == null ? "false" : result;
   }

   /**
    * Set the 'use-secure-random' attribute. Default is false.
    * 
    * @param value the value for use-secure-random
    * @return the current RandomConfiguration to chain calls
    */
   public RandomConfiguration withUseSecureRandom(String value) {
      return extend("use-secure-random", value);
   }
}
