package io.smallrye.stork.servicediscovery.consul;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import io.smallrye.stork.api.config.ConfigWithType;

/**
 *  Configuration for the {@code ConsulServiceDiscoveryProvider} ServiceDiscovery.
 */
 public class ConsulConfiguration implements io.smallrye.stork.api.config.ConfigWithType{
   private final Map<String, String> parameters;

   /**
    * Creates a new ConsulConfiguration
    *
    * @param params the parameters, must not be {@code null}
    */
   public ConsulConfiguration(Map<String, String> params) {
      parameters = Collections.unmodifiableMap(params);
   }

   /**
    * Creates a new ConsulConfiguration
    */
   public ConsulConfiguration() {
      parameters = Collections.emptyMap();
   }


  /**
   * @return the type
   */
   @Override
   public String type() {
      return "consul";
   }


   /**
    * @return the parameters
    */
   @Override
   public Map<String, String> parameters() {
      return parameters;
   }

   private ConsulConfiguration extend(String key, String value) {
      Map<String, String> copy = new HashMap<>(parameters);
      copy.put(key, value);
      return new ConsulConfiguration(copy);
   }

   /**
    * The Consul host. By default: localhost
    *
    * @return the configured consul-host, {@code localhost} if not set
    */
   public String getConsulHost() {
      String result = parameters.get("consul-host");
      return result == null ? "localhost" : result;
   }

   /**
    * Set the 'consul-host' attribute. Default is localhost.
    * 
    * @param value the value for consul-host
    * @return the current ConsulConfiguration to chain calls
    */
   public ConsulConfiguration withConsulHost(String value) {
      return extend("consul-host", value);
   }

   /**
    * The Consul port. By default: 8500
    *
    * @return the configured consul-port, {@code 8500} if not set
    */
   public String getConsulPort() {
      String result = parameters.get("consul-port");
      return result == null ? "8500" : result;
   }

   /**
    * Set the 'consul-port' attribute. Default is 8500.
    * 
    * @param value the value for consul-port
    * @return the current ConsulConfiguration to chain calls
    */
   public ConsulConfiguration withConsulPort(String value) {
      return extend("consul-port", value);
   }

   /**
    * Whether to use health check. By default: true
    *
    * @return the configured use-health-checks, {@code true} if not set
    */
   public String getUseHealthChecks() {
      String result = parameters.get("use-health-checks");
      return result == null ? "true" : result;
   }

   /**
    * Set the 'use-health-checks' attribute. Default is true.
    * 
    * @param value the value for use-health-checks
    * @return the current ConsulConfiguration to chain calls
    */
   public ConsulConfiguration withUseHealthChecks(String value) {
      return extend("use-health-checks", value);
   }

   /**
    * The application name; if not defined Stork service name will be used.
    *
    * @return the configured application, @{code null} if not set
    */
   public String getApplication() {
      return parameters.get("application");
   }

   /**
    * Set the 'application' attribute.
    * 
    * @param value the value for application
    * @return the current ConsulConfiguration to chain calls
    */
   public ConsulConfiguration withApplication(String value) {
      return extend("application", value);
   }

   /**
    * Service discovery cache refresh period. By default: 5M
    *
    * @return the configured refresh-period, {@code 5M} if not set
    */
   public String getRefreshPeriod() {
      String result = parameters.get("refresh-period");
      return result == null ? "5M" : result;
   }

   /**
    * Set the 'refresh-period' attribute. Default is 5M.
    * 
    * @param value the value for refresh-period
    * @return the current ConsulConfiguration to chain calls
    */
   public ConsulConfiguration withRefreshPeriod(String value) {
      return extend("refresh-period", value);
   }

   /**
    * whether the connection with the service should be encrypted with TLS.
    *
    * @return the configured secure, @{code null} if not set
    */
   public String getSecure() {
      return parameters.get("secure");
   }

   /**
    * Set the 'secure' attribute.
    * 
    * @param value the value for secure
    * @return the current ConsulConfiguration to chain calls
    */
   public ConsulConfiguration withSecure(String value) {
      return extend("secure", value);
   }
}
