package io.smallrye.stork.servicediscovery.dns;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import io.smallrye.stork.api.config.ConfigWithType;

/**
 *  Configuration for the {@code DnsServiceDiscoveryProvider} ServiceDiscovery.
 */
 public class DnsConfiguration implements io.smallrye.stork.api.config.ConfigWithType{
   private final Map<String, String> parameters;

   /**
    * Creates a new DnsConfiguration
    *
    * @param params the parameters, must not be {@code null}
    */
   public DnsConfiguration(Map<String, String> params) {
      parameters = Collections.unmodifiableMap(params);
   }

   /**
    * Creates a new DnsConfiguration
    */
   public DnsConfiguration() {
      parameters = Collections.emptyMap();
   }


  /**
   * @return the type
   */
   @Override
   public String type() {
      return "dns";
   }


   /**
    * @return the parameters
    */
   @Override
   public Map<String, String> parameters() {
      return parameters;
   }

   private DnsConfiguration extend(String key, String value) {
      Map<String, String> copy = new HashMap<>(parameters);
      copy.put(key, value);
      return new DnsConfiguration(copy);
   }

   /**
    * Comma separated list of dns servers. Servers can either be in the `server:port` or just `server` form. Use `none` to use the system resolver. By default: none
    *
    * @return the configured dns-servers, {@code none} if not set
    */
   public String getDnsServers() {
      String result = parameters.get("dns-servers");
      return result == null ? "none" : result;
   }

   /**
    * Set the 'dns-servers' attribute. Default is none.
    * 
    * @param value the value for dns-servers
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withDnsServers(String value) {
      return extend("dns-servers", value);
   }

   /**
    * The hostname to look up; if not defined Stork service name will be used.
    *
    * @return the configured hostname, @{code null} if not set
    */
   public String getHostname() {
      return parameters.get("hostname");
   }

   /**
    * Set the 'hostname' attribute.
    * 
    * @param value the value for hostname
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withHostname(String value) {
      return extend("hostname", value);
   }

   /**
    * Type of the DNS record. A, AAAA and SRV records are supported By default: SRV
    *
    * @return the configured record-type, {@code SRV} if not set
    */
   public String getRecordType() {
      String result = parameters.get("record-type");
      return result == null ? "SRV" : result;
   }

   /**
    * Set the 'record-type' attribute. Default is SRV.
    * 
    * @param value the value for record-type
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withRecordType(String value) {
      return extend("record-type", value);
   }

   /**
    * Port of the service instances. Required if the record type is other than SRV.
    *
    * @return the configured port, @{code null} if not set
    */
   public String getPort() {
      return parameters.get("port");
   }

   /**
    * Set the 'port' attribute.
    * 
    * @param value the value for port
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withPort(String value) {
      return extend("port", value);
   }

   /**
    * Service discovery cache refresh period. By default: 5M
    *
    * @return the configured refresh-period, {@code 5M} if not set
    */
   public String getRefreshPeriod() {
      String result = parameters.get("refresh-period");
      return result == null ? "5M" : result;
   }

   /**
    * Set the 'refresh-period' attribute. Default is 5M.
    * 
    * @param value the value for refresh-period
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withRefreshPeriod(String value) {
      return extend("refresh-period", value);
   }

   /**
    * Whether the connection with the service should be encrypted with TLS.
    *
    * @return the configured secure, @{code null} if not set
    */
   public String getSecure() {
      return parameters.get("secure");
   }

   /**
    * Set the 'secure' attribute.
    * 
    * @param value the value for secure
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withSecure(String value) {
      return extend("secure", value);
   }

   /**
    * Whether DNS recursion is desired By default: true
    *
    * @return the configured recursion-desired, {@code true} if not set
    */
   public String getRecursionDesired() {
      String result = parameters.get("recursion-desired");
      return result == null ? "true" : result;
   }

   /**
    * Set the 'recursion-desired' attribute. Default is true.
    * 
    * @param value the value for recursion-desired
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withRecursionDesired(String value) {
      return extend("recursion-desired", value);
   }

   /**
    * Timeout for DNS queries By default: 5s
    *
    * @return the configured dns-timeout, {@code 5s} if not set
    */
   public String getDnsTimeout() {
      String result = parameters.get("dns-timeout");
      return result == null ? "5s" : result;
   }

   /**
    * Set the 'dns-timeout' attribute. Default is 5s.
    * 
    * @param value the value for dns-timeout
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withDnsTimeout(String value) {
      return extend("dns-timeout", value);
   }

   /**
    * Whether an error in retrieving service instances from one of the DNS servers should cause a failure of the discovery attempt. By default: false
    *
    * @return the configured fail-on-error, {@code false} if not set
    */
   public String getFailOnError() {
      String result = parameters.get("fail-on-error");
      return result == null ? "false" : result;
   }

   /**
    * Set the 'fail-on-error' attribute. Default is false.
    * 
    * @param value the value for fail-on-error
    * @return the current DnsConfiguration to chain calls
    */
   public DnsConfiguration withFailOnError(String value) {
      return extend("fail-on-error", value);
   }
}
