package io.smallrye.stork.servicediscovery.kubernetes;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import io.smallrye.stork.api.config.ConfigWithType;

/**
 *  Configuration for the {@code KubernetesServiceDiscoveryProvider} ServiceDiscovery.
 */
 public class KubernetesConfiguration implements io.smallrye.stork.api.config.ConfigWithType{
   private final Map<String, String> parameters;

   /**
    * Creates a new KubernetesConfiguration
    *
    * @param params the parameters, must not be {@code null}
    */
   public KubernetesConfiguration(Map<String, String> params) {
      parameters = Collections.unmodifiableMap(params);
   }

   /**
    * Creates a new KubernetesConfiguration
    */
   public KubernetesConfiguration() {
      parameters = Collections.emptyMap();
   }


  /**
   * @return the type
   */
   @Override
   public String type() {
      return "kubernetes";
   }


   /**
    * @return the parameters
    */
   @Override
   public Map<String, String> parameters() {
      return parameters;
   }

   private KubernetesConfiguration extend(String key, String value) {
      Map<String, String> copy = new HashMap<>(parameters);
      copy.put(key, value);
      return new KubernetesConfiguration(copy);
   }

   /**
    * The Kubernetes API host.
    *
    * @return the configured k8s-host, @{code null} if not set
    */
   public String getK8sHost() {
      return parameters.get("k8s-host");
   }

   /**
    * Set the 'k8s-host' attribute.
    * 
    * @param value the value for k8s-host
    * @return the current KubernetesConfiguration to chain calls
    */
   public KubernetesConfiguration withK8sHost(String value) {
      return extend("k8s-host", value);
   }

   /**
    * The namespace of the service. Use all to discover all namespaces.
    *
    * @return the configured k8s-namespace, @{code null} if not set
    */
   public String getK8sNamespace() {
      return parameters.get("k8s-namespace");
   }

   /**
    * Set the 'k8s-namespace' attribute.
    * 
    * @param value the value for k8s-namespace
    * @return the current KubernetesConfiguration to chain calls
    */
   public KubernetesConfiguration withK8sNamespace(String value) {
      return extend("k8s-namespace", value);
   }

   /**
    * The Kubernetes application Id; if not defined Stork service name will be used.
    *
    * @return the configured application, @{code null} if not set
    */
   public String getApplication() {
      return parameters.get("application");
   }

   /**
    * Set the 'application' attribute.
    * 
    * @param value the value for application
    * @return the current KubernetesConfiguration to chain calls
    */
   public KubernetesConfiguration withApplication(String value) {
      return extend("application", value);
   }

   /**
    * Service discovery cache refresh period. By default: 5M
    *
    * @return the configured refresh-period, {@code 5M} if not set
    */
   public String getRefreshPeriod() {
      String result = parameters.get("refresh-period");
      return result == null ? "5M" : result;
   }

   /**
    * Set the 'refresh-period' attribute. Default is 5M.
    * 
    * @param value the value for refresh-period
    * @return the current KubernetesConfiguration to chain calls
    */
   public KubernetesConfiguration withRefreshPeriod(String value) {
      return extend("refresh-period", value);
   }

   /**
    * Whether the connection with the service should be encrypted with TLS.
    *
    * @return the configured secure, @{code null} if not set
    */
   public String getSecure() {
      return parameters.get("secure");
   }

   /**
    * Set the 'secure' attribute.
    * 
    * @param value the value for secure
    * @return the current KubernetesConfiguration to chain calls
    */
   public KubernetesConfiguration withSecure(String value) {
      return extend("secure", value);
   }
}
