package io.smallrye.stork.serviceregistration.consul;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import io.smallrye.stork.api.config.ConfigWithType;

/**
 *  Configuration for the {@code ConsulServiceRegistrarProvider} ServiceRegistrar.
 */
 public class ConsulRegistrarConfiguration implements io.smallrye.stork.api.config.ConfigWithType{
   private final Map<String, String> parameters;

   /**
    * Creates a new ConsulRegistrarConfiguration
    *
    * @param params the parameters, must not be {@code null}
    */
   public ConsulRegistrarConfiguration(Map<String, String> params) {
      parameters = Collections.unmodifiableMap(params);
   }

   /**
    * Creates a new ConsulRegistrarConfiguration
    */
   public ConsulRegistrarConfiguration() {
      parameters = Collections.emptyMap();
   }


  /**
   * @return the type
   */
   @Override
   public String type() {
      return "consul";
   }


   /**
    * @return the parameters
    */
   @Override
   public Map<String, String> parameters() {
      return parameters;
   }

   private ConsulRegistrarConfiguration extend(String key, String value) {
      Map<String, String> copy = new HashMap<>(parameters);
      copy.put(key, value);
      return new ConsulRegistrarConfiguration(copy);
   }

   /**
    * The Consul host. By default: localhost
    *
    * @return the configured consul-host, {@code localhost} if not set
    */
   public String getConsulHost() {
      String result = parameters.get("consul-host");
      return result == null ? "localhost" : result;
   }

   /**
    * Set the 'consul-host' attribute. Default is localhost.
    * 
    * @param value the value for consul-host
    * @return the current ConsulRegistrarConfiguration to chain calls
    */
   public ConsulRegistrarConfiguration withConsulHost(String value) {
      return extend("consul-host", value);
   }

   /**
    * The Consul port. By default: 8500
    *
    * @return the configured consul-port, {@code 8500} if not set
    */
   public String getConsulPort() {
      String result = parameters.get("consul-port");
      return result == null ? "8500" : result;
   }

   /**
    * Set the 'consul-port' attribute. Default is 8500.
    * 
    * @param value the value for consul-port
    * @return the current ConsulRegistrarConfiguration to chain calls
    */
   public ConsulRegistrarConfiguration withConsulPort(String value) {
      return extend("consul-port", value);
   }

   /**
    * The liveness http address. By default: 
    *
    * @return the configured health-check-url, {@code } if not set
    */
   public String getHealthCheckUrl() {
      String result = parameters.get("health-check-url");
      return result == null ? "" : result;
   }

   /**
    * Set the 'health-check-url' attribute. Default is .
    * 
    * @param value the value for health-check-url
    * @return the current ConsulRegistrarConfiguration to chain calls
    */
   public ConsulRegistrarConfiguration withHealthCheckUrl(String value) {
      return extend("health-check-url", value);
   }

   /**
    * How often Consul performs the health check By default: 30s
    *
    * @return the configured health-check-interval, {@code 30s} if not set
    */
   public String getHealthCheckInterval() {
      String result = parameters.get("health-check-interval");
      return result == null ? "30s" : result;
   }

   /**
    * Set the 'health-check-interval' attribute. Default is 30s.
    * 
    * @param value the value for health-check-interval
    * @return the current ConsulRegistrarConfiguration to chain calls
    */
   public ConsulRegistrarConfiguration withHealthCheckInterval(String value) {
      return extend("health-check-interval", value);
   }

   /**
    * How long after the check is in critical status Consul will remove the service from the catalogue. By default: 1m
    *
    * @return the configured health-check-deregister-after, {@code 1m} if not set
    */
   public String getHealthCheckDeregisterAfter() {
      String result = parameters.get("health-check-deregister-after");
      return result == null ? "1m" : result;
   }

   /**
    * Set the 'health-check-deregister-after' attribute. Default is 1m.
    * 
    * @param value the value for health-check-deregister-after
    * @return the current ConsulRegistrarConfiguration to chain calls
    */
   public ConsulRegistrarConfiguration withHealthCheckDeregisterAfter(String value) {
      return extend("health-check-deregister-after", value);
   }
}
