/*
 * Copyright (c) 2010-2015 Pivotal Software, Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You
 * may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License. See accompanying
 * LICENSE file.
 */
package container.app.lang.support;

import container.app.lang.EventSource;
import container.app.util.Assert;
import container.app.util.ClassUtils;
import container.app.util.SystemUtils;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.*;

public class EventSourceSupport<L extends EventListener> implements EventSource<L>, Iterable<L> {

  private final List<L> eventListeners = new ArrayList<L>(19);

  private final Object eventSource;

  public EventSourceSupport(final Object source) {
    Assert.notNull(source, "The source object for the events generated cannot be null!");
    this.eventSource = source;
  }

  public <T> T getSource() {
    return (T) this.eventSource;
  }

  public boolean addEventListener(final L eventListener) {
    Assert.notNull(eventListener, "The EventListener to add to the event source ({0}) cannot be null!",
        ClassUtils.getClassName(getSource()));
    return eventListeners.add(eventListener);
  }

  public void fireEvent(final String eventHandlerMethodName, final EventObject event) {
    Assert.hasValue(eventHandlerMethodName, "The name of the event handler method must be specified!");
    Assert.notNull(event, MessageFormat.format("The event to fire from event source ({0}) should not be null!",
        ClassUtils.getClassName(getSource())));

    L currentEventListener = null;

    if (!eventListeners.isEmpty()) {
      try {
        currentEventListener = eventListeners.get(0);

        final Method eventHandlerMethod = currentEventListener.getClass().getMethod(
            eventHandlerMethodName, new Class[] { event.getClass() });

        for (final L eventListener : eventListeners) {
          currentEventListener = eventListener;
          eventHandlerMethod.invoke(eventListener, new Object[] { event });
        }
      }
      catch (NoSuchMethodException e) {
        // TODO should all EventListeners necessarily conform to the same interface, meaning should all
        // EventListeners be interested in the same set of events, hmmm?
        throw new IllegalArgumentException(MessageFormat.format(
            "No event handler method with signature {0}({1}) exists on EventListener class ({2})!",
            eventHandlerMethodName, ClassUtils.getClassName(event), ClassUtils.getClassName(currentEventListener)), e);
      }
      catch (InvocationTargetException e) {
        // Capture any Exception thrown from the invocation of the event handler method
        // of a single EventListener to allow other EventListeners to be notified.
        SystemUtils.printToStandardError("An error occurred while invoking event handler method ({0}) of EventListener ({1}): {2}",
            eventHandlerMethodName, ClassUtils.getClassName(currentEventListener), e.getMessage());
      }
      catch (Exception e) {
        throw new RuntimeException(MessageFormat.format(
            "Failed to notify listeners of event ({0}) generated by event source ({1})!",
            event, ClassUtils.getClassName(getSource())), e);
      }
    }
  }

  public Iterator<L> iterator() {
    return Collections.unmodifiableList(eventListeners).iterator();
  }

  public boolean removeEventListener(final L eventListener) {
    return eventListeners.remove(eventListener);
  }

}
