/*
 * Copyright (c) 2010-2015 Pivotal Software, Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You
 * may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License. See accompanying
 * LICENSE file.
 */

package hydra;

import com.gemstone.gemfire.internal.*;
import com.gemstone.gemfire.distributed.internal.*;

import java.io.*;
import java.rmi.RemoteException;
import java.util.*;

/**
 *
 *  Encodes information needed to manage a GemFire system.
 *
 */
public class GemFireDescription extends AbstractDescription
implements Serializable {

  private transient String sysDirName; // generated by client

  private String  name;         // name
  private HostDescription hd;   // host description used to create system

  private boolean stopSystemAfterTest;

  private String  distributedSystem;
  private List<String>  remoteDistributedSystems;
  private Integer distributedSystemId;

  private String  systemDir;    // system directory

  private String  id;           // a short string describing the system

  private String  logLevel;
  private Integer logFileSizeLimit;
  private Integer logDiskSpaceLimit;
  private String  statisticArchiveFile;
  private Boolean statisticSamplingEnabled;
  private Integer statisticSampleRate;
  private Boolean enableTimeStatistics;
  private Integer archiveFileSizeLimit;
  private Integer archiveDiskSpaceLimit;

  private Integer ackWaitThreshold;

  private Boolean conserveSockets;
  private Boolean deltaPropagation;
  private Long    socketLeaseTime;
  private Long    socketBufferSize;

  private Integer asyncDistributionTimeout;
  private Integer asyncQueueTimeout;
  private Integer asyncMaxQueueSize;

  private String  roles;
  private Integer maxWaitTimeForReconnect;
  private Integer maxNumReconnectTries;

  private Boolean useLocator;
  private Boolean isPeerLocator;
  private Boolean isServerLocator;
  private Boolean enableMcast;
  private String  mcastAddress;
  private Integer mcastPort;
  private Integer mcastTtl;
  private Integer mcastSendBufferSize;
  private Integer mcastRecvBufferSize;
  private Integer mcastFlowControlByteAllowance;
  private Float   mcastFlowControlRechargeThreshold;
  private Integer mcastFlowControlRechargeBlockMs;
  private Integer udpFragmentSize;
  private Integer udpSendBufferSize;
  private Integer udpRecvBufferSize;
  private Integer tcpPort;
  private Boolean disableTcp;
  private String  membershipPortRange;
  private Integer memberTimeout;

  private Boolean enableDurableClient;
  private Integer durableClientTimeout;
  private String  conflateEvents;

  private Boolean enforceUniqueHost;
  private String redundancyZone;

  private Integer ackSevereAlertThreshold;
  private Boolean enableNetworkPartitionDetection;
  private Boolean disableAutoReconnect;

  /** Extra VM args for master-managed locators */
  private String extraLocatorVMArgs;

  /** Extra classpath for master-managed locators */
  private String locatorClassPath;

  private Boolean jmxManager;
  private String jmxManagerAccessFile;
  private Boolean jmxManagerHttpPort;
  private String jmxManagerPasswordFile;
  private Boolean jmxManagerPort;
  private Boolean jmxManagerSSL;
  private Boolean jmxManagerStart;

  private String securityName;
  private SecurityDescription securityDescription; // from securityName
  private String sslName;
  private SSLDescription sslDescription; // from sslName

  private String offHeapMemorySize;
  private Boolean lockMemory;

  //////////////////////////////////////////////////////////////////////////////
  ////    CONSTRUCTORS                                                      ////
  //////////////////////////////////////////////////////////////////////////////

  public GemFireDescription() {
  }

  //////////////////////////////////////////////////////////////////////////////
  ////    ACCESSORS                                                         ////
  //////////////////////////////////////////////////////////////////////////////

  public String getName() {
    return this.name;
  }
  private void setName( String name ) {
    this.name = name;
  }
  public HostDescription getHostDescription() {
    return this.hd;
  }
  private void setHostDescription( HostDescription hd ) {
    this.hd = hd;
  }
  public boolean stopSystemAfterTest() {
    return this.stopSystemAfterTest;
  }
  private void setStopSystemAfterTest( boolean flag ) {
    this.stopSystemAfterTest = flag;
  }
  public String getDistributedSystem() {
    return this.distributedSystem;
  }
  private void setDistributedSystem( String sys ) {
    this.distributedSystem = sys;
  }
  public List<String> getRemoteDistributedSystems() {
    return this.remoteDistributedSystems;
  }
  private void setRemoteDistributedSystems(List<String> systems) {
    this.remoteDistributedSystems = systems;
  }
  public Integer getDistributedSystemId() {
    return this.distributedSystemId;
  }
  private void setDistributedSystemId(Integer i) {
    this.distributedSystemId = i;
  }
  public String getSystemDirectoryStr() {
    return this.systemDir;
  }
  public File getSystemDirectory() {
    return new File( this.systemDir );
  }
  private void setSystemDirectory( String dir ) {
    this.systemDir = dir;
  }
  public Integer getAckWaitThreshold() {
    return this.ackWaitThreshold;
  }
  private void setAckWaitThreshold( Integer threshold ) {
    this.ackWaitThreshold = threshold;
  }
  public String getLogLevel() {
    return this.logLevel;
  }
  private void setLogLevel( String level ) {
    this.logLevel = level;
  }
  public Integer getLogFileSizeLimit() {
    return this.logFileSizeLimit;
  }
  private void setLogFileSizeLimit( Integer limit ) {
    this.logFileSizeLimit = limit;
  }
  public Integer getLogDiskSpaceLimit() {
    return this.logDiskSpaceLimit;
  }
  private void setLogDiskSpaceLimit( Integer limit ) {
    this.logDiskSpaceLimit = limit;
  }
  public String getStatisticArchiveFile() {
    return this.statisticArchiveFile;
  }
  private void setStatisticArchiveFile( String file ) {
    this.statisticArchiveFile = file;
  }
  public Boolean getStatisticSamplingEnabled() {
    return this.statisticSamplingEnabled;
  }
  private void setStatisticSamplingEnabled( Boolean flag ) {
    this.statisticSamplingEnabled = flag;
  }
  public Integer getStatisticSampleRate() {
    return this.statisticSampleRate;
  }
  private void setStatisticSampleRate( Integer rate ) {
    this.statisticSampleRate = rate;
  }
  public Boolean getEnableTimeStatistics() {
    return this.enableTimeStatistics;
  }
  private void setEnableTimeStatistics( Boolean flag ) {
    this.enableTimeStatistics = flag;
  }
  public Integer getArchiveFileSizeLimit() {
    return this.archiveFileSizeLimit;
  }
  private void setArchiveFileSizeLimit( Integer limit ) {
    this.archiveFileSizeLimit = limit;
  }
  public Integer getArchiveDiskSpaceLimit() {
    return this.archiveDiskSpaceLimit;
  }
  private void setArchiveDiskSpaceLimit( Integer limit ) {
    this.archiveDiskSpaceLimit = limit;
  }
  public Boolean getUseLocator() {
    return this.useLocator;
  }
  private void setUseLocator( Boolean flag ) {
    this.useLocator = flag;
  }
  public Boolean isPeerLocator() {
    return this.isPeerLocator;
  }
  private void setPeerLocator(Boolean flag) {
    this.isPeerLocator = flag;
  }
  public Boolean isServerLocator() {
    return this.isServerLocator;
  }
  private void setServerLocator(Boolean flag) {
    this.isServerLocator = flag;
  }

  public Boolean getEnableMcast() {
    return this.enableMcast;
  }
  private void setEnableMcast( Boolean flag ) {
    this.enableMcast = flag;
  }
  public String getMcastAddress() {
    return this.mcastAddress;
  }

  private void setMcastAddress( String addressStr ) {
    this.mcastAddress = addressStr;
  }  
  public Integer getMcastPort() {
    return this.mcastPort;
  }
  protected void setMcastPort( Integer port ) {
    this.mcastPort = port;
  }  
  public Integer getMcastTtl() {
    return this.mcastTtl;
  }
  private void setMcastTtl( Integer Ttl ) {
    this.mcastTtl = Ttl;
  }
  public Integer getMcastSendBufferSize() {
    return this.mcastSendBufferSize;
  }
  private void setMcastSendBufferSize( Integer size ) {
    this.mcastSendBufferSize = size;
  }
  public Integer getMcastRecvBufferSize() {
    return this.mcastRecvBufferSize;
  }
  private void setMcastRecvBufferSize( Integer size ) {
    this.mcastRecvBufferSize = size;
  }
  public Integer getMcastFlowControlByteAllowance() {
    return this.mcastFlowControlByteAllowance;
  }
  private void setMcastFlowControlByteAllowance(Integer allowance) {
    this.mcastFlowControlByteAllowance = allowance;
  }
  public Float getMcastFlowControlRechargeThreshold() {
    return this.mcastFlowControlRechargeThreshold;
  }
  private void setMcastFlowControlRechargeThreshold(Float threshold) {
    this.mcastFlowControlRechargeThreshold = threshold;
  }
  public Integer getMcastFlowControlRechargeBlockMs() {
    return this.mcastFlowControlRechargeBlockMs;
  }
  private void setMcastFlowControlRechargeBlockMs(Integer ms) {
    this.mcastFlowControlRechargeBlockMs = ms;
  }

  public Integer getUdpFragmentSize() {
    return this.udpFragmentSize;
  }
  private void setUdpFragmentSize( Integer size ) {
    this.udpFragmentSize = size;
  }
  public Integer getUdpSendBufferSize() {
    return this.udpSendBufferSize;
  }
  private void setUdpSendBufferSize( Integer size ) {
    this.udpSendBufferSize = size;
  }
  public Integer getUdpRecvBufferSize() {
    return this.udpRecvBufferSize;
  }
  private void setUdpRecvBufferSize( Integer size ) {
    this.udpRecvBufferSize = size;
  }
  public Integer getTcpPort() {
    return this.tcpPort;
  }
  protected void setTcpPort( Integer port ) {
    this.tcpPort = port;
  }  
  public Boolean getDisableTcp() {
    return this.disableTcp;
  }
  private void setDisableTcp( Boolean flag ) {
    this.disableTcp = flag;
  }
  public String getMembershipPortRange() {
    return this.membershipPortRange;
  }
  protected void setMembershipPortRange(String range) {
    this.membershipPortRange = range;
  }  
  public Integer getMemberTimeout() {
    return this.memberTimeout;
  }
  private void setMemberTimeout( Integer timeout ) {
    this.memberTimeout = timeout;
  }
  public String getLocatorClassPath() {
    return this.locatorClassPath;
  }
  private void setLocatorClassPath( String path ) {
    this.locatorClassPath = path;
  }
  public String getExtraLocatorVMArgs() {
    return this.extraLocatorVMArgs;
  }
  private void setExtraLocatorVMArgs( String args ) {
    this.extraLocatorVMArgs = args;
  }

  //////////////////////////////
  // DurableClient properties //
  //////////////////////////////
  public Boolean getEnableDurableClient() {
    return this.enableDurableClient;
  }
  private void setEnableDurableClient( Boolean flag ) {
    this.enableDurableClient = flag;
  }
  public Integer getDurableClientTimeout() {
    return this.durableClientTimeout;
  }
  private void setDurableClientTimeout( Integer timeout ) {
    this.durableClientTimeout = timeout;
  }

  ///////////////////////////
  // SplitBrain properties //
  ///////////////////////////

  public Integer getAckSevereAlertThreshold() {
    return this.ackSevereAlertThreshold;
  }
  private void setAckSevereAlertThreshold(Integer i) {
    this.ackSevereAlertThreshold = i;
  }
  public Boolean getEnableNetworkPartitionDetection() {
    return this.enableNetworkPartitionDetection;
  }
  private void setEnableNetworkPartitionDetection(Boolean bool) {
    this.enableNetworkPartitionDetection = bool;
  }
  private void setDisableAutoReconnect(Boolean  bool) {
    this.disableAutoReconnect = bool;
  }
  public Boolean getDisableAutoReconnect() {
    return this.disableAutoReconnect;
  }

  /**
   * Returns the role names for this GemFire system.
   *
   * @since 5.0
   */
  public String getRoles() {
    return this.roles;
  }
  private void setRoles( String roles) {
    this.roles = roles;
  }
  
  private  void setMaxWaitTimeForReconnect(Integer timeOut){
    this.maxWaitTimeForReconnect = timeOut;
  }
 
  private Integer getMaxWaitTimeForReconnect(){
     return this.maxWaitTimeForReconnect;
  }

  private void setMaxNumReconnectTries(Integer maxTime){
     this.maxNumReconnectTries = maxTime;
  }

  public Integer getMaxNumReconnectTries(){
      return this.maxNumReconnectTries;
  }

  //////////////////////////////////////
  // 4.1 UseCase12 performance enhancements
  //////////////////////////////////////
   public Boolean getConserveSockets() {
     return this.conserveSockets;
   }
   private void setConserveSockets( Boolean flag ) {
     this.conserveSockets = flag;
   }
   public Boolean getDeltaPropagation() {
     return this.deltaPropagation;
   }
   private void setDeltaPropagation( Boolean flag ) {
     this.deltaPropagation = flag;
   }
   public Long getSocketLeaseTime() {
     return this.socketLeaseTime;
   }
   private void setSocketLeaseTime( Long socketLeaseTime ) {
     this.socketLeaseTime = socketLeaseTime;
   }
   public Long getSocketBufferSize() {
     return this.socketBufferSize;
   }
   private void setSocketBufferSize( Long socketBufferSize ) {
     this.socketBufferSize = socketBufferSize;
   }

  //////////////////////////////////////////////////////////////////////////////
  ////    ASYNC MESSAGE PROPERTIES                                          ////
  //////////////////////////////////////////////////////////////////////////////

   public Integer getAsyncDistributionTimeout() {
     return this.asyncDistributionTimeout;
   } 

   private void setAsyncDistributionTimeout( Integer timeout ) {
     this.asyncDistributionTimeout = timeout;
   }

   public Integer getAsyncQueueTimeout() {
     return this.asyncQueueTimeout;
   }

   private void setAsyncQueueTimeout( Integer timeout ) {
     this.asyncQueueTimeout = timeout;
   }

   public Integer getAsyncMaxQueueSize() {
     return this.asyncMaxQueueSize;
   }

   private void setAsyncMaxQueueSize( Integer size ) {
     this.asyncMaxQueueSize = size;
   }

  /**
   * Returns the conflate events.
   */
  private String getConflateEvents() {
    return this.conflateEvents;
  }

  /**
   * Sets the conflate events.
   */
  private void setConflateEvents(String str) {
    this.conflateEvents = str;
  }

  /**
   * Returns the enforce unique host
   */
  public Boolean getEnforceUniqueHost() {
    return this.enforceUniqueHost;
  }

  /**
   * Sets the enforce unique host.
   */
  private void setEnforceUniqueHost(Boolean bool) {
    this.enforceUniqueHost = bool;
  }

  /**
   * Returns the redundancy zone
   */
  public String getRedundancyZone() {
    return this.redundancyZone;
  }

  /**
   * Sets the redundancy zone.
   */
  private void setRedundancyZone(String str) {
    this.redundancyZone = str;
  }

  /**
   * Returns the JMX manager.
   */
  public Boolean getJMXManager() {
    return this.jmxManager;
  }

  /**
   * Sets the JMX manager.
   */
  private void setJMXManager(Boolean bool) {
    this.jmxManager = bool;
  }

  /**
   * Returns the JMX manager access file.
   */
  public String getJMXManagerAccessFile() {
    return this.jmxManagerAccessFile;
  }

  /**
   * Sets the JMX manager access file.
   */
  private void setJMXManagerAccessFile(String str) {
    this.jmxManagerAccessFile = str;
  }

  /**
   * Returns the JMX manager http port.
   */
  protected Boolean getJMXManagerHttpPort() {
    return this.jmxManagerHttpPort;
  }

  /**
   * Returns the JMX manager password file.
   */
  public String getJMXManagerPasswordFile() {
    return this.jmxManagerPasswordFile;
  }

  /**
   * Sets the JMX manager password file.
   */
  private void setJMXManagerPasswordFile(String str) {
    this.jmxManagerPasswordFile = str;
  }

  /**
   * Sets the JMX manager http port.
   */
  private void setJMXManagerHttpPort(Boolean bool) {
    this.jmxManagerHttpPort= bool;
  }

  /**
   * Returns the JMX manager port.
   */
  protected Boolean getJMXManagerPort() {
    return this.jmxManagerPort;
  }

  /**
   * Sets the JMX manager port.
   */
  private void setJMXManagerPort(Boolean bool) {
    this.jmxManagerPort= bool;
  }

  /**
   * Returns the JMX manager SSL.
   */
  protected Boolean getJMXManagerSSL() {
    return this.jmxManagerSSL;
  }

  /**
   * Sets the JMX manager SSL.
   */
  private void setJMXManagerSSL(Boolean bool) {
    this.jmxManagerSSL= bool;
  }

  /**
   * Returns the JMX manager start.
   */
  public Boolean getJMXManagerStart() {
    return this.jmxManagerStart;
  }

  /**
   * Sets the JMX manager start.
   */
  private void setJMXManagerStart(Boolean bool) {
    this.jmxManagerStart= bool;
  }

  /**
   * Returns the security name.
   */
  public String getSecurityName() {
    return this.securityName;
  }

  /**
   * Sets the security name.
   */
  private void setSecurityName(String str) {
    this.securityName = str;
  }

  /**
   * Returns the security description.
   */
  public SecurityDescription getSecurityDescription() {
    return this.securityDescription;
  }

  /**
   * Sets the security description.
   */
  private void setSecurityDescription(SecurityDescription sd) {
    this.securityDescription = sd;
  }

  /**
   * Returns the SSL name.
   */
  public String getSSLName() {
    return this.sslName;
  }

  /**
   * Sets the SSL name.
   */
  private void setSSLName(String str) {
    this.sslName = str;
  }

  /**
   * Returns the SSL description.
   */
  public SSLDescription getSSLDescription() {
    return this.sslDescription;
  }

  /**
   * Sets the SSL description.
   */
  private void setSSLDescription(SSLDescription sd) {
    this.sslDescription = sd;
  }

  /**
   * Returns off-heap memory size.
   */
  public String getOffHeapMemorySize() {
    return this.offHeapMemorySize;
  }

  /**
   * Sets the off-heap memory size.
   */
  private void setOffHeapMemorySize(String aStr) {
    this.offHeapMemorySize= aStr;
  }

  public Boolean getLockMemory() {
    return this.lockMemory;
  }
  private void setLockMemory(Boolean bool) {
    this.lockMemory = bool;
  }

  //////////////////////////////////////////////////////////////////////////////
  ////    DISTRIBUTED SYSTEM PROPERTIES                                     ////
  //////////////////////////////////////////////////////////////////////////////

  /**
   * For internal hydra use only.  Hydra client-side method only.
   * <p>
   * Returns the name of the system directory for this incarnation of the
   * logical vm, and lazily creates and registers it.
   */
  public synchronized String getSysDirName() {
    if (this.sysDirName == null) {
      if (GemFireLocatorAgent.MasterManagedLocator) {
        this.sysDirName = GemFireLocatorAgent.getSysDirName() + "_"
                        + ProcessMgr.getProcessId();
      } else {
        this.sysDirName = this.getSystemDirectory() + "_"
                         + ProcessMgr.getProcessId();
      }
      if (!FileUtil.exists(this.sysDirName)) {
        FileUtil.mkdir(new File(this.sysDirName));
        HostDescription hd = this.getHostDescription();
        try {
          if (GemFireLocatorAgent.MasterManagedLocator) {
            HostDescription mhd = TestConfig.getInstance()
                .getMasterDescription().getVmDescription().getHostDescription();
            GemFireLocatorAgent.Master.recordDir(mhd, this.name, sysDirName);
          } else {
            RemoteTestModule.Master.recordDir(hd, this.name, sysDirName);
          }
        } catch (RemoteException e) {
          String s = "Unable to access master to record directory: "
                   + sysDirName;
          throw new HydraRuntimeException(s, e);
        }
      }
    }
    return this.sysDirName;
  }

  /**
   * For internal hydra use only.  Hydra client-side method only.
   * <p>
   * Returns the name of the log file in the system directory.
   */
  protected String getLogFileName() {
    return this.getSysDirName() + "/system.log";
  }

  /**
   * Generates and returns the distributed system properties for this
   * gemfire description, including security and SSL and JMX manager
   * properties, if any.
   * <p>
   * If SSL is enabled or a JMX manager is enabled with SSL, sets
   * SSL-related properties on the JVM as a side-effect.
   */
  public Properties getDistributedSystemProperties() {
    Properties p = new Properties();

    RemoteTestModule client = RemoteTestModule.getCurrentThread();
    if ( client != null ) {
      //---- member name ----//
      String mn = this.name
                + "_" + RemoteTestModule.getMyHost()
                + "_" + RemoteTestModule.getMyPid();
      // Since gemfire now requires that NAME_NAME be unique
      // we only set it if this is a hydra client
      p.setProperty( DistributionConfig.NAME_NAME, mn );
    }

    //---- distributed system id ----//
    if (!this.getUseLocator() && this.getEnableMcast()) {
      // multicast discovery
      GemFireVersionHelper.setDistributedSystemId(p, this.distributedSystemId);
    } // locators set the id when they are started, loners are unsupported

    //---- system files (log and archive) ----//
    p.setProperty( DistributionConfig.LOG_FILE_NAME,
                   this.getLogFileName());
    p.setProperty( DistributionConfig.LOG_LEVEL_NAME,
                   this.logLevel );
    p.setProperty( DistributionConfig.LOG_FILE_SIZE_LIMIT_NAME,
                   this.logFileSizeLimit.toString() );
    p.setProperty( DistributionConfig.LOG_DISK_SPACE_LIMIT_NAME,
                   this.logDiskSpaceLimit.toString() );
    p.setProperty( DistributionConfig.STATISTIC_ARCHIVE_FILE_NAME,
                   getSysDirName() + "/" + this.statisticArchiveFile );
    p.setProperty( DistributionConfig.STATISTIC_SAMPLING_ENABLED_NAME,
                   String.valueOf( this.statisticSamplingEnabled ) );
    p.setProperty( DistributionConfig.STATISTIC_SAMPLE_RATE_NAME,
                   this.statisticSampleRate.toString() );
    p.setProperty( DistributionConfig.ENABLE_TIME_STATISTICS_NAME,
                   String.valueOf( this.enableTimeStatistics ) );
    p.setProperty( DistributionConfig.ARCHIVE_FILE_SIZE_LIMIT_NAME,
                   this.archiveFileSizeLimit.toString() );
    p.setProperty( DistributionConfig.ARCHIVE_DISK_SPACE_LIMIT_NAME,
                   this.archiveDiskSpaceLimit.toString() );

    //---- distribution configuration ----//
    p.setProperty( DistributionConfig.ACK_WAIT_THRESHOLD_NAME,
                   this.ackWaitThreshold.toString() );

    //---- distribution type ----// @todo do not set mcast port until needed for connection 
    if (this.distributedSystem.equals(GemFirePrms.LONER)) {
      p.setProperty( DistributionConfig.LOCATORS_NAME, "" );
    } else if (TestConfig.tab().booleanAt(Prms.manageLocatorAgents)) {
      p.setProperty( DistributionConfig.LOCATORS_NAME,
                     TestConfig.getInstance().getMasterDescription()
                     .getLocator( this.distributedSystem ) );
    } else { // client-managed
      List endpoints =
        DistributedSystemHelper.getEndpoints(this.distributedSystem);
      p.setProperty(DistributionConfig.LOCATORS_NAME,
                    DistributedSystemHelper.endpointsToString(endpoints));
    }

    if (this.remoteDistributedSystems != null) {
      GemFireVersionHelper.setRemoteLocators(p,
                           this.remoteDistributedSystems);
    }

    //---- protocol ----//
    if (TestConfig.tab().booleanAt(Prms.useIPv6)) {
      String ipv6 = HostHelper.getHostAddress();
      if (ipv6 == null) {
        String s = "IPv6 address is not available for host " + hd.getHostName();
        throw new HydraRuntimeException(s);
      }
      p.setProperty(DistributionConfig.BIND_ADDRESS_NAME, ipv6);
      p.setProperty(DistributionConfig.SERVER_BIND_ADDRESS_NAME, ipv6);
    }

    p.setProperty( DistributionConfig.MCAST_ADDRESS_NAME,
                   this.mcastAddress );
    p.setProperty( DistributionConfig.MCAST_PORT_NAME,
                   this.mcastPort.toString() );
    p.setProperty( DistributionConfig.MCAST_TTL_NAME,
                   this.mcastTtl.toString() );
    p.setProperty( DistributionConfig.MCAST_SEND_BUFFER_SIZE_NAME,
                   this.mcastSendBufferSize.toString() );
    p.setProperty( DistributionConfig.MCAST_RECV_BUFFER_SIZE_NAME,
                   this.mcastRecvBufferSize.toString() );
    String mcastFlowControlString = this.mcastFlowControlByteAllowance + ", "
                                  + this.mcastFlowControlRechargeThreshold + ", "
	                          + this.mcastFlowControlRechargeBlockMs;
    p.setProperty( DistributionConfig.MCAST_FLOW_CONTROL_NAME,
                   mcastFlowControlString );
    p.setProperty( DistributionConfig.UDP_FRAGMENT_SIZE_NAME,
                   this.udpFragmentSize.toString() );
    p.setProperty( DistributionConfig.UDP_SEND_BUFFER_SIZE_NAME,
                   this.udpSendBufferSize.toString() );
    p.setProperty( DistributionConfig.UDP_RECV_BUFFER_SIZE_NAME,
                   this.udpRecvBufferSize.toString() );
    GemFireVersionHelper.setTcpPort(p, this.tcpPort);
    p.setProperty( DistributionConfig.DISABLE_TCP_NAME,
                   this.disableTcp.toString() );
    GemFireVersionHelper.setMembershipPortRange(p, this.membershipPortRange);
    p.setProperty( DistributionConfig.MEMBER_TIMEOUT_NAME,
                   this.memberTimeout.toString() );

    //---- sockets ----// @todo add bind-address
    p.setProperty( DistributionConfig.CONSERVE_SOCKETS_NAME,
                   this.conserveSockets.toString() );
    GemFireVersionHelper.setDeltaPropagation(p, this.deltaPropagation);
    p.setProperty( DistributionConfig.SOCKET_LEASE_TIME_NAME,
                   this.socketLeaseTime.toString() );
    p.setProperty( DistributionConfig.SOCKET_BUFFER_SIZE_NAME,
                   this.socketBufferSize.toString() );

    //--- roles ---//
    p.setProperty( DistributionConfig.ROLES_NAME,
                   this.roles.toString());
    p.setProperty(DistributionConfig.MAX_WAIT_TIME_FOR_RECONNECT_NAME,
                   this.maxWaitTimeForReconnect.toString());
    p.setProperty(DistributionConfig.MAX_NUM_RECONNECT_TRIES,
    		       this.maxNumReconnectTries.toString());   


    p.setProperty( DistributionConfig.ASYNC_DISTRIBUTION_TIMEOUT_NAME,
                   this.asyncDistributionTimeout.toString() );
    p.setProperty( DistributionConfig.ASYNC_QUEUE_TIMEOUT_NAME,
                   this.asyncQueueTimeout.toString() );
    p.setProperty( DistributionConfig.ASYNC_MAX_QUEUE_SIZE_NAME,
                   this.asyncMaxQueueSize.toString() );

    //--- durableClient ---//
    if ( client != null && this.enableDurableClient.booleanValue() ) {
      p.setProperty( DistributionConfig.DURABLE_CLIENT_ID_NAME,
                     "vm_" + RemoteTestModule.getMyVmid());
      p.setProperty( DistributionConfig.DURABLE_CLIENT_TIMEOUT_NAME,
                     this.durableClientTimeout.toString() );
    }

    //-- conflate-events --//
    p.setProperty(DistributionConfig.CLIENT_CONFLATION_PROP_NAME,
                  this.conflateEvents);

    //-- redundancy --//
    GemFireVersionHelper.setEnforceUniqueHost(p, this.enforceUniqueHost);
    if (this.redundancyZone != null) {
      GemFireVersionHelper.setRedundancyZone(p, this.redundancyZone);
    }

    //--- splitBrain ---//
    p.setProperty(DistributionConfig.ACK_SEVERE_ALERT_THRESHOLD_NAME,
                  this.ackSevereAlertThreshold.toString());
    p.setProperty(DistributionConfig.ENABLE_NETWORK_PARTITION_DETECTION_NAME,
                  this.enableNetworkPartitionDetection.toString());

    //--- jmxManager ---//
    if (this.jmxManager != null) {
      GemFireVersionHelper.setJMXManager(p, this.jmxManager.toString());
    }
    GemFireVersionHelper.setJMXManagerAccessFile(p,
                         this.jmxManagerAccessFile, this.hd);
    GemFireVersionHelper.setJMXManagerPasswordFile(p,
                         this.jmxManagerPasswordFile, this.hd);
    GemFireVersionHelper.setJMXManagerPorts(p,
                         this.jmxManagerPort.booleanValue(),
                         this.jmxManagerHttpPort.booleanValue());
    GemFireVersionHelper.setJMXManagerSSL(p, this.jmxManagerSSL.toString());
    // set the vm-level properties related to ssl as a side-effect
    if (this.jmxManagerSSL.booleanValue()) {
      if (this.sslDescription != null) {
        this.sslDescription.setSystemProperties(this.mcastPort.toString());
        // unfortunately we don't know if this will be used without passing
        // in locatorness from DistributedSystemHelper (tbd)
      }
    }
    GemFireVersionHelper.setJMXManagerStart(p, this.jmxManagerStart.toString());

    //--- ssl ---//
    if (this.sslDescription != null && !GemFireLocatorAgent.MasterManagedLocator) {
      addProperties(this.sslDescription.getDistributedSystemProperties(),
                    p);
      // set the vm-level properties related to ssl as a side-effect
      if (this.sslDescription.getSSLEnabled()) {
        this.sslDescription.setSystemProperties(this.mcastPort.toString());
      }
    }

    //--- security ---//
    if (this.securityDescription != null && !GemFireLocatorAgent.MasterManagedLocator) {
      addProperties(this.securityDescription
                        .getDistributedSystemProperties(this), p);
    }

    //--- offHeapMemorySize ---//
    if (this.offHeapMemorySize != null) {
      GemFireVersionHelper.setOffHeapMemorySize(p, this.offHeapMemorySize);
    }

    //--- lockMemory ---//
    p.setProperty(DistributionConfig.LOCK_MEMORY_NAME,
                  getLockMemory().toString());

    //--- disableAutoReconnect ---//
    if (this.disableAutoReconnect != null) {
      GemFireVersionHelper.setDisableAutoReconnect(p, this.disableAutoReconnect);
    }
    return p;
  }

  //////////////////////////////////////////////////////////////////////////////
  ////    PRINTING                                                          ////
  //////////////////////////////////////////////////////////////////////////////

  public SortedMap toSortedMap() {
    SortedMap map = new TreeMap();
    String header = this.getClass().getName() + "." + this.getName() + ".";
    map.put( header + "hostName", this.getHostDescription().getName() );
    map.put( header + "distributedSystem", this.getDistributedSystem() );
    map.put( header + "remoteDistributedSystems", this.getRemoteDistributedSystems() );
    map.put( header + "distributedSystemId", this.getDistributedSystemId() );
    map.put( header + "systemDir", this.getSystemDirectoryStr() );
    map.put( header + "stopSystemAfterTest", String.valueOf( this.stopSystemAfterTest() ) );
    map.put( header + "ackWaitThreshold", this.getAckWaitThreshold() );
    map.put( header + "logLevel", this.getLogLevel() );
    map.put( header + "logFileSizeLimit", this.getLogFileSizeLimit() );
    map.put( header + "logDiskSpaceLimit", this.getLogDiskSpaceLimit() );
    map.put( header + "statisticArchiveFile", this.getStatisticArchiveFile() );
    map.put( header + "statisticSamplingEnabled", this.getStatisticSamplingEnabled() );
    map.put( header + "statisticSampleRate", this.getStatisticSampleRate() );
    map.put( header + "enableTimeStatistics", this.getEnableTimeStatistics() );
    map.put( header + "archiveFileSizeLimit", this.getArchiveFileSizeLimit() );
    map.put( header + "archiveDiskSpaceLimit", this.getArchiveDiskSpaceLimit() );
    map.put( header + "conserveSockets", this.getConserveSockets() );
    map.put( header + "deltaPropagation", this.getDeltaPropagation() );
    map.put( header + "socketLeaseTime", this.getSocketLeaseTime() );
    map.put( header + "socketBufferSize", this.getSocketBufferSize() );
    map.put( header + "roles", this.getRoles() );
    // for reconnect
    map.put( header + "max-wait-time-reconnect" , this.getMaxWaitTimeForReconnect());
    map.put( header + "max-num-reconnect-tries" , this.getMaxNumReconnectTries());

    map.put( header + "useLocator", this.getUseLocator() );
    map.put( header + "isPeerLocator", this.isPeerLocator() );
    map.put( header + "isServerLocator", this.isServerLocator() );
    map.put( header + "enableMcast", this.getEnableMcast() );
    map.put( header + "mcastAddress", this.getMcastAddress() );
    map.put( header + "mcastPort", this.getMcastPort() );
    map.put( header + "mcastTtl", this.getMcastTtl() );
    map.put( header + "mcastSendBufferSize", this.getMcastSendBufferSize() );
    map.put( header + "mcastRecvBufferSize", this.getMcastRecvBufferSize() );
    map.put( header + "mcastFlowControlByteAllowance", this.getMcastFlowControlByteAllowance() );
    map.put( header + "mcastFlowControlRechargeThreshold", this.getMcastFlowControlRechargeThreshold() );
    map.put( header + "mcastFlowControlRechargeBlockMs", this.getMcastFlowControlRechargeBlockMs() );
    map.put( header + "udpFragmentSize", this.getUdpFragmentSize() );
    map.put( header + "udpSendBufferSize", this.getUdpSendBufferSize() );
    map.put( header + "udpRecvBufferSize", this.getUdpRecvBufferSize() );
    map.put( header + "tcpPort", this.getTcpPort() );
    map.put( header + "disableTcp", this.getDisableTcp() );
    map.put( header + "membershipPortRange", this.getMembershipPortRange() );
    map.put( header + "memberTimeout", this.getMemberTimeout() );
    if ( this.getLocatorClassPath() != null )
      map.put( header + "locatorClassPath", this.getLocatorClassPath() );
    if ( this.getExtraLocatorVMArgs() != null )
      map.put( header + "extraLocatorVMArgs", this.getExtraLocatorVMArgs() );
   
    map.put( header + "asyncDistributionTimeout", this.getAsyncDistributionTimeout() );
    map.put( header + "asyncQueueTimeout", this.getAsyncQueueTimeout() );
    map.put( header + "asyncMaxQueueSize", this.getAsyncMaxQueueSize() );

    map.put( header + "enableDurableClient", this.getEnableDurableClient() );
    map.put( header + "durableClientTimeout", this.getDurableClientTimeout() );
    map.put(header + "conflateEvents", this.getConflateEvents());
    map.put(header + "enforceUniqueHost", this.getEnforceUniqueHost());
    map.put(header + "redundancyZone", this.getRedundancyZone());
    map.put(header + "securityName", this.getSecurityName());
    if (this.getJMXManager() == null) {
      map.put(header + "jmxManager", "uses_product_default");
    } else {
      map.put(header + "jmxManager", this.getJMXManager().booleanValue());
    }
    map.put(header + "jmxManagerAccessFile", this.getJMXManagerAccessFile());
    if (this.getJMXManagerHttpPort().booleanValue()) {
      map.put(header + "jmxManagerHttpPort", "autogenerated");
    } else {
      map.put(header + "jmxManagerHttpPort", JMXManagerHelper.DEFAULT_JMX_MANAGER_HTTP_PORT);
    }
    map.put(header + "jmxManagerPasswordFile", this.getJMXManagerPasswordFile());
    if (this.getJMXManagerPort().booleanValue()) {
      map.put(header + "jmxManagerPort", "autogenerated");
    } else {
      map.put(header + "jmxManagerPort", JMXManagerHelper.DEFAULT_JMX_MANAGER_PORT);
    }
    map.put(header + "jmxManagerSSL", this.getJMXManagerSSL().booleanValue());
    map.put(header + "jmxManagerStart", this.getJMXManagerStart().booleanValue());
    map.put(header + "sslName", this.getSSLName());
    map.put( header + "ackSevereAlertThreshold", this.getAckSevereAlertThreshold());
    map.put( header + "enableNetworkPartitionDetection", this.getEnableNetworkPartitionDetection());
    map.put( header + "disableAutoReconnect", this.getDisableAutoReconnect());
    map.put( header + "offHeapMemorySize", this.getOffHeapMemorySize());
    map.put( header + "lockMemory", this.getLockMemory());
    return map;
  }

  //////////////////////////////////////////////////////////////////////////////
  ////    CONFIGURATION                                                     ////
  //////////////////////////////////////////////////////////////////////////////

  /**
   *  Creates gemfire descriptions from the gemfire parameters in the test
   *  configuration.
   */
  protected static void configure( TestConfig config ) {

    ConfigHashtable tab = config.getParameters();
    // All gfd's that do not specify addresss will use this one.
    // Different distributed systems will have same address different port
    // C++ testing uses 224.10.11.[1-254] & 224.10.10.*
    GsRandom rand = new GsRandom();
    String randomMcastAddress;
    if (tab.booleanAt(Prms.useIPv6)) {
      int addr = rand.nextInt(1111, 8888);
      randomMcastAddress = "FF38::" + addr;
    } else { // use IPv4
      int addr1 = rand.nextInt(81, 254);
      int addr2 = rand.nextInt(1, 254);
      randomMcastAddress = "239.192." + addr1 + "." + addr2;
    }

    // create a description for each system name
    Vector names = tab.vecAt( GemFirePrms.names, new HydraVector() );
    Map<String, Integer> distributedSystems = new HashMap();
    int nextDSID = 0;

    for ( int i = 0; i < names.size(); i++ ) {

      GemFireDescription gfd = new GemFireDescription();

      // name
      String name = (String) names.elementAt(i);
      gfd.setName( name );

      // hostName
      String hostName = tab.stringAtWild( GemFirePrms.hostNames, i, null );
      if ( hostName == null )
        throw new HydraConfigException( "Missing " + BasePrms.nameForKey( GemFirePrms.hostNames ) );
      HostDescription hd = config.getHostDescription( hostName );
      if ( hd == null )
        throw new HydraConfigException( "Undefined value in " + BasePrms.nameForKey( GemFirePrms.hostNames ) + ": " + hostName );
      gfd.setHostDescription( hd );

      // distributedSystem and distributedSystemId
      String distributedSystem = tab.stringAtWild( GemFirePrms.distributedSystem, i, GemFirePrms.DEFAULT_DISTRIBUTED_SYSTEM_NAME );
      if (distributedSystem.equalsIgnoreCase(GemFirePrms.LONER)) {
        gfd.setDistributedSystem(GemFirePrms.LONER);
        gfd.setDistributedSystemId(DistributionConfig.DEFAULT_DISTRIBUTED_SYSTEM_ID);
        // loners unsupported at this time (requires unique id per loner jvm)
      } else {
        gfd.setDistributedSystem( distributedSystem );
        // use the same dsid for all descriptions for a given system
        Integer dsid = distributedSystems.get(distributedSystem);
        if (dsid == null) {
          // try to use the dsid in the system name, if possible
          dsid = toDSID(distributedSystem);
          if (dsid == null || distributedSystems.values().contains(dsid)) {
            // use the next available dsid
            do {
              dsid = Integer.valueOf(++nextDSID);
            } while (distributedSystems.values().contains(dsid));
          }
          distributedSystems.put(distributedSystem, dsid);
        }
        gfd.setDistributedSystemId(dsid);
        // note that the id is only used for locators and multicast
      }

      // remoteDistributedSystems
      {
        Long key = GemFirePrms.remoteDistributedSystems;
        Vector strs = tab.vecAtWild(key, i, null);
        if (strs != null) {
          for (Iterator it = strs.iterator(); it.hasNext();) {
            String str = tab.getString(key, it.next());
            if (str == null || str.equalsIgnoreCase(BasePrms.NONE)) {
              it.remove();
            }
          }
          if (strs.size() > 0) {
            // disallow with master-managed locators
            if (TestConfig.tab().booleanAt(Prms.manageLocatorAgents)) {
              String s = BasePrms.nameForKey(key) + " cannot be used when "
                       + BasePrms.nameForKey(Prms.manageLocatorAgents)
                       + "=true.  Use hydra client-managed locators.";
              throw new HydraConfigException(s);
            }
            // disallow with loners
            if (gfd.getDistributedSystem().equals(GemFirePrms.LONER)) {
              String s = BasePrms.nameForKey(key) + " cannot be used when "
                       + BasePrms.nameForKey(GemFirePrms.distributedSystem)
                       + "=" + GemFirePrms.LONER + ".  Set remote distributed "
                       + "systems for loners to " + BasePrms.NONE;
              throw new HydraConfigException(s);
            }
            gfd.setRemoteDistributedSystems(strs);
          }
        }
      }

      // systemDir
      gfd.setSystemDirectory(hd.getResourceDir() + hd.getFileSep() + name);

      // stopSystemAfterTest
      boolean stopSystemAfterTest = tab.booleanAtWild( GemFirePrms.stopSystemsAfterTest, i, Boolean.TRUE );
      gfd.setStopSystemAfterTest( stopSystemAfterTest );

      // ackWaitThreshold
      int ackWaitThreshold = tab.intAtWild( GemFirePrms.ackWaitThreshold, i, DistributionConfig.DEFAULT_ACK_WAIT_THRESHOLD );
      gfd.setAckWaitThreshold( new Integer( ackWaitThreshold ) );

      // logLevel
      String logLevel = tab.stringAtWild( GemFirePrms.logLevel, i, LogWriterImpl.levelToString( DistributionConfig.DEFAULT_LOG_LEVEL ) );
      gfd.setLogLevel( logLevel );

      // logFileSizeLimit
      int logFileSizeLimit = tab.intAtWild( GemFirePrms.logFileSizeLimit, i, DistributionConfig.DEFAULT_LOG_FILE_SIZE_LIMIT );
      gfd.setLogFileSizeLimit( new Integer( logFileSizeLimit ) );

      // logDiskSpaceLimit
      int logDiskSpaceLimit = tab.intAtWild( GemFirePrms.logDiskSpaceLimit, i, DistributionConfig.DEFAULT_LOG_DISK_SPACE_LIMIT );
      gfd.setLogDiskSpaceLimit( new Integer( logDiskSpaceLimit ) );

      // statisticArchiveFile
      gfd.setStatisticArchiveFile("statArchive.gfs");

      // statisticSamplingEnabled
      boolean statisticSamplingEnabled = tab.booleanAtWild( GemFirePrms.statisticSamplingEnabled, i, Boolean.TRUE );
      gfd.setStatisticSamplingEnabled( Boolean.valueOf( statisticSamplingEnabled ) );

      // statisticSampleRate
      int statisticSampleRate = tab.intAtWild( GemFirePrms.statisticSampleRate, i, DistributionConfig.DEFAULT_STATISTIC_SAMPLE_RATE );
      gfd.setStatisticSampleRate( new Integer( statisticSampleRate ) );

      // enableTimeStatistics
      boolean enableTimeStatistics = tab.booleanAtWild(GemFirePrms.enableTimeStatistics, i, Boolean.TRUE);
      gfd.setEnableTimeStatistics(Boolean.valueOf(enableTimeStatistics));

      // archiveFileSizeLimit
      int archiveFileSizeLimit = tab.intAtWild( GemFirePrms.archiveFileSizeLimit, i, DistributionConfig.DEFAULT_ARCHIVE_FILE_SIZE_LIMIT );
      gfd.setArchiveFileSizeLimit( new Integer( archiveFileSizeLimit ) );

      // archiveDiskSpaceLimit
      int archiveDiskSpaceLimit = tab.intAtWild( GemFirePrms.archiveDiskSpaceLimit, i, DistributionConfig.DEFAULT_ARCHIVE_DISK_SPACE_LIMIT );
      gfd.setArchiveDiskSpaceLimit( new Integer( archiveDiskSpaceLimit ) );

      // conserveSockets
      boolean conserveSockets = tab.booleanAtWild( GemFirePrms.conserveSockets, i, Boolean.valueOf(DistributionConfig.DEFAULT_CONSERVE_SOCKETS) );
      gfd.setConserveSockets( Boolean.valueOf(conserveSockets) );

      // deltaPropagation
      gfd.setDeltaPropagation(GemFireVersionHelper.getDeltaPropagation(tab, i));

      // socketLeaseTime
      long socketLeaseTime = tab.longAtWild( GemFirePrms.socketLeaseTime, i, DistributionConfig.DEFAULT_SOCKET_LEASE_TIME );
      gfd.setSocketLeaseTime( new Long( socketLeaseTime ) );

      // socketBufferSize
      long socketBufferSize = tab.longAtWild( GemFirePrms.socketBufferSize, i, DistributionConfig.DEFAULT_SOCKET_BUFFER_SIZE );
      gfd.setSocketBufferSize( new Long( socketBufferSize) );

      // roles 
      String roles = tab.stringAtWild( GemFirePrms.roles, i, DistributionConfig.DEFAULT_ROLES );
      if (roles.equalsIgnoreCase(BasePrms.DEFAULT) ||
          roles.equalsIgnoreCase(BasePrms.NONE)) {
        roles = DistributionConfig.DEFAULT_ROLES;
      }
      gfd.setRoles( roles );

      // reconnect
      int maxWaitTimeForReconnect = tab.intAtWild(GemFirePrms.maxWaitTimeForReconnect, i, DistributionConfig.DEFAULT_MAX_WAIT_TIME_FOR_RECONNECT);
      gfd.setMaxWaitTimeForReconnect(new Integer(maxWaitTimeForReconnect));

      int reconnectTimes = tab.intAtWild(GemFirePrms.maxNumReconnectTries, i , DistributionConfig.DEFAULT_MAX_NUM_RECONNECT_TRIES);
      gfd.setMaxNumReconnectTries(new Integer(reconnectTimes));

      // multicastAddress 
      String configMcastAddress = tab.stringAtWild( GemFirePrms.mcastAddress, i, "null" );
      if (configMcastAddress.equalsIgnoreCase("null")) {
        gfd.setMcastAddress( randomMcastAddress );
      }
      else {
        gfd.setMcastAddress( configMcastAddress );
      }

      // enableMcast - defaults to false, and is always false for loners
      boolean enableMcast = false;
      if (gfd.getDistributedSystem().equals(GemFirePrms.LONER)) {
        gfd.setEnableMcast(Boolean.FALSE);
      } else {
        enableMcast = tab.booleanAtWild(GemFirePrms.enableMcast, i, Boolean.FALSE);
        gfd.setEnableMcast(Boolean.valueOf(enableMcast));
      }

      // useLocator - defaults to true if multicast not enabled, and is always
      // false for loners
      if (gfd.getDistributedSystem().equals(GemFirePrms.LONER)) {
        gfd.setUseLocator(Boolean.FALSE);
      } else {
        Boolean defaultVal = null;
        if (! enableMcast) {
          defaultVal = Boolean.TRUE;
        }
        boolean useLocator = tab.booleanAtWild(GemFirePrms.useLocator, i, defaultVal);
        gfd.setUseLocator(Boolean.valueOf(useLocator));
      }

      // isPeerLocator
      {
        Long key = GemFirePrms.isPeerLocator;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = Boolean.TRUE;
        }
        gfd.setPeerLocator(bool);
      }
      // isServerLocator
      {
        Long key = GemFirePrms.isServerLocator;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = Boolean.TRUE;
        }
        gfd.setServerLocator(bool);
      }

      // mcastPort - defaults to dynamic, and is always 0 for loners
      if (gfd.getDistributedSystem().equals(GemFirePrms.LONER)) {
        gfd.setMcastPort(new Integer(0));
      } else {
        int mcastPort = DistributionConfig.MIN_MCAST_PORT;
        if (enableMcast) {
          // If user has specifed port in conf, use it
          mcastPort = tab.intAtWild( GemFirePrms.mcastPort, i, DistributionConfig.MIN_MCAST_PORT);
        }
        gfd.setMcastPort ( new Integer(mcastPort) );
      }

      // mcastTtl 
      int mcastTtl = tab.intAtWild( GemFirePrms.mcastTtl, i, 0 );
      gfd.setMcastTtl( new Integer( mcastTtl ) );

      // mcastSendBufferSize
      int mcastSendBufferSize = tab.intAtWild( GemFirePrms.mcastSendBufferSize, i, DistributionConfig.DEFAULT_MCAST_SEND_BUFFER_SIZE );
      gfd.setMcastSendBufferSize( new Integer( mcastSendBufferSize ) );

      // mcastRecvBufferSize
      int mcastRecvBufferSize = tab.intAtWild( GemFirePrms.mcastRecvBufferSize, i, DistributionConfig.DEFAULT_MCAST_RECV_BUFFER_SIZE );
      gfd.setMcastRecvBufferSize( new Integer( mcastRecvBufferSize) );
 
      // mcastFlowControlByteAllowance
      int mcastFlowControlByteAllowance = tab.intAtWild( GemFirePrms.mcastFlowControlByteAllowance, i, DistributionConfig.DEFAULT_MCAST_FLOW_CONTROL.getByteAllowance() );
      gfd.setMcastFlowControlByteAllowance( new Integer( mcastFlowControlByteAllowance ) );

      // mcastFlowControlRechargeThreshold 
      Float mcastFlowControlRechargeThreshold = new Float( tab.stringAtWild( GemFirePrms.mcastFlowControlRechargeThreshold, i,  Float.toString(DistributionConfig.DEFAULT_MCAST_FLOW_CONTROL.getRechargeThreshold() )));
      gfd.setMcastFlowControlRechargeThreshold( mcastFlowControlRechargeThreshold );

      // mcastFlowControlRechargeBlockMs
      int mcastFlowControlRechargeBlockMs = tab.intAtWild( GemFirePrms.mcastFlowControlRechargeBlockMs, i, DistributionConfig.DEFAULT_MCAST_FLOW_CONTROL.getRechargeBlockMs() );
      gfd.setMcastFlowControlRechargeBlockMs( new Integer( mcastFlowControlRechargeBlockMs ) );

      // udpFragmentSize
      int udpFragmentSize = tab.intAtWild( GemFirePrms.udpFragmentSize, i, DistributionConfig.DEFAULT_UDP_FRAGMENT_SIZE );
      gfd.setUdpFragmentSize( new Integer( udpFragmentSize) );
        
      // udpSendBufferSize
      int udpSendBufferSize = tab.intAtWild( GemFirePrms.udpSendBufferSize, i, DistributionConfig.DEFAULT_UDP_SEND_BUFFER_SIZE );
      gfd.setUdpSendBufferSize( new Integer( udpSendBufferSize) );

      // udpRecvBufferSize
      int udpRecvBufferSize = tab.intAtWild( GemFirePrms.udpRecvBufferSize, i, DistributionConfig.DEFAULT_UDP_RECV_BUFFER_SIZE );
      gfd.setUdpRecvBufferSize( new Integer( udpRecvBufferSize) );

      // disableTcp
      boolean disableTcp = tab.booleanAtWild( GemFirePrms.disableTcp, i, Boolean.valueOf( DistributionConfig.DEFAULT_DISABLE_TCP ));
      gfd.setDisableTcp( Boolean.valueOf( disableTcp) );

      // tcpPort
      gfd.setTcpPort(GemFireVersionHelper.getTcpPort(tab, i));

      // membershipPortRange
      gfd.setMembershipPortRange(GemFireVersionHelper.getMembershipPortRange(tab, i));

      // memberTimeout
      int memberTimeout = tab.intAtWild( GemFirePrms.memberTimeout, i, DistributionConfig.DEFAULT_MEMBER_TIMEOUT );
      gfd.setMemberTimeout( new Integer( memberTimeout) );

      // master-managed locator classpath (same for all names)
      HostDescription mhd = config.getMasterDescription().getVmDescription()
                                  .getHostDescription();
      Vector classPath = new Vector();
      // add custom classpath, if any
      Vector extraLocatorClassPath =
        tab.vecAt( GemFirePrms.extraLocatorClassPath, null );
      if ( extraLocatorClassPath != null ) {
        extraLocatorClassPath =
          EnvHelper.expandEnvVars(extraLocatorClassPath, mhd);
        classPath.addAll( extraLocatorClassPath );
      }
      // add default classPath -- test classes, junit,  and product jar
      classPath.add(mhd.getTestDir());
      classPath.add(mhd.getTestDir() + mhd.getFileSep() + "junit.jar");
      if (mhd.getExtraTestDir() != null) {
        classPath.add(mhd.getExtraTestDir());
      }
      if (mhd.getGemFireHome() != null) {
        classPath.add(mhd.getGemFireHome() + mhd.getFileSep() + "lib"
                + mhd.getFileSep() + "snappydata-store-" +
                ProductVersionHelper.getInfo().getProperty(ProductVersionHelper.SNAPPYRELEASEVERSION) + ".jar");
        classPath.add(mhd.getGemFireHome() + mhd.getFileSep() + "lib");
        classPath.add(VmDescription.getSnappyJarPath(hd.getTestDir() + hd.getFileSep() + ".." + hd.getFileSep() + ".." + hd.getFileSep() + ".." + hd.getFileSep() + ".." + hd.getFileSep() + ".." + hd.getFileSep() + ".." + hd.getFileSep() + ".." + hd.getFileSep() + "dtests" + hd.getFileSep() +
                "build-artifacts" + hd.getFileSep() + "scala-2.11" + hd.getFileSep() + "libs", "snappydata-store-scala-tests"));
      }
      // set classpath at last
      Vector expandedClassPath = EnvHelper.expandEnvVars(classPath, mhd);
      gfd.setLocatorClassPath(EnvHelper.asPath(expandedClassPath, mhd));

      // extraLocatorVMArgs (same for all names)
      String extraLocatorVMArgs = GemFirePrms.getExtraLocatorVMArgs(i);
      if ( extraLocatorVMArgs != null ) {
        gfd.setExtraLocatorVMArgs( extraLocatorVMArgs.trim() );
      }

      // asyncDistributionTimeout
      int asyncDistributionTimeout = tab.intAtWild( GemFirePrms.asyncDistributionTimeout, i, DistributionConfig.DEFAULT_ASYNC_DISTRIBUTION_TIMEOUT );
      gfd.setAsyncDistributionTimeout( new Integer( asyncDistributionTimeout ) );

      // asyncQueueTimeout
      int asyncQueueTimeout = tab.intAtWild( GemFirePrms.asyncQueueTimeout, i, DistributionConfig.DEFAULT_ASYNC_QUEUE_TIMEOUT );
      gfd.setAsyncQueueTimeout( new Integer( asyncQueueTimeout ) );
 
      // asyncMaxQueueSize
      int asyncMaxQueueSize = tab.intAtWild( GemFirePrms.asyncMaxQueueSize, i, DistributionConfig.DEFAULT_ASYNC_MAX_QUEUE_SIZE );
      gfd.setAsyncMaxQueueSize( new Integer( asyncMaxQueueSize) );

      // enableDurableClient
      boolean enableDurableClient = tab.booleanAtWild( GemFirePrms.enableDurableClient, i, Boolean.FALSE );
      gfd.setEnableDurableClient( Boolean.valueOf(enableDurableClient) );

      // durableClientTimeout
      int durableClientTimeout = tab.intAtWild( GemFirePrms.durableClientTimeout, i, DistributionConfig.DEFAULT_DURABLE_CLIENT_TIMEOUT );
      gfd.setDurableClientTimeout( new Integer(durableClientTimeout) );

      // conflateEvents
      {
        Long key = GemFirePrms.conflateEvents;
        String str = tab.getString(key, tab.getWild(key, i, null));
        if (str == null) {
          str = DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_DEFAULT;
        }
        gfd.setConflateEvents(getConflateEvents(str, key));
      }
      // enforceUniqueHost
      {
        Long key = GemFirePrms.enforceUniqueHost;
        Boolean b = tab.getBoolean(key, tab.getWild(key, i, null));
        if (b == null) {
          b = Boolean.FALSE;
        }
        gfd.setEnforceUniqueHost(b);
      }
      // redundancyZone
      {
        Long key = GemFirePrms.redundancyZone;
        String str = tab.getString(key, tab.getWild(key, i, null));
        if (str != null) {
          gfd.setRedundancyZone(str);
        }
      }
      // jmxManager
      {
        Long key = GemFirePrms.jmxManager;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool != null) {
          gfd.setJMXManager(bool);
        }
      }
      // jmxManagerAccessFile
      {
        Long key = GemFirePrms.jmxManagerAccessFile;
        String str = tab.getString(key, tab.getWild(key, i, null));
        if (str != null) {
          gfd.setJMXManagerAccessFile(str);
        }
      }
      // jmxManagerHttpPort
      {
        Long key = GemFirePrms.jmxManagerHttpPort;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = Boolean.FALSE;
        }
        gfd.setJMXManagerHttpPort(bool);
      }
      // jmxManagerPasswordFile
      {
        Long key = GemFirePrms.jmxManagerPasswordFile;
        String str = tab.getString(key, tab.getWild(key, i, null));
        if (str != null) {
          gfd.setJMXManagerPasswordFile(str);
        }
      }
      // jmxManagerPort
      {
        Long key = GemFirePrms.jmxManagerPort;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = Boolean.FALSE;
        }
        gfd.setJMXManagerPort(bool);
      }
      // jmxManagerSSL
      {
        Long key = GemFirePrms.jmxManagerSSL;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = Boolean.valueOf(DistributionConfig.DEFAULT_JMX_MANAGER_SSL);
        }
        gfd.setJMXManagerSSL(bool);
      }
      // jmxManagerStart
      {
        Long key = GemFirePrms.jmxManagerStart;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = Boolean.valueOf(DistributionConfig.DEFAULT_JMX_MANAGER_START);
        }
        gfd.setJMXManagerStart(bool);
      }
      // securityName (generates securityDescription)
      {
        Long key = GemFirePrms.securityName;
        String str = tab.getString(key, tab.getWild(key, i, null));
        if (str != null && !str.equalsIgnoreCase(BasePrms.NONE)) {
          gfd.setSecurityName("SecurityDescription." + str);
          gfd.setSecurityDescription(getSecurityDescription(str, key, config));
        }
      }
      // sslName (generates sslDescription)
      {
        Long key = GemFirePrms.sslName;
        String str = tab.getString(key, tab.getWild(key, i, null));
        if (str != null && !str.equalsIgnoreCase(BasePrms.NONE)) {
          gfd.setSSLName("SSLDescription." + str);
          gfd.setSSLDescription(getSSLDescription(str, key, config));
        }
      }
      // disallow use of multicast with ipv6 if asked
      {
        // allowMcastWithIPv6
        boolean allowMcastWithIPv6 =
          tab.booleanAtWild(GemFirePrms.allowMcastWithIPv6, i, Boolean.FALSE);
        if (!allowMcastWithIPv6) {
          boolean ipv6 = Boolean.getBoolean(Prms.USE_IPV6_PROPERTY);
          if (ipv6 && gfd.getEnableMcast().booleanValue()) {
            String s = "Cannot use IPv6 ("
                     + BasePrms.nameForKey(Prms.useIPv6) + "="
                     + ipv6 + ") with multicast ("
                     + BasePrms.nameForKey(GemFirePrms.enableMcast) + "="
                     + gfd.getEnableMcast() + ") unless explicitly allowed ("
                     + BasePrms.nameForKey(GemFirePrms.allowMcastWithIPv6) + "="
                     + allowMcastWithIPv6 + ")";
            throw new HydraConfigException(s);
          }
        }
      }
      // disallow use of security with multicast
      {
        if ((gfd.getSecurityName() != null &&
             gfd.getMcastPort().intValue() != 0)) {
          String s = "Cannot use security ("
                   + BasePrms.nameForKey(GemFirePrms.securityName) + "="
                   + gfd.getSecurityName() + ") with non-zero multicast port ("
                   + BasePrms.nameForKey(GemFirePrms.mcastPort) + "="
                   + gfd.getMcastPort() + ")";
          throw new HydraConfigException(s);
        }
      }
      // disallow use of SSL with multicast
      {
        if ((gfd.getSSLName() != null &&
             gfd.getMcastPort().intValue() != 0)) {
          String s = "Cannot use SSL ("
                   + BasePrms.nameForKey(GemFirePrms.sslName) + "="
                   + gfd.getSSLName() + ") with non-zero multicast port ("
                   + BasePrms.nameForKey(GemFirePrms.mcastPort) + "="
                   + gfd.getMcastPort() + ")";
          throw new HydraConfigException(s);
        }
      }
      // disallow use of SSL with conserveSockets
      /*
         @todo lises check this at runtime for edge clients or file bug
                     about silent failure
      {
        if ((gfd.getSSLName() != null &&
             gfd.getConserveSockets().booleanValue())) {
          String s = "Cannot use SSL ("
                   + BasePrms.nameForKey(GemFirePrms.sslName) + "="
                   + gfd.getSSLName() + ") with "
                   + BasePrms.nameForKey(GemFirePrms.conserveSockets) + "="
                   + gfd.getConserveSockets();
          throw new HydraConfigException(s);
        }
      }
      */
      // ackSevereAlertThreshold
      int ackSevereAlertThreshold = tab.intAtWild(GemFirePrms.ackSevereAlertThreshold, i, DistributionConfig.DEFAULT_ACK_SEVERE_ALERT_THRESHOLD);
      gfd.setAckSevereAlertThreshold(new Integer(ackSevereAlertThreshold));

      // enableNetworkPartitionDetection
      boolean enableNetworkPartitionDetection = tab.booleanAtWild(GemFirePrms.enableNetworkPartitionDetection, i, Boolean.valueOf(DistributionConfig.DEFAULT_ENABLE_NETWORK_PARTITION_DETECTION));
      gfd.setEnableNetworkPartitionDetection(Boolean.valueOf(enableNetworkPartitionDetection));
      
      // disableAutoReconnect
      {
        Long key = GemFirePrms.disableAutoReconnect;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool != null) {
          gfd.setDisableAutoReconnect(bool);
        }
      }

      // offHeapMemorySize
      {
        Long key = GemFirePrms.offHeapMemorySize;
        String aStr = tab.getString(key, tab.getWild(key, i, null));
        if (aStr != null) {
          gfd.setOffHeapMemorySize(aStr);
        }
      }
      // lockMemory
      {
        Long key = GemFirePrms.lockMemory;
        Boolean bool = tab.getBoolean(key, tab.getWild(key, i, null));
        if (bool == null) {
          bool = DistributionConfig.DEFAULT_LOCK_MEMORY;
        }
        gfd.setLockMemory(bool);
      }

      // all done
      config.addGemFireDescription( gfd );
    }
  }

//------------------------------------------------------------------------------
// Conflate events configuration support
//------------------------------------------------------------------------------

  /**
   * Returns the conflate events setting for the given string.
   */
  private static String getConflateEvents(String str, Long key)
  {
    if (str.equalsIgnoreCase(
             DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_DEFAULT)) {
      return DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_DEFAULT;
    }
    else if (str.equalsIgnoreCase(
             DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_ON)) {
      return DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_ON;
    }
    else if (str.equalsIgnoreCase(
             DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_OFF)) {
      return DistributionConfig.CLIENT_CONFLATION_PROP_VALUE_OFF;
    }
    else { 
      String s = BasePrms.nameForKey(key) + " has illegal value: " + str;
      throw new HydraConfigException(s);
    }
  }

//------------------------------------------------------------------------------
// Distributed system id configuration support
//------------------------------------------------------------------------------

  /**
   * Extracts a distributed system id from a distributed system name of the
   * form "dsname_dsid".  For example, "ds_4" would return 4.  Returns null
   * if the name is not in this form.
   */
  private static Integer toDSID(String dsName) {
    String dsid = dsName.substring(dsName.indexOf("_") + 1, dsName.length());
    try {
      return Integer.parseInt(dsid);
    } catch (NumberFormatException e) {
      return null;
    }
  }

//------------------------------------------------------------------------------
// Security configuration support
//------------------------------------------------------------------------------

  /**
   * Returns the security description for the given string.
   * @throws HydraConfigException if the given string is not listed in {@link
   *         SecurityPrms#names}.
   */
  private static SecurityDescription getSecurityDescription(
                                     String str, Long key, TestConfig config) {
    SecurityDescription sd = config.getSecurityDescription(str);
    if (sd == null) {
      String s = BasePrms.nameForKey(key) + " not found in "
               + BasePrms.nameForKey(SecurityPrms.names) + ": " + str;
      throw new HydraConfigException(s);
    } else {
      return sd;
    }
  }
}
