/*
 * Decompiled with CFR 0.152.
 */
package hydra;

import hydra.EnvHelper;
import hydra.FileUtil;
import hydra.HadoopDescription;
import hydra.HadoopInfo;
import hydra.HadoopPrms;
import hydra.HostDescription;
import hydra.HostHelper;
import hydra.HydraInternalException;
import hydra.HydraRuntimeException;
import hydra.HydraTimeoutException;
import hydra.Log;
import hydra.PortHelper;
import hydra.ProcessMgr;
import hydra.RemoteTestModule;
import hydra.TestConfig;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

public class HadoopHelper {
    private static final String SHELL = "/bin/bash";
    private static final String HADOOP_DAEMON = "/sbin/hadoop-daemon.sh";
    private static final String YARN_DAEMON = "/sbin/yarn-daemon.sh";
    private static final String JSVC_HOME = "/export/gcm/where/java/hadoop/hadoop-secure-utils/bigtop-utils";
    public static final String GFXD_SECURE_KEYTAB_FILE = "/export/gcm/where/java/hadoop/hadoop-secure-keytabs/gfxd-secure.keytab";
    public static final String HDFS_SECURE_KEYTAB_FILE = "/export/gcm/where/java/hadoop/hadoop-secure-keytabs/hdfs-secure.keytab";
    public static final String MAPRED_SECURE_KEYTAB_FILE = "/export/gcm/where/java/hadoop/hadoop-secure-keytabs/mapred.keytab";
    public static final String YARN_SECURE_KEYTAB_FILE = "/export/gcm/where/java/hadoop/hadoop-secure-keytabs/yarn.keytab";
    private static final String JAVA_SECURITY_KRB5_CONF = "/export/gcm/where/java/hadoop/hadoop-secure-utils/krb5.conf";
    private static final String ROOT_SCRIPT = "/export/localnew/scripts/commands_for_secure_hdfs.sh";
    private static final String KINIT_CMD = "/usr/bin/kinit";
    private static final String KINIT = "/usr/bin/kinit -k -t";
    private static final String KDESTROY_CMD = "/usr/bin/kdestroy";
    private static final String KDESTROY = "/usr/bin/kdestroy -q";
    private static final String SUDO = "/usr/bin/sudo";
    private static final String[] ConfigurationFiles = new String[]{"capacity-scheduler.xml", "commons-logging.properties", "configuration.xsl", "hadoop-metrics.properties", "hadoop-metrics2.properties", "hadoop-policy.xml", "log4j.properties", "mapred-queues.xml"};

    public static synchronized void configureHadoop(String hadoopConfig) {
        Log.getLogWriter().info("Configuring hadoop for " + hadoopConfig);
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        StringBuilder sb = new StringBuilder();
        for (HadoopDescription.DataNodeDescription dnd : hdd.getDataNodeDescriptions()) {
            sb.append(dnd.getHostDescription().getCanonicalHostName()).append("\n");
        }
        String slaves = sb.toString();
        Log.getLogWriter().info("Configured hadoop slaves file for " + hadoopConfig + "\n" + slaves);
        ArrayList<HadoopDescription.NodeDescription> nds = new ArrayList<HadoopDescription.NodeDescription>();
        nds.addAll(hdd.getNameNodeDescriptions());
        nds.addAll(hdd.getDataNodeDescriptions());
        if (hdd.getResourceManagerDescription() != null) {
            nds.add(hdd.getResourceManagerDescription());
            nds.addAll(hdd.getNodeManagerDescriptions());
        }
        for (HadoopDescription.NodeDescription nd : nds) {
            String confDir = nd.getConfDir();
            Log.getLogWriter().info("Generating hadoop configuration files for " + nd.getName() + " in " + confDir);
            HadoopHelper.generateHadoopEnv(hdd, nd);
            HadoopHelper.generateYarnEnv(hdd, nd);
            HadoopHelper.generateCoreSiteXMLFile(hdd, nd);
            HadoopHelper.generateHDFSSiteXMLFile(hdd, nd);
            HadoopHelper.generateMapRedSiteXMLFile(hdd, nd);
            HadoopHelper.generateYarnSiteXMLFile(hdd, nd);
            if (hdd.isSecure()) {
                HadoopHelper.generateContainerExecutorCFGFile(hdd, nd);
            }
            HadoopHelper.writeSlavesFile(nd, slaves);
            HadoopHelper.copyMiscFiles(nd);
            Log.getLogWriter().info("Generated hadoop configuration files for " + nd.getName() + " in " + confDir);
        }
    }

    public static synchronized void configureHDFS(String hadoopConfig) {
        Log.getLogWriter().info("Configuring HDFS for " + hadoopConfig);
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        StringBuilder sb = new StringBuilder();
        for (String host : hdd.getDataNodeHosts()) {
            sb.append(host).append("\n");
        }
        String slaves = sb.toString();
        Log.getLogWriter().info("Configured hadoop slaves file for " + hadoopConfig + "\n" + slaves);
        ArrayList<HadoopDescription.NodeDescription> nds = new ArrayList<HadoopDescription.NodeDescription>();
        nds.addAll(hdd.getNameNodeDescriptions());
        nds.addAll(hdd.getDataNodeDescriptions());
        for (HadoopDescription.NodeDescription nd : nds) {
            String confDir = nd.getConfDir();
            Log.getLogWriter().info("Generating hadoop configuration files for " + nd.getName() + " in " + confDir);
            HadoopHelper.generateHadoopEnv(hdd, nd);
            HadoopHelper.generateCoreSiteXMLFile(hdd, nd);
            HadoopHelper.generateHDFSSiteXMLFile(hdd, nd);
            HadoopHelper.writeSlavesFile(nd, slaves);
            HadoopHelper.copyMiscFiles(nd);
            Log.getLogWriter().info("Generated hadoop configuration files for " + nd.getName() + " in " + confDir);
        }
    }

    private static void generateHadoopEnv(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        Log.getLogWriter().info("Generating hadoop-env for " + nd.getName());
        StringBuilder opts = new StringBuilder();
        opts.append("-Djava.net.preferIPv4Stack=true");
        opts.append(" -Dfs.defaultFS=").append(hdd.getNameNodeURL());
        opts.append(" -Dyarn.resourcemanager.address=").append(hdd.getResourceManagerURL());
        switch (nd.getNodeType()) {
            case DataNode: {
                opts.append(" -Ddfs.data.dir=").append(nd.getDataDirsAsString());
                break;
            }
            case NameNode: {
                opts.append(" -Ddfs.name.dir=").append(nd.getDataDirsAsString());
            }
        }
        if (hdd.isSecure()) {
            opts.append(" -Djava.security.krb5.conf=").append(JAVA_SECURITY_KRB5_CONF);
        }
        HostDescription hd = nd.getHostDescription();
        StringBuilder sb = new StringBuilder();
        sb.append("export JAVA_HOME=").append(hd.getJavaHome()).append("\n").append("export HADOOP_CONF_DIR=").append(nd.getConfDir()).append("\n").append("export HADOOP_LOG_DIR=").append(nd.getLogDir()).append("\n").append("export HADOOP_OPTS=\"").append(opts.toString().trim()).append("\"").append("\n").append("export HADOOP_ZKFC_OPTS=\"-Xmx256m\"").append("\n").append("export HADOOP_PID_DIR=").append(nd.getPIDDir()).append("\n").append("export HADOOP_NAMENODE_OPTS=\"-Dhadoop.security.logger=INFO,RFAS -Dhdfs.audit.logger=INFO,NullAppender\"").append("\n").append("export HADOOP_SECONDARYNAMENODE_OPTS=\"-Dhadoop.security.logger=INFO,RFAS -Dhdfs.audit.logger=INFO,NullAppender\"").append("\n").append("export HADOOP_DATANODE_OPTS=\"-Dhadoop.security.logger=ERROR,RFAS -Dhdfs.audit.logger=INFO,NullAppender\"").append("\n").append("export HADOOP_CLIENT_OPTS=\"-Xmx128m\"").append("\n").append("export HADOOP_IDENT_STRING=").append(System.getProperty("user.name")).append("\n").append("for f in $HADOOP_HOME/contrib/capacity-scheduler/*.jar; do\n").append("  if [ \"$HADOOP_CLASSPATH\" ]; then\n").append("    export HADOOP_CLASSPATH=$HADOOP_CLASSPATH:$f\n").append("  else\n").append("    export HADOOP_CLASSPATH=$f\n").append("  fi\n").append("done\n").append("export HADOOP_CLASSPATH=").append(hd.getTestDir()).append(":$HADOOP_CLASSPATH").append("\n");
        if (hdd.isSecure()) {
            sb.append("export HADOOP_SECURE_DN_USER=").append(System.getProperty("user.name")).append("\n").append("export HADOOP_SECURE_DN_LOG_DIR=").append(nd.getLogDir()).append("\n").append("export HADOOP_SECURE_DN_PID_DIR=").append(nd.getPIDDir()).append("\n").append("export JSVC_HOME=/export/gcm/where/java/hadoop/hadoop-secure-utils/bigtop-utils").append("\n");
        }
        String env = sb.toString();
        String fn = nd.getConfDir() + "/hadoop-env.sh";
        FileUtil.writeToFile(fn, env);
        Log.getLogWriter().info("Generated hadoop-env.sh for " + nd.getClusterName() + " in " + fn + ":\n" + env);
    }

    private static void generateYarnEnv(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        Log.getLogWriter().info("Generating yarn-env for " + nd.getName());
        HostDescription hd = nd.getHostDescription();
        StringBuilder opts = new StringBuilder();
        opts.append(" -Dfs.defaultFS=").append(hdd.getNameNodeURL());
        opts.append(" -Dyarn.resourcemanager.address=").append(hdd.getResourceManagerURL());
        opts.append(" -Dyarn.resourcemanager.resource-tracker.address=").append(hdd.getResourceTrackerAddress());
        opts.append(" -Dyarn.resourcemanager.scheduler.address=").append(hdd.getSchedulerAddress());
        StringBuilder sb = new StringBuilder();
        sb.append("export JAVA_HOME=").append(hd.getJavaHome()).append("\n").append("export JSVC_HOME=/export/gcm/where/java/hadoop/hadoop-secure-utils/bigtop-utils").append("\n").append("export HADOOP_CONF_DIR=").append(nd.getConfDir()).append("\n").append("export YARN_USER=yarn").append("\n").append("export YARN_CONF_DIR=").append(nd.getConfDir()).append("\n").append("export YARN_LOG_DIR=").append(nd.getLogDir()).append("\n").append("export YARN_IDENT_STRING=").append(System.getProperty("user.name")).append("\n").append("export YARN_PID_DIR=").append(nd.getPIDDir()).append("\n").append("export YARN_OPTS=\"").append(opts.toString().trim()).append("\"").append("\n");
        String env = sb.toString();
        String fn = nd.getConfDir() + "/yarn-env.sh";
        FileUtil.writeToFile(fn, env);
        Log.getLogWriter().info("Generated yarn-env.sh for " + nd.getClusterName() + " in " + fn + ":\n" + env);
    }

    private static void generateCoreSiteXMLFile(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String infn;
        String confDir = nd.getConfDir();
        HostDescription hd = nd.getHostDescription();
        String security = hdd.getSecurityAuthentication();
        if (security.equals("simple")) {
            infn = "$JTESTS/hydra/hadoop/conf/core-site.xml";
        } else if (security.equals("kerberos")) {
            infn = "$JTESTS/hydra/hadoop/conf/core-site-secure.xml";
        } else if (security.equals("kerberosKinit")) {
            infn = "$JTESTS/hydra/hadoop/conf/core-site-secure-kinit.xml";
        } else {
            String s = "Unsupported security: " + security;
            throw new UnsupportedOperationException(s);
        }
        String xml = HadoopHelper.readFile(infn, hd);
        xml = xml.replace("${fs.defaultFS}", hdd.getNameNodeURL());
        if (hdd.isSecure()) {
            xml = xml.replace("${securityAuthentication}", "kerberos");
            xml = xml.replace("${securityAuthorization}", "true");
        } else {
            xml = xml.replace("${securityAuthorization}", "false");
        }
        String outfn = confDir + "/core-site.xml";
        Log.getLogWriter().info("Writing core-site.xml file for " + nd.getName() + " to " + outfn);
        FileUtil.writeToFile(outfn, xml);
        Log.getLogWriter().info("Wrote core-site.xml file for " + nd.getName() + " to " + outfn);
    }

    private static void generateHDFSSiteXMLFile(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String infn;
        String confDir = nd.getConfDir();
        HostDescription hd = nd.getHostDescription();
        String security = hdd.getSecurityAuthentication();
        if (security.equals("simple")) {
            infn = "$JTESTS/hydra/hadoop/conf/hdfs-site.xml";
        } else if (security.equals("kerberos")) {
            infn = "$JTESTS/hydra/hadoop/conf/hdfs-site-secure.xml";
        } else if (security.equals("kerberosKinit")) {
            infn = "$JTESTS/hydra/hadoop/conf/hdfs-site-secure-kinit.xml";
        } else {
            String s = "Unsupported security: " + security;
            throw new UnsupportedOperationException(s);
        }
        String xml = HadoopHelper.readFile(infn, hd);
        xml = xml.replace("${dfs.replication}", String.valueOf(hdd.getReplication()));
        xml = xml.replace("${dn.ipc.port}", HadoopHelper.getRandomPort());
        xml = xml.replace("${nn.http.port}", HadoopHelper.getRandomPort());
        xml = xml.replace("${nn.secondary.http.port}", HadoopHelper.getRandomPort());
        if (hdd.isSecure()) {
            xml = xml.replace("${dn.port}", "1004");
            xml = xml.replace("${dn.http.port}", "1006");
            xml = xml.replace("${hdfs.secure.keytab.file}", HDFS_SECURE_KEYTAB_FILE);
        } else {
            xml = xml.replace("${dn.port}", HadoopHelper.getRandomPort());
            xml = xml.replace("${dn.http.port}", HadoopHelper.getRandomPort());
        }
        String outfn = confDir + "/hdfs-site.xml";
        Log.getLogWriter().info("Writing hdfs-site.xml file for " + nd.getName() + " to " + outfn);
        FileUtil.writeToFile(outfn, xml);
        Log.getLogWriter().info("Wrote hdfs-site.xml file for " + nd.getName() + " to " + outfn);
    }

    private static void generateMapRedSiteXMLFile(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String confDir = nd.getConfDir();
        HostDescription hd = nd.getHostDescription();
        String infn = hdd.isSecure() ? "$JTESTS/hydra/hadoop/conf/mapred-site-secure.xml" : "$JTESTS/hydra/hadoop/conf/mapred-site.xml";
        String xml = HadoopHelper.readFile(infn, hd);
        xml = xml.replace("${yarn.resourcemanager.address}", hdd.getResourceManagerURL());
        xml = xml.replace("${mapreduce.shuffle.port}", HadoopHelper.getRandomPort());
        if (hdd.isSecure()) {
            xml = xml.replace("${mapred.secure.keytab.file}", MAPRED_SECURE_KEYTAB_FILE);
        }
        String outfn = confDir + "/mapred-site.xml";
        Log.getLogWriter().info("Writing mapred-site.xml file for " + nd.getName() + " to " + outfn);
        FileUtil.writeToFile(outfn, xml);
        Log.getLogWriter().info("Wrote mapred-site.xml file for " + nd.getName() + " to " + outfn);
    }

    private static void generateContainerExecutorCFGFile(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String infn = "$JTESTS/hydra/hadoop/conf/container-executor-secure.cfg";
        String xml = HadoopHelper.readFile(infn, nd.getHostDescription());
        xml = xml.replace("${yarn.nodemanager.local-dirs}", nd.getLogDir());
        xml = xml.replace("${yarn.nodemanager.log-dirs}", nd.getLogDir());
        String hadoopDist = hdd.getHadoopDist();
        String src = System.getProperty("user.dir") + "/container-executor.cfg";
        String dst = hadoopDist + "/etc/hadoop/container-executor.cfg";
        String host = nd.getHostName();
        String user = System.getProperty("user.name");
        Log.getLogWriter().info("Writing container-executor.cfg file for " + nd.getName() + " to " + src);
        FileUtil.writeToFile(src, xml);
        Log.getLogWriter().info("Wrote container-executor.cfg file for " + nd.getName() + " to " + src);
        HadoopHelper.copyFile(src, dst, host);
        HadoopHelper.setReadPermission(dst, host);
    }

    private static void generateYarnSiteXMLFile(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String confDir = nd.getConfDir();
        HostDescription hd = nd.getHostDescription();
        String infn = hdd.isSecure() ? "$JTESTS/hydra/hadoop/conf/yarn-site-secure.xml" : "$JTESTS/hydra/hadoop/conf/yarn-site.xml";
        String xml = HadoopHelper.readFile(infn, hd);
        xml = xml.replace("${yarn.resourcemanager.address}", hdd.getResourceManagerURL());
        xml = xml.replace("${yarn.resourcemanager.resource-tracker.address}", hdd.getResourceTrackerAddress());
        xml = xml.replace("${yarn.resourcemanager.scheduler.address}", hdd.getSchedulerAddress());
        xml = xml.replace("${yarn.nodemanager.local-dirs}", nd.getLogDir());
        xml = xml.replace("${yarn.nodemanager.log-dirs}", nd.getLogDir());
        if (hdd.isSecure()) {
            xml = xml.replace("${yarn.secure.keytab.file}", YARN_SECURE_KEYTAB_FILE);
        }
        String outfn = confDir + "/yarn-site.xml";
        Log.getLogWriter().info("Writing yarn-site.xml file for " + nd.getName() + " to " + outfn);
        FileUtil.writeToFile(outfn, xml);
        Log.getLogWriter().info("Wrote yarn-site.xml file for " + nd.getName() + " to " + outfn);
    }

    private static void writeSlavesFile(HadoopDescription.NodeDescription nd, String slaves) {
        String confDir = nd.getConfDir();
        String fn = confDir + "/slaves";
        Log.getLogWriter().info("Writing slaves file for " + nd.getName() + " to " + fn);
        FileUtil.writeToFile(fn, slaves);
        Log.getLogWriter().info("Wrote slaves file for " + nd.getName() + " to " + fn);
    }

    private static void copyMiscFiles(HadoopDescription.NodeDescription nd) {
        String confDir = nd.getConfDir();
        Log.getLogWriter().info("Copying miscellaneous files for " + nd.getName() + " to " + confDir);
        String src = TestConfig.getInstance().getMasterDescription().getVmDescription().getHostDescription().getTestDir() + "/hydra/hadoop/conf/";
        String dst = confDir + "/";
        for (String fn : ConfigurationFiles) {
            Log.getLogWriter().info("Copying " + fn + " to " + confDir);
            FileUtil.copyFile(src + fn, dst + fn);
        }
        Log.getLogWriter().info("Copied miscellaneous files for " + nd.getName() + " to " + confDir);
    }

    public static synchronized void startCluster(String hadoopConfig) {
        HadoopHelper.formatNameNodes(hadoopConfig);
        HadoopHelper.startNameNodes(hadoopConfig);
        HadoopHelper.startDataNodes(hadoopConfig);
        HadoopHelper.startResourceManager(hadoopConfig);
        HadoopHelper.startNodeManagers(hadoopConfig);
    }

    public static synchronized void stopCluster(String hadoopConfig) {
        HadoopHelper.stopNodeManagers(hadoopConfig);
        HadoopHelper.stopResourceManager(hadoopConfig);
        HadoopHelper.stopDataNodes(hadoopConfig);
        HadoopHelper.stopNameNodes(hadoopConfig);
    }

    public static synchronized void startHDFSCluster(String hadoopConfig) {
        HadoopHelper.formatNameNodes(hadoopConfig);
        HadoopHelper.startNameNodes(hadoopConfig);
        HadoopHelper.startDataNodes(hadoopConfig);
    }

    public static synchronized void stopHDFSCluster(String hadoopConfig) {
        HadoopHelper.stopDataNodes(hadoopConfig);
        HadoopHelper.stopNameNodes(hadoopConfig);
    }

    public static synchronized void formatNameNodes(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NameNodeDescription> nnds = hdd.getNameNodeDescriptions();
        Log.getLogWriter().info("Formatting name nodes: " + nnds);
        for (HadoopDescription.NodeDescription nodeDescription : nnds) {
            HadoopHelper.formatNameNode(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Formatted name nodes: " + nnds);
    }

    public static synchronized void formatNameNode(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NameNodeDescription> nnds = hdd.getNameNodeDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : nnds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.formatNameNode(hdd, nodeDescription);
            return;
        }
        String s = "No name node found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    private static synchronized void formatNameNode(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String cmd = "/bin/bash " + hdd.getHadoopDist() + "/bin/hdfs" + " --config " + nd.getConfDir() + " namenode -format " + nd.getClusterName();
        String host = nd.getHostName();
        Log.getLogWriter().info("Formatting name node on host " + host + " using command: " + cmd);
        HostDescription hd = nd.getHostDescription();
        String logfn = nd.getLogDir() + "/namenode-format.log";
        int pid = ProcessMgr.bgexec(host, cmd, hd.getUserDir(), logfn);
        HadoopHelper.recordPID("formatNameNode", hd, pid, false);
        int maxWaitSec = 180;
        if (!ProcessMgr.waitForDeath(host, pid, maxWaitSec)) {
            String s = "Waited more than " + maxWaitSec + " seconds for NameNode to be formatted";
            throw new HydraTimeoutException(s);
        }
        HadoopHelper.removePID("formatNameNode", hd, pid, false);
        Log.getLogWriter().info("Formatted name node on host " + host + " using command: " + cmd + ", see " + logfn + " for output");
    }

    public static synchronized void startNameNodes(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NameNodeDescription> nnds = hdd.getNameNodeDescriptions();
        Log.getLogWriter().info("Starting name nodes: " + nnds);
        for (HadoopDescription.NodeDescription nodeDescription : nnds) {
            HadoopHelper.startNameNode(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Started name nodes: " + nnds);
    }

    public static synchronized void startNameNode(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NameNodeDescription> nnds = hdd.getNameNodeDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : nnds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.startNameNode(hdd, nodeDescription);
            return;
        }
        String s = "No name node found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    public static synchronized void startNameNode(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String cmd = "/bin/bash " + hdd.getHadoopDist() + HADOOP_DAEMON + " --config " + nd.getConfDir() + " --script hdfs start namenode";
        String host = nd.getHostName();
        Log.getLogWriter().info("Starting name node on host " + host + " using command: " + cmd);
        String result = null;
        try {
            result = HadoopHelper.executeUserCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to start NameNode on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        String pidfn = nd.getPIDDir() + "/hadoop-" + System.getProperty("user.name") + "-namenode.pid";
        Integer pid = HadoopHelper.readPID("startNameNode", pidfn);
        HadoopHelper.recordHadoop("startNameNode", hdd, nd, pid, false);
        if (!ProcessMgr.processExists(host, pid)) {
            String s = "Failed to start name node on host " + host + " using command: " + cmd + ", see " + nd.getLogDir() + " for output";
            throw new HydraRuntimeException(s);
        }
        Log.getLogWriter().info("Started name node on host " + host + " using command: " + cmd + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void stopNameNodes(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NameNodeDescription> nnds = hdd.getNameNodeDescriptions();
        Log.getLogWriter().info("Stopping name nodes: " + nnds);
        for (HadoopDescription.NodeDescription nodeDescription : nnds) {
            HadoopHelper.stopNameNode(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Stopped name nodes: " + nnds);
    }

    public static synchronized void stopNameNode(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NameNodeDescription> nnds = hdd.getNameNodeDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : nnds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.stopNameNode(hdd, nodeDescription);
            return;
        }
        String s = "No name node found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    private static synchronized void stopNameNode(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String cmd = "/bin/bash " + hdd.getHadoopDist() + HADOOP_DAEMON + " --config " + nd.getConfDir() + " --script hdfs stop namenode";
        String host = nd.getHostName();
        Log.getLogWriter().info("Stopping name node on host " + host + " using command: " + cmd);
        String result = null;
        try {
            result = HadoopHelper.executeUserCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to stop NameNode on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        String pidfn = nd.getPIDDir() + "/hadoop-" + System.getProperty("user.name") + "-namenode.pid";
        Integer pid = HadoopHelper.readPID("stopNameNode", pidfn);
        HadoopHelper.removeHadoop("stopNameNode", hdd, nd, pid, false);
        Log.getLogWriter().info("Stopped name node on host " + host + " using command: " + cmd + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void startDataNodes(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.DataNodeDescription> dnds = hdd.getDataNodeDescriptions();
        Log.getLogWriter().info("Starting data nodes: " + dnds);
        for (HadoopDescription.NodeDescription nodeDescription : dnds) {
            HadoopHelper.startDataNode(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Started data nodes: " + dnds);
    }

    public static synchronized void startDataNode(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.DataNodeDescription> dnds = hdd.getDataNodeDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : dnds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.startDataNode(hdd, nodeDescription);
            return;
        }
        String s = "No data node found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    public static synchronized void startDataNode(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        HostDescription hd = nd.getHostDescription();
        String host = nd.getHostName();
        Log.getLogWriter().info("Starting data node on host " + host);
        String result = null;
        String masterHostName = TestConfig.getInstance().getMasterDescription().getVmDescription().getHostDescription().getHostName();
        try {
            String cmd;
            if (hdd.isSecure()) {
                HadoopHelper.setWritePermission(nd.getPIDDir(), masterHostName);
                Log.getLogWriter().info("Using secure mode (kerberos) as root");
                cmd = "startSecureDataNode " + hdd.getHadoopDist() + " " + nd.getConfDir();
                result = HadoopHelper.executeRootCommand(host, cmd, 120);
            } else {
                cmd = "/bin/bash " + hdd.getHadoopDist() + HADOOP_DAEMON + " --config " + nd.getConfDir() + " --script hdfs start datanode";
                result = HadoopHelper.executeUserCommand(host, cmd, 120);
            }
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to start data node on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        if (hdd.isSecure()) {
            HadoopHelper.setOwnership(nd.getPIDDir(), masterHostName, System.getProperty("user.name"));
            HadoopHelper.setOwnership(nd.getLogDir(), host, System.getProperty("user.name"));
            HadoopHelper.setReadPermission(nd.getPIDDir(), masterHostName);
            HadoopHelper.setReadPermission(nd.getLogDir(), host);
            String spidfn = nd.getPIDDir() + "/hadoop_secure_dn.pid";
            Integer spid = HadoopHelper.readPID("startDataNode", spidfn);
            Log.getLogWriter().info("DataNode secure PID=" + spid);
            HadoopHelper.recordHadoop("startDataNode", hdd, nd, spid, true);
            boolean exists = HadoopHelper.processExists(host, spid);
            if (!exists) {
                String s = "Failed to start DataNode on host " + host + ", see " + nd.getLogDir();
                throw new HydraRuntimeException(s);
            }
            if (hdd.getSecurityAuthentication().equals("kerberosKinit")) {
                String cmd1 = "/export/gcm/where/java/hadoop/hadoop-secure-keytabs/hdfs-secure.keytab hdfs/" + hd.getCanonicalHostName() + "@GEMSTONE.COM";
                HadoopHelper.executeKinitCommand(host, cmd1, 120);
            }
            String cmd3 = "fs -chown gfxd-secure:users /";
            String outputFile = nd.getLogDir() + "/secure_chown.log";
            HadoopHelper.executeHadoopCommand(host, hdd, cmd3, outputFile);
            if (hdd.getSecurityAuthentication().equals("kerberosKinit")) {
                String cmd2 = "/export/gcm/where/java/hadoop/hadoop-secure-keytabs/gfxd-secure.keytab gfxd-secure@GEMSTONE.COM";
                HadoopHelper.executeKinitCommand(host, cmd2, 120);
            }
        } else {
            String pidfn = nd.getPIDDir() + "/hadoop-" + System.getProperty("user.name") + "-datanode.pid";
            Integer pid = HadoopHelper.readPID("startDataNode", pidfn);
            Log.getLogWriter().info("DataNode PID=" + pid);
            HadoopHelper.recordHadoop("startDataNode", hdd, nd, pid, false);
            boolean exists = ProcessMgr.processExists(host, pid);
            if (!exists) {
                String s = "Failed to start DataNode on host " + host + ", see " + nd.getLogDir();
                throw new HydraRuntimeException(s);
            }
        }
        Log.getLogWriter().info("Started data node on host " + host + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void stopDataNodes(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.DataNodeDescription> dnds = hdd.getDataNodeDescriptions();
        Log.getLogWriter().info("Stopping data nodes: " + dnds);
        for (HadoopDescription.NodeDescription nodeDescription : dnds) {
            HadoopHelper.stopDataNode(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Stopped data nodes: " + dnds);
    }

    public static synchronized void stopDataNode(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.DataNodeDescription> dnds = hdd.getDataNodeDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : dnds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.stopDataNode(hdd, nodeDescription);
            return;
        }
        String s = "No data node found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    private static synchronized void stopDataNode(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        HostDescription hd = nd.getHostDescription();
        String host = nd.getHostName();
        Log.getLogWriter().info("Stopping data node on host " + host);
        Integer spid = 0;
        Integer pid = 0;
        String result = null;
        try {
            if (hdd.isSecure()) {
                Log.getLogWriter().info("Using secure mode (kerberos) as root");
                String spidfn = nd.getPIDDir() + "/hadoop_secure_dn.pid";
                File pidFile = new File(spidfn);
                if (pidFile.exists()) {
                    spid = HadoopHelper.readPID("stopDataNode", spidfn);
                }
                Log.getLogWriter().info("DataNode secure PID=" + spid);
                String cmd = "stopSecureDataNode " + hdd.getHadoopDist() + " " + nd.getConfDir();
                result = HadoopHelper.executeRootCommand(host, cmd, 120);
            } else {
                String pidfn = nd.getPIDDir() + "/hadoop-" + System.getProperty("user.name") + "-datanode.pid";
                File pidFile = new File(pidfn);
                if (pidFile.exists()) {
                    pid = HadoopHelper.readPID("stopDataNode", pidfn);
                }
                String cmd = "/bin/bash " + hdd.getHadoopDist() + HADOOP_DAEMON + " --config " + nd.getConfDir() + " --script hdfs stop datanode";
                result = HadoopHelper.executeUserCommand(host, cmd, 120);
            }
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to stop DataNode on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("DataNode PID=" + pid);
        if (hdd.isSecure() && spid > 0) {
            HadoopHelper.removeHadoop("stopDataNode", hdd, nd, spid, true);
        } else {
            HadoopHelper.removeHadoop("stopDataNode", hdd, nd, pid, false);
        }
        if (hdd.getSecurityAuthentication().equals("kerberosKinit")) {
            HadoopHelper.executeKdestroyCommand(host, 120);
        }
        Log.getLogWriter().info("Stopped data node on host " + host + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void startResourceManager(String hadoopConfig) {
        HadoopDescription hd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopDescription.ResourceManagerDescription rmd = hd.getResourceManagerDescription();
        Log.getLogWriter().info("Starting YARN resource manager: " + rmd);
        HadoopHelper.startResourceManager(hd, rmd);
        Log.getLogWriter().info("Started YARN resource manager: " + rmd);
    }

    public static synchronized void startResourceManager(String hadoopConfig, String host) {
        HadoopDescription hd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopDescription.ResourceManagerDescription rmd = hd.getResourceManagerDescription();
        if (rmd.getHostName().equals(host)) {
            HadoopHelper.startResourceManager(hd, rmd);
            return;
        }
        String s = "No resource manager found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    public static synchronized void startResourceManager(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String hadoopDist = hdd.getHadoopDist();
        String cmd = "/bin/bash " + hadoopDist + YARN_DAEMON + " --config " + nd.getConfDir() + " start resourcemanager";
        String host = nd.getHostName();
        Log.getLogWriter().info("Starting YARN resource manager on host " + host + " using command: " + cmd);
        HostDescription hd = nd.getHostDescription();
        String result = null;
        try {
            result = HadoopHelper.executeUserCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to start ResourceManager on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        String pidfn = nd.getPIDDir() + "/yarn-" + System.getProperty("user.name") + "-resourcemanager.pid";
        Integer pid = HadoopHelper.readPID("startResourceManager", pidfn);
        HadoopHelper.recordHadoop("startResourceManager", hdd, nd, pid, false);
        if (!ProcessMgr.processExists(host, pid)) {
            String s = "Failed to start resource manager on host " + host + " using command: " + cmd + ", see " + nd.getLogDir() + " for output";
            throw new HydraRuntimeException(s);
        }
        Log.getLogWriter().info("Started resource manager on host " + host + " using command: " + cmd + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void stopResourceManager(String hadoopConfig) {
        HadoopDescription hd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopDescription.ResourceManagerDescription rmd = hd.getResourceManagerDescription();
        Log.getLogWriter().info("Stopping resource manager: " + rmd);
        HadoopHelper.stopResourceManager(hd, rmd);
        Log.getLogWriter().info("Stopped resource manager: " + rmd);
    }

    public static synchronized void stopResourceManager(String hadoopConfig, String host) {
        HadoopDescription hd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopDescription.ResourceManagerDescription rmd = hd.getResourceManagerDescription();
        if (rmd.getHostName().equals(host)) {
            HadoopHelper.stopResourceManager(hd, rmd);
            return;
        }
        String s = "No resource manager found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    private static synchronized void stopResourceManager(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        String cmd = "/bin/bash " + hdd.getHadoopDist() + YARN_DAEMON + " --config " + nd.getConfDir() + " stop resourcemanager";
        String host = nd.getHostName();
        Log.getLogWriter().info("Stopping resource manager on host " + host + " using command: " + cmd);
        HostDescription hd = nd.getHostDescription();
        String pidfn = nd.getPIDDir() + "/yarn-" + System.getProperty("user.name") + "-resourcemanager.pid";
        Integer pid = HadoopHelper.readPID("stopResourceManager", pidfn);
        String result = null;
        try {
            result = HadoopHelper.executeUserCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to stop resource manager on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("ResourceManager pid=" + pid);
        HadoopHelper.removeHadoop("stopResourceManager", hdd, nd, pid, false);
        Log.getLogWriter().info("Stopped resource manager on host " + host + " using command: " + cmd + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void startNodeManagers(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NodeManagerDescription> nmds = hdd.getNodeManagerDescriptions();
        Log.getLogWriter().info("Starting node managers: " + nmds);
        for (HadoopDescription.NodeDescription nodeDescription : nmds) {
            HadoopHelper.startNodeManager(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Started node managers: " + nmds);
    }

    public static synchronized void startNodeManager(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NodeManagerDescription> nmds = hdd.getNodeManagerDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : nmds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.startNodeManager(hdd, nodeDescription);
            return;
        }
        String s = "No node manager node found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    public static synchronized void startNodeManager(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        HostDescription hd = nd.getHostDescription();
        String host = nd.getHostName();
        Log.getLogWriter().info("Starting node manager on host " + host);
        String result = null;
        try {
            String cmd;
            if (hdd.isSecure()) {
                Log.getLogWriter().info("Using secure mode (kerberos) as root");
                cmd = "/bin/bash " + hdd.getHadoopDist() + YARN_DAEMON + " --config " + nd.getConfDir() + " start nodemanager";
                result = HadoopHelper.executeUserCommand(host, cmd, 120);
                HadoopHelper.recordOwnershipCommand(hd, nd.getLogDir(), System.getProperty("user.name"));
                HadoopHelper.recordReadPermissionCommand(hd, nd.getLogDir());
            } else {
                cmd = "/bin/bash " + hdd.getHadoopDist() + YARN_DAEMON + " --config " + nd.getConfDir() + " start nodemanager";
                result = HadoopHelper.executeUserCommand(host, cmd, 120);
            }
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to start node manager on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        String pidfn = nd.getPIDDir() + "/yarn-" + System.getProperty("user.name") + "-nodemanager.pid";
        Integer pid = HadoopHelper.readPID("startNodeManager", pidfn);
        Log.getLogWriter().info("NodeManager PID=" + pid);
        HadoopHelper.recordHadoop("startNodeManager", hdd, nd, pid, false);
        boolean exists = ProcessMgr.processExists(host, pid);
        if (!exists) {
            String s = "Failed to start NodeManager on host " + host + ", see " + nd.getLogDir();
            throw new HydraRuntimeException(s);
        }
        Log.getLogWriter().info("Started node manager on host " + host + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void stopNodeManagers(String hadoopConfig) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NodeManagerDescription> nmds = hdd.getNodeManagerDescriptions();
        Log.getLogWriter().info("Stopping node managers: " + nmds);
        for (HadoopDescription.NodeDescription nodeDescription : nmds) {
            HadoopHelper.stopNodeManager(hdd, nodeDescription);
        }
        Log.getLogWriter().info("Stopped node managers: " + nmds);
    }

    public static synchronized void stopNodeManager(String hadoopConfig, String host) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        List<HadoopDescription.NodeManagerDescription> nmds = hdd.getNodeManagerDescriptions();
        for (HadoopDescription.NodeDescription nodeDescription : nmds) {
            if (!nodeDescription.getHostName().equals(host)) continue;
            HadoopHelper.stopNodeManager(hdd, nodeDescription);
            return;
        }
        String s = "No node manager found for host: " + host;
        throw new HydraRuntimeException(s);
    }

    private static synchronized void stopNodeManager(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        HostDescription hd = nd.getHostDescription();
        String host = nd.getHostName();
        Log.getLogWriter().info("Stopping node manager on host " + host);
        String pidfn = nd.getPIDDir() + "/yarn-" + System.getProperty("user.name") + "-nodemanager.pid";
        Integer pid = HadoopHelper.readPID("stopNodeManager", pidfn);
        String result = null;
        try {
            String cmd = "/bin/bash " + hdd.getHadoopDist() + YARN_DAEMON + " --config " + nd.getConfDir() + " stop nodemanager";
            result = HadoopHelper.executeUserCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to stop NodeManager on " + host + ", see " + nd.getLogDir() + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("NodeManager PID=" + pid);
        HadoopHelper.removeHadoop("stopNodeManager", hdd, nd, pid, false);
        Log.getLogWriter().info("Stopped data node on host " + host + ", see " + nd.getLogDir() + " for output");
    }

    public static synchronized void startHadoopProcesses(List<HadoopInfo> hadoops) {
        HadoopHelper.startNameNodes(hadoops);
        HadoopHelper.startDataNodes(hadoops);
        HadoopHelper.startResourceManagers(hadoops);
        HadoopHelper.startNodeManagers(hadoops);
    }

    public static synchronized void startNameNodes(List<HadoopInfo> hadoops) {
        for (HadoopInfo hadoop : hadoops) {
            if (hadoop.getNodeDescription().getNodeType() != HadoopPrms.NodeType.NameNode) continue;
            HadoopHelper.startNameNode(hadoop.getHadoopDescription(), hadoop.getNodeDescription());
        }
    }

    public static synchronized void startDataNodes(List<HadoopInfo> hadoops) {
        for (HadoopInfo hadoop : hadoops) {
            if (hadoop.getNodeDescription().getNodeType() != HadoopPrms.NodeType.DataNode) continue;
            HadoopHelper.startDataNode(hadoop.getHadoopDescription(), hadoop.getNodeDescription());
        }
    }

    public static synchronized void startResourceManagers(List<HadoopInfo> hadoops) {
        for (HadoopInfo hadoop : hadoops) {
            if (hadoop.getNodeDescription().getNodeType() != HadoopPrms.NodeType.ResourceManager) continue;
            HadoopHelper.startResourceManager(hadoop.getHadoopDescription(), hadoop.getNodeDescription());
        }
    }

    public static synchronized void startNodeManagers(List<HadoopInfo> hadoops) {
        for (HadoopInfo hadoop : hadoops) {
            if (hadoop.getNodeDescription().getNodeType() != HadoopPrms.NodeType.NodeManager) continue;
            HadoopHelper.startResourceManager(hadoop.getHadoopDescription(), hadoop.getNodeDescription());
        }
    }

    public static synchronized void startHadoopProcess(HadoopDescription hdd, HadoopDescription.NodeDescription nd) {
        switch (nd.getNodeType()) {
            case DataNode: {
                HadoopHelper.startDataNode(hdd, nd);
                break;
            }
            case NameNode: {
                HadoopHelper.startNameNode(hdd, nd);
                break;
            }
            case NodeManager: {
                HadoopHelper.startNodeManager(hdd, nd);
                break;
            }
            case ResourceManager: {
                HadoopHelper.startResourceManager(hdd, nd);
                break;
            }
            default: {
                String s = "Should not happen";
                throw new HydraInternalException(s);
            }
        }
    }

    public static synchronized void runHadoopCommand(String hadoopConfig, String command, String outfn) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopHelper.runHadoopCommand(hdd, command, outfn);
    }

    public static synchronized void runHadoopCommand(HadoopDescription hdd, String command, String outfn) {
        HadoopDescription.NodeDescription nd = hdd.getNameNodeDescriptions().get(0);
        String cmd = "/bin/bash " + hdd.getHadoopDist() + "/bin/hadoop --config " + nd.getConfDir() + " " + command;
        Log.getLogWriter().info("Running hadoop command: " + cmd);
        HostDescription hd = TestConfig.getInstance().getAnyPhysicalHostDescription(HostHelper.getLocalHost());
        int pid = ProcessMgr.bgexec(cmd, System.getProperty("user.dir"), outfn);
        HadoopHelper.recordPID("runHadoopCommand", hd, pid, false);
        int maxWaitSec = 180;
        if (!ProcessMgr.waitForDeath(hd.getHostName(), pid, maxWaitSec)) {
            String s = "Waited more than " + maxWaitSec + " seconds for " + cmd + " to be executed";
            throw new HydraTimeoutException(s);
        }
        HadoopHelper.removePID("runHadoopCommand", hd, pid, false);
        Log.getLogWriter().info("Ran hadoop command: " + cmd + ", see " + outfn);
    }

    public static synchronized void executeHadoopCommand(String host, String hadoopConfig, String command, String outfn) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopHelper.executeHadoopCommand(host, hdd, command, outfn);
    }

    public static synchronized void executeHadoopCommand(String host, HadoopDescription hdd, String command, String outfn) {
        HadoopDescription.NodeDescription nd = hdd.getNameNodeDescriptions().get(0);
        String cmd = "/bin/bash " + hdd.getHadoopDist() + "/bin/hadoop --config " + nd.getConfDir() + " " + command;
        Log.getLogWriter().info("Executing hadoop command: " + cmd + " on host " + host);
        HostDescription hd = TestConfig.getInstance().getAnyPhysicalHostDescription(host);
        int pid = ProcessMgr.bgexec(host, cmd, System.getProperty("user.dir"), outfn);
        HadoopHelper.recordPID("runHadoopCommand", hd, pid, false);
        int maxWaitSec = 180;
        if (!ProcessMgr.waitForDeath(hd.getHostName(), pid, maxWaitSec)) {
            String s = "Waited more than " + maxWaitSec + " seconds for " + cmd + " to be executed on host " + host;
            throw new HydraTimeoutException(s);
        }
        HadoopHelper.removePID("runHadoopCommand", hd, pid, false);
        Log.getLogWriter().info("Executed hadoop command: " + cmd + " on host " + host + ", see " + outfn);
    }

    public static synchronized void runYarnCommand(String hadoopConfig, String command, String outfn) {
        HadoopDescription hdd = HadoopHelper.getHadoopDescription(hadoopConfig);
        HadoopDescription.ResourceManagerDescription rmd = hdd.getResourceManagerDescription();
        String cmd = "env CLASSPATH=" + System.getProperty("java.class.path") + " " + hdd.getHadoopDist() + "/bin/yarn" + " --config " + rmd.getConfDir() + " " + command;
        Log.getLogWriter().info("Running yarn command: " + cmd);
        HostDescription hd = TestConfig.getInstance().getAnyPhysicalHostDescription(HostHelper.getLocalHost());
        int pid = ProcessMgr.bgexec(cmd, System.getProperty("user.dir"), outfn);
        HadoopHelper.recordPID("runYarnCommand", hd, pid, false);
        int maxWaitSec = 180;
        if (!ProcessMgr.waitForDeath(hd.getHostName(), pid, maxWaitSec)) {
            String s = "Waited more than " + maxWaitSec + " seconds for yarn " + cmd + " to be executed";
            throw new HydraTimeoutException(s);
        }
        HadoopHelper.removePID("runYarnCommand", hd, pid, false);
        Log.getLogWriter().info("Ran yarn command: " + cmd + ", see " + outfn);
    }

    public static HadoopDescription getHadoopDescription(String hadoopConfig) {
        return TestConfig.getInstance().getHadoopDescription(hadoopConfig);
    }

    private static String readFile(String fn, HostDescription hd) {
        try {
            return FileUtil.getText(EnvHelper.expandEnvVars(fn, hd));
        }
        catch (FileNotFoundException e) {
            String s = fn + " not found";
            throw new HydraRuntimeException(s, e);
        }
        catch (IOException e) {
            String s = "Problem reading: " + fn;
            throw new HydraRuntimeException(s, e);
        }
    }

    private static String getRandomPort() {
        return String.valueOf(PortHelper.getRandomPort());
    }

    private static Integer readPID(String caller, String pidfn) {
        if (Log.getLogWriter().fineEnabled()) {
            Log.getLogWriter().fine(caller + " reading PID from: " + pidfn);
        }
        Integer pid = null;
        try {
            pid = new Integer(FileUtil.getContents(pidfn));
        }
        catch (NumberFormatException e) {
            String s = "Cannot read PID from file: " + pidfn;
            throw new HydraRuntimeException(s, e);
        }
        catch (IOException e) {
            String s = "Cannot read PID file: " + pidfn;
            throw new HydraRuntimeException(s, e);
        }
        return pid;
    }

    private static void recordHadoop(String caller, HadoopDescription hdd, HadoopDescription.NodeDescription nd, Integer pid, boolean secure) {
        try {
            if (Log.getLogWriter().fineEnabled()) {
                Log.getLogWriter().fine(caller + " recording PID: " + pid + " on host " + nd.getHostDescription().getHostName());
            }
            RemoteTestModule.Master.recordHadoop(hdd, nd, pid, secure);
        }
        catch (RemoteException e) {
            String s = "Failed to record PID: " + pid;
            throw new HydraRuntimeException(s, e);
        }
    }

    private static void removeHadoop(String caller, HadoopDescription hdd, HadoopDescription.NodeDescription nd, Integer pid, boolean secure) {
        try {
            if (Log.getLogWriter().fineEnabled()) {
                Log.getLogWriter().fine(caller + " removing PID: " + pid + " on host " + nd.getHostDescription().getHostName());
            }
            RemoteTestModule.Master.removeHadoop(hdd, nd, pid, secure);
        }
        catch (RemoteException e) {
            String s = "Failed to remove PID: " + pid;
            throw new HydraRuntimeException(s, e);
        }
    }

    private static void recordPID(String caller, HostDescription hd, Integer pid, boolean secure) {
        try {
            if (Log.getLogWriter().fineEnabled()) {
                Log.getLogWriter().fine(caller + " recording PID: " + pid + " on host " + hd.getHostName());
            }
            RemoteTestModule.Master.recordHDFSPIDNoDumps(hd, pid, secure);
        }
        catch (RemoteException e) {
            String s = "Failed to record PID: " + pid;
            throw new HydraRuntimeException(s, e);
        }
    }

    private static void removePID(String caller, HostDescription hd, Integer pid, boolean secure) {
        try {
            if (Log.getLogWriter().fineEnabled()) {
                Log.getLogWriter().fine(caller + " removing PID: " + pid + " on host " + hd.getHostName());
            }
            RemoteTestModule.Master.removeHDFSPIDNoDumps(hd, pid, secure);
        }
        catch (RemoteException e) {
            String s = "Failed to remove PID: " + pid;
            throw new HydraRuntimeException(s, e);
        }
    }

    private static boolean processExists(String host, int pid) {
        String command = "processExists " + pid;
        String cmd = null;
        String result = null;
        try {
            cmd = HadoopHelper.getRootCommand(command);
            result = ProcessMgr.fgexec(host, cmd, 120);
            int exitCode = Integer.parseInt(result);
            return exitCode == 0;
        }
        catch (NumberFormatException e) {
            String s = "Unable to parse exit code for command: " + cmd + " from script output: \"" + result + "\"";
            throw new HydraRuntimeException(s, e);
        }
    }

    private static String executeRootCommand(String host, String command, int maxWaitSec) {
        String cmd = HadoopHelper.getRootCommand(command);
        return ProcessMgr.fgexec(host, cmd, maxWaitSec);
    }

    private static String executeUserCommand(String host, String command, int maxWaitSec) {
        return ProcessMgr.fgexec(host, command, maxWaitSec);
    }

    public static void executeKinitCommand(String host, String command, int maxWaitSec) {
        String cmd = HadoopHelper.kinit() + " " + command;
        String result = null;
        try {
            result = HadoopHelper.executeUserCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to run " + cmd + " on " + host + ":" + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
    }

    public static void executeKdestroyCommand(String host, int maxWaitSec) {
        block2: {
            String cmd = HadoopHelper.kdestroy();
            String result = null;
            try {
                result = HadoopHelper.executeUserCommand(host, cmd, 120);
            }
            catch (HydraRuntimeException e) {
                if (e.getMessage().startsWith("kdestroy: No credentials cache found")) break block2;
                String s = "Failed to run " + cmd + " on " + host + ":" + result;
                throw new HydraRuntimeException(s, (Exception)((Object)e));
            }
        }
    }

    private static void executeHadoopCommand(String host, String command, int maxWaitSec) {
        String result = null;
        try {
            result = HadoopHelper.executeUserCommand(host, command, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to run " + command + " on " + host + ":" + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
    }

    private static void setOwnership(String fn, String host, String user) {
        Log.getLogWriter().info("Setting ownership of " + fn + " to " + user + " on " + host);
        String cmd = "setOwnership " + user + " " + fn;
        String result = null;
        try {
            result = HadoopHelper.executeRootCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to set ownership of " + fn + " to " + user + " on " + host + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("Set ownership of " + fn + " to " + user + " on " + host);
    }

    private static void setReadPermission(String fn, String host) {
        Log.getLogWriter().info("Setting read permission for " + fn + " on " + host);
        String cmd = "setReadPermission " + fn;
        String result = null;
        try {
            result = HadoopHelper.executeRootCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to set read permission for " + fn + " on " + host + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("Set read permission for " + fn + " on " + host);
    }

    private static void setWritePermission(String fn, String host) {
        Log.getLogWriter().info("Setting write permission for " + fn + " on " + host);
        String cmd = "setWritePermission " + fn;
        String result = null;
        try {
            result = HadoopHelper.executeRootCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to set write permission for " + fn + " on " + host + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("Set write permission for " + fn + " on " + host);
    }

    private static void recordOwnershipCommand(HostDescription hd, String dir, String user) {
        HadoopHelper.recordRootCommand(hd, "setOwnership " + user + " " + dir, dir);
    }

    private static void recordReadPermissionCommand(HostDescription hd, String dir) {
        HadoopHelper.recordRootCommand(hd, "setReadPermission " + dir, dir);
    }

    private static void recordRootCommand(HostDescription hd, String cmd, String dir) {
        String rootCmd = HadoopHelper.getRootCommand(cmd);
        try {
            Log.getLogWriter().info("Recording root command on host " + hd.getHostName() + ": " + rootCmd);
            RemoteTestModule.Master.recordRootCommand(hd, rootCmd, dir);
            Log.getLogWriter().info("Recorded root command on host " + hd.getHostName() + ": " + rootCmd);
        }
        catch (RemoteException e) {
            String s = "Failed to record root command on host " + hd.getHostName() + ": " + rootCmd;
            throw new HydraRuntimeException(s, e);
        }
    }

    private static void copyFile(String src, String dst, String host) {
        Log.getLogWriter().info("Copying " + src + " to " + dst + " on " + host);
        String cmd = "copyFile " + src + " " + dst;
        String result = null;
        try {
            result = HadoopHelper.executeRootCommand(host, cmd, 120);
        }
        catch (HydraRuntimeException e) {
            String s = "Failed to copy " + src + " to " + dst + " on " + host + ": " + result;
            throw new HydraRuntimeException(s, (Exception)((Object)e));
        }
        Log.getLogWriter().info("Copied " + src + " to " + dst + " on " + host);
    }

    private static String getRootCommand(String command) {
        return HadoopHelper.sudo() + " " + HadoopHelper.rootScript() + " " + command;
    }

    private static String rootScript() {
        if (!FileUtil.fileExists(ROOT_SCRIPT)) {
            String s = "Script needed to manage root processes and files not found: /export/localnew/scripts/commands_for_secure_hdfs.sh";
            throw new HydraRuntimeException(s);
        }
        return ROOT_SCRIPT;
    }

    private static String kinit() {
        if (!FileUtil.fileExists(KINIT_CMD)) {
            String s = "Unable to find kinit: /usr/bin/kinit";
            throw new HydraRuntimeException(s);
        }
        return KINIT;
    }

    private static String kdestroy() {
        if (!FileUtil.fileExists(KDESTROY_CMD)) {
            String s = "Unable to find kdestroy: /usr/bin/kdestroy";
            throw new HydraRuntimeException(s);
        }
        return KDESTROY;
    }

    private static String sudo() {
        if (!FileUtil.exists(SUDO)) {
            String s = "Unable to find sudo: /usr/bin/sudo";
            throw new HydraRuntimeException(s);
        }
        return SUDO;
    }
}

