/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.cats.redis.cache;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.hash.Hashing;
import com.netflix.spinnaker.cats.cache.CacheData;
import com.netflix.spinnaker.cats.cache.DefaultCacheData;
import com.netflix.spinnaker.cats.redis.cache.AbstractRedisCache;
import com.netflix.spinnaker.cats.redis.cache.RedisCacheOptions;
import com.netflix.spinnaker.kork.jedis.RedisClientDelegate;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;

public class RedisCache
extends AbstractRedisCache {
    private final CacheMetrics cacheMetrics;

    public RedisCache(String prefix, RedisClientDelegate redisClientDelegate, ObjectMapper objectMapper, RedisCacheOptions options, CacheMetrics cacheMetrics) {
        super(prefix, redisClientDelegate, objectMapper, options);
        this.cacheMetrics = cacheMetrics == null ? new CacheMetrics.NOOP() : cacheMetrics;
    }

    @Override
    protected void mergeItems(String type, Collection<CacheData> items) {
        if (items.isEmpty()) {
            return;
        }
        HashSet<String> relationshipNames = new HashSet<String>();
        LinkedList<String> keysToSet = new LinkedList<String>();
        HashSet<String> idSet = new HashSet<String>();
        HashMap<String, Integer> ttlSecondsByKey = new HashMap<String, Integer>();
        int skippedWrites = 0;
        Map<String, String> hashes = this.getHashes(type, items);
        TreeMap<String, String> updatedHashes = new TreeMap<String, String>();
        for (CacheData item : items) {
            MergeOp op = this.buildMergeOp(type, item, hashes);
            relationshipNames.addAll(op.relNames);
            keysToSet.addAll(op.keysToSet);
            idSet.add(item.getId());
            updatedHashes.putAll(op.hashesToSet);
            skippedWrites += op.skippedWrites;
            if (item.getTtlSeconds() <= 0) continue;
            for (String key : op.keysToSet) {
                ttlSecondsByKey.put(key, item.getTtlSeconds());
            }
        }
        AtomicInteger saddOperations = new AtomicInteger();
        AtomicInteger msetOperations = new AtomicInteger();
        AtomicInteger hmsetOperations = new AtomicInteger();
        AtomicInteger pipelineOperations = new AtomicInteger();
        AtomicInteger expireOperations = new AtomicInteger();
        if (keysToSet.size() > 0) {
            this.redisClientDelegate.withMultiKeyPipeline(pipeline -> {
                for (List idPart : Iterables.partition((Iterable)idSet, (int)this.options.getMaxSaddSize())) {
                    String[] ids = idPart.toArray(new String[idPart.size()]);
                    pipeline.sadd(this.allOfTypeId(type), ids);
                    saddOperations.incrementAndGet();
                }
                for (List keys : Lists.partition((List)keysToSet, (int)this.options.getMaxMsetSize())) {
                    pipeline.mset(keys.toArray(new String[keys.size()]));
                    msetOperations.incrementAndGet();
                }
                if (!relationshipNames.isEmpty()) {
                    for (List relNamesPart : Iterables.partition((Iterable)relationshipNames, (int)this.options.getMaxSaddSize())) {
                        pipeline.sadd(this.allRelationshipsId(type), relNamesPart.toArray(new String[relNamesPart.size()]));
                        saddOperations.incrementAndGet();
                    }
                }
                if (!updatedHashes.isEmpty()) {
                    for (List hashPart : Iterables.partition(updatedHashes.keySet(), (int)this.options.getMaxHmsetSize())) {
                        pipeline.hmset(this.hashesId(type), updatedHashes.subMap((String)hashPart.get(0), true, (String)hashPart.get(hashPart.size() - 1), true));
                        hmsetOperations.incrementAndGet();
                    }
                }
                pipeline.sync();
                pipelineOperations.incrementAndGet();
            });
            this.redisClientDelegate.withMultiKeyPipeline(pipeline -> {
                for (List ttlPart : Iterables.partition(ttlSecondsByKey.entrySet(), (int)this.options.getMaxPipelineSize())) {
                    for (Map.Entry ttlEntry : ttlPart) {
                        pipeline.expire((String)ttlEntry.getKey(), ((Integer)ttlEntry.getValue()).intValue());
                    }
                    expireOperations.addAndGet(ttlPart.size());
                    pipeline.sync();
                    pipelineOperations.incrementAndGet();
                }
            });
        }
        this.cacheMetrics.merge(this.prefix, type, items.size(), keysToSet.size() / 2, relationshipNames.size(), skippedWrites, updatedHashes.size(), saddOperations.get(), msetOperations.get(), hmsetOperations.get(), pipelineOperations.get(), expireOperations.get());
    }

    @Override
    protected void evictItems(String type, List<String> identifiers, Collection<String> allRelationships) {
        ArrayList<String> delKeys = new ArrayList<String>((allRelationships.size() + 1) * identifiers.size());
        for (String id : identifiers) {
            for (String relationship : allRelationships) {
                delKeys.add(this.relationshipId(type, id, relationship));
            }
            delKeys.add(this.attributesId(type, id));
        }
        AtomicInteger delOperations = new AtomicInteger();
        AtomicInteger hdelOperations = new AtomicInteger();
        AtomicInteger sremOperations = new AtomicInteger();
        this.redisClientDelegate.withMultiKeyPipeline(pipeline -> {
            for (List delPartition : Lists.partition((List)delKeys, (int)this.options.getMaxDelSize())) {
                pipeline.del(delPartition.toArray(new String[delPartition.size()]));
                delOperations.incrementAndGet();
                pipeline.hdel(this.hashesId(type), delPartition.toArray(new String[delPartition.size()]));
                hdelOperations.incrementAndGet();
            }
            for (List idPartition : Lists.partition((List)identifiers, (int)this.options.getMaxDelSize())) {
                String[] ids = idPartition.toArray(new String[idPartition.size()]);
                pipeline.srem(this.allOfTypeId(type), ids);
                sremOperations.incrementAndGet();
            }
            pipeline.sync();
        });
        this.cacheMetrics.evict(this.prefix, type, identifiers.size(), delKeys.size(), delKeys.size(), delOperations.get(), hdelOperations.get(), sremOperations.get());
    }

    @Override
    protected Collection<CacheData> getItems(String type, List<String> ids, List<String> knownRels) {
        int singleResultSize = knownRels.size() + 1;
        ArrayList<String> keysToGet = new ArrayList<String>(singleResultSize * ids.size());
        for (String id : ids) {
            keysToGet.add(this.attributesId(type, id));
            for (String rel : knownRels) {
                keysToGet.add(this.relationshipId(type, id, rel));
            }
        }
        ArrayList keyResult = new ArrayList(keysToGet.size());
        int mgetOperations = (Integer)this.redisClientDelegate.withMultiClient(c -> {
            int ops = 0;
            for (List part : Lists.partition((List)keysToGet, (int)this.options.getMaxMgetSize())) {
                ++ops;
                keyResult.addAll(c.mget(part.toArray(new String[part.size()])));
            }
            return ops;
        });
        if (keyResult.size() != keysToGet.size()) {
            throw new RuntimeException("Expected same size result as request");
        }
        ArrayList<CacheData> results = new ArrayList<CacheData>(ids.size());
        Iterator<String> idIterator = ids.iterator();
        for (int ofs = 0; ofs < keyResult.size(); ofs += singleResultSize) {
            CacheData item = this.extractItem(idIterator.next(), keyResult.subList(ofs, ofs + singleResultSize), knownRels);
            if (item == null) continue;
            results.add(item);
        }
        this.cacheMetrics.get(this.prefix, type, results.size(), ids.size(), keysToGet.size(), knownRels.size(), mgetOperations);
        return results;
    }

    private CacheData extractItem(String id, List<String> keyResult, List<String> knownRels) {
        if (keyResult.get(0) == null) {
            return null;
        }
        try {
            Map attributes = (Map)this.objectMapper.readValue(keyResult.get(0), ATTRIBUTES);
            HashMap<String, Collection> relationships = new HashMap<String, Collection>(keyResult.size() - 1);
            for (int relIdx = 1; relIdx < keyResult.size(); ++relIdx) {
                String rel = keyResult.get(relIdx);
                if (rel == null) continue;
                String relType = knownRels.get(relIdx - 1);
                Collection deserializedRel = (Collection)this.objectMapper.readValue(rel, this.getRelationshipsTypeReference());
                relationships.put(relType, deserializedRel);
            }
            return new DefaultCacheData(id, attributes, relationships);
        }
        catch (IOException deserializationException) {
            throw new RuntimeException("Deserialization failed", deserializationException);
        }
    }

    private MergeOp buildMergeOp(String type, CacheData cacheData, Map<String, String> hashes) {
        String serializedAttributes;
        int skippedWrites = 0;
        boolean hasTtl = cacheData.getTtlSeconds() > 0;
        try {
            serializedAttributes = cacheData.getAttributes().isEmpty() ? null : this.objectMapper.writeValueAsString((Object)cacheData.getAttributes());
        }
        catch (JsonProcessingException serializationException) {
            throw new RuntimeException("Attribute serialization failed", serializationException);
        }
        HashMap<String, String> hashesToSet = new HashMap<String, String>();
        ArrayList<String> keysToSet = new ArrayList<String>((cacheData.getRelationships().size() + 1) * 2);
        if (serializedAttributes != null && this.hashCheck(hashes, this.attributesId(type, cacheData.getId()), serializedAttributes, keysToSet, hashesToSet, hasTtl)) {
            ++skippedWrites;
        }
        if (!cacheData.getRelationships().isEmpty()) {
            for (Map.Entry relationship : cacheData.getRelationships().entrySet()) {
                String relationshipValue;
                try {
                    relationshipValue = this.objectMapper.writeValueAsString(new LinkedHashSet((Collection)relationship.getValue()));
                }
                catch (JsonProcessingException serializationException) {
                    throw new RuntimeException("Relationship serialization failed", serializationException);
                }
                if (!this.hashCheck(hashes, this.relationshipId(type, cacheData.getId(), (String)relationship.getKey()), relationshipValue, keysToSet, hashesToSet, hasTtl)) continue;
                ++skippedWrites;
            }
        }
        return new MergeOp(cacheData.getRelationships().keySet(), keysToSet, hashesToSet, skippedWrites);
    }

    private List<String> getKeys(String type, Collection<CacheData> cacheDatas) {
        HashSet<String> keys = new HashSet<String>();
        for (CacheData cacheData : cacheDatas) {
            if (!cacheData.getAttributes().isEmpty()) {
                keys.add(this.attributesId(type, cacheData.getId()));
            }
            if (cacheData.getRelationships().isEmpty()) continue;
            for (String relationship : cacheData.getRelationships().keySet()) {
                keys.add(this.relationshipId(type, cacheData.getId(), relationship));
            }
        }
        return new ArrayList<String>(keys);
    }

    private List<String> getHashValues(List<String> hashKeys, String hashesId) {
        ArrayList<String> hashValues = new ArrayList<String>(hashKeys.size());
        this.redisClientDelegate.withCommandsClient(c -> {
            for (List hashPart : Lists.partition((List)hashKeys, (int)this.options.getMaxHmgetSize())) {
                hashValues.addAll(c.hmget(hashesId, (String[])Arrays.copyOf(hashPart.toArray(), hashPart.size(), String[].class)));
            }
        });
        return hashValues;
    }

    private boolean hashCheck(Map<String, String> hashes, String id, String serializedValue, List<String> keys, Map<String, String> updatedHashes, boolean hasTtl) {
        if (this.options.isHashingEnabled() && !hasTtl) {
            String existingHash;
            String hash = Hashing.sha1().newHasher().putUnencodedChars((CharSequence)serializedValue).hash().toString();
            if (hash.equals(existingHash = hashes.get(id))) {
                return true;
            }
            updatedHashes.put(id, hash);
        }
        keys.add(id);
        keys.add(serializedValue);
        return false;
    }

    private Map<String, String> getHashes(String type, Collection<CacheData> items) {
        if (this.isHashingDisabled(type)) {
            return Collections.emptyMap();
        }
        List<String> hashKeys = this.getKeys(type, items);
        if (hashKeys.isEmpty()) {
            return Collections.emptyMap();
        }
        List<String> hashValues = this.getHashValues(hashKeys, this.hashesId(type));
        if (hashValues.size() != hashKeys.size()) {
            throw new RuntimeException("Expected same size result as request");
        }
        HashMap<String, String> hashes = new HashMap<String, String>(hashKeys.size());
        for (int i = 0; i < hashValues.size(); ++i) {
            String hashValue = hashValues.get(i);
            if (hashValue == null) continue;
            hashes.put(hashKeys.get(i), hashValue);
        }
        return hashes;
    }

    private String hashesId(String type) {
        return String.format("%s:%s:hashes", this.prefix, type);
    }

    private static class MergeOp {
        public final Set<String> relNames;
        public final List<String> keysToSet;
        public final Map<String, String> hashesToSet;
        public final int skippedWrites;

        MergeOp(Set<String> relNames, List<String> keysToSet, Map<String, String> hashesToSet, int skippedWrites) {
            this.relNames = relNames;
            this.keysToSet = keysToSet;
            this.hashesToSet = hashesToSet;
            this.skippedWrites = skippedWrites;
        }
    }

    public static interface CacheMetrics {
        default public void merge(String prefix, String type, int itemCount, int keysWritten, int relationshipCount, int hashMatches, int hashUpdates, int saddOperations, int msetOperations, int hmsetOperations, int pipelineOperations, int expireOperations) {
        }

        default public void evict(String prefix, String type, int itemCount, int keysDeleted, int hashesDeleted, int delOperations, int hdelOperations, int sremOperations) {
        }

        default public void get(String prefix, String type, int itemCount, int requestedSize, int keysRequested, int relationshipsRequested, int mgetOperations) {
        }

        public static class NOOP
        implements CacheMetrics {
        }
    }
}

