/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.artifacts.gitRepo;

import com.netflix.spinnaker.clouddriver.artifacts.gitRepo.GitRepoArtifactAccount;
import com.netflix.spinnaker.clouddriver.jobs.JobExecutor;
import com.netflix.spinnaker.clouddriver.jobs.JobRequest;
import com.netflix.spinnaker.clouddriver.jobs.JobResult;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import lombok.Generated;
import org.apache.commons.io.FileUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

public class GitJobExecutor {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(GitJobExecutor.class);
    private static final String SSH_KEY_PWD_ENV_VAR = "SSH_KEY_PWD";
    private static final Pattern FULL_SHA_PATTERN = Pattern.compile("[0-9a-f]{40}");
    private static final Pattern SHORT_SHA_PATTERN = Pattern.compile("[0-9a-f]{7}");
    private static Path genericAskPassBinary;
    private final GitRepoArtifactAccount account;
    private final JobExecutor jobExecutor;
    private final String gitExecutable;
    private final AuthType authType;
    private final Path askPassBinary;

    public GitJobExecutor(GitRepoArtifactAccount account, JobExecutor jobExecutor, String gitExecutable) throws IOException {
        this.account = account;
        this.jobExecutor = jobExecutor;
        this.gitExecutable = gitExecutable;
        this.authType = !StringUtils.isEmpty((Object)account.getUsername()) && !StringUtils.isEmpty((Object)account.getPassword()) ? AuthType.USER_PASS : (!StringUtils.isEmpty((Object)account.getUsername()) && account.getTokenAsString().filter(t -> !StringUtils.isEmpty((Object)t)).isPresent() ? AuthType.USER_TOKEN : (account.getTokenAsString().filter(t -> !StringUtils.isEmpty((Object)t)).isPresent() ? AuthType.TOKEN : (!StringUtils.isEmpty((Object)account.getSshPrivateKeyFilePath()) ? AuthType.SSH : AuthType.NONE)));
        this.askPassBinary = this.initAskPass();
    }

    public void cloneOrPull(String repoUrl, String branch, Path localPath, String repoBasename) throws IOException {
        File localPathFile = localPath.toFile();
        if (!localPathFile.exists()) {
            this.clone(repoUrl, branch, localPath, repoBasename);
            return;
        }
        if (!localPathFile.isDirectory()) {
            throw new IllegalArgumentException("Local path " + localPath.toString() + " is not a directory");
        }
        File[] localPathFiles = localPathFile.listFiles();
        if (localPathFiles == null || localPathFiles.length == 0) {
            this.clone(repoUrl, branch, localPath, repoBasename);
            return;
        }
        Path dotGitPath = Paths.get(localPath.toString(), repoBasename, ".git");
        if (!dotGitPath.toFile().exists()) {
            log.warn("Directory {} for git/repo {}, branch {} has files or directories but {} was not found. The directory will be recreated to start with a new clone.", new Object[]{localPath.toString(), repoUrl, branch, dotGitPath.toString()});
            this.clone(repoUrl, branch, localPath, repoBasename);
            return;
        }
        this.pull(repoUrl, branch, dotGitPath.getParent());
    }

    private void clone(String repoUrl, String branch, Path destination, String repoBasename) throws IOException {
        if (!this.isValidReference(repoUrl)) {
            throw new IllegalArgumentException("Git reference \"" + repoUrl + "\" is invalid for credentials with auth type " + String.valueOf((Object)this.authType));
        }
        File destinationFile = destination.toFile();
        if (destinationFile.exists()) {
            FileUtils.deleteDirectory((File)destinationFile);
        }
        FileUtils.forceMkdir((File)destinationFile);
        if (FULL_SHA_PATTERN.matcher(branch).matches()) {
            this.fetchFullSha(repoUrl, branch, destination, repoBasename);
        } else {
            this.cloneBranchOrTag(repoUrl, branch, destination, repoBasename);
        }
    }

    private void cloneBranchOrTag(String repoUrl, String branch, Path destination, String repoBasename) throws IOException {
        log.info("Cloning git/repo {} into {}", (Object)repoUrl, (Object)destination.toString());
        String command = this.gitExecutable + " clone --branch " + branch + " --depth 1 " + this.repoUrlWithAuth(repoUrl);
        JobResult<String> result = new CommandChain(destination).addCommand(command).runAll();
        if (result.getResult() == JobResult.Result.SUCCESS) {
            return;
        }
        String errorMsg = command + " failed. Error: " + result.getError() + " Output: " + (String)result.getOutput();
        if (!SHORT_SHA_PATTERN.matcher(branch).matches()) {
            throw new IOException(errorMsg);
        }
        log.warn(errorMsg + ". Trying a full clone and checkout " + branch);
        File destFile = destination.toFile();
        FileUtils.deleteDirectory((File)destFile);
        FileUtils.forceMkdir((File)destFile);
        this.cloneAndCheckoutSha(repoUrl, branch, destination, repoBasename);
    }

    private void fetchFullSha(String repoUrl, String sha, Path destination, String repoBasename) throws IOException {
        log.info("Fetching git/repo {} sha {} into {}", new Object[]{repoUrl, sha, destination.toString()});
        Path repoPath = Paths.get(destination.toString(), repoBasename);
        if (!repoPath.toFile().mkdirs()) {
            throw new IOException("Unable to create directory " + repoPath.toString());
        }
        JobResult<String> result = new CommandChain(repoPath).addCommand(this.gitExecutable + " init").addCommand(this.gitExecutable + " remote add origin " + this.repoUrlWithAuth(repoUrl)).addCommand(this.gitExecutable + " fetch --depth 1 origin " + sha).addCommand(this.gitExecutable + " reset --hard FETCH_HEAD").runAll();
        if (result.getResult() == JobResult.Result.SUCCESS) {
            return;
        }
        log.warn("Unable to directly fetch specific sha, trying full clone. Error: " + result.getError());
        FileUtils.forceDelete((File)repoPath.toFile());
        this.cloneAndCheckoutSha(repoUrl, sha, destination, repoBasename);
    }

    private void cloneAndCheckoutSha(String repoUrl, String sha, Path destination, String repoBasename) throws IOException {
        Path repoPath = Paths.get(destination.toString(), repoBasename);
        new CommandChain(destination).addCommand(this.gitExecutable + " clone " + this.repoUrlWithAuth(repoUrl)).runAllOrFail();
        new CommandChain(repoPath).addCommand(this.gitExecutable + " checkout " + sha).runAllOrFail();
    }

    private void pull(String repoUrl, String branch, Path localPath) throws IOException {
        if (FULL_SHA_PATTERN.matcher(branch).matches()) {
            log.info("Contents of git/repo {} for sha {} already downloaded, no \"git pull\" needed.", (Object)repoUrl, (Object)branch);
            return;
        }
        JobResult<String> result = new CommandChain(localPath).addCommand(this.gitExecutable + " symbolic-ref HEAD").runAll();
        if (result.getResult() != JobResult.Result.SUCCESS) {
            log.info("git/repo {} is in detached HEAD state for version {}, skipping \"git pull\"", (Object)repoUrl, (Object)branch);
            return;
        }
        log.info("Pulling git/repo {} into {}", (Object)repoUrl, (Object)localPath.toString());
        new CommandChain(localPath).addCommand(this.gitExecutable + " pull").runAllOrFail();
        if (!localPath.getParent().toFile().setLastModified(System.currentTimeMillis())) {
            log.warn("Unable to set last modified time on {}", (Object)localPath.getParent().toString());
        }
    }

    public void archive(Path localClone, String branch, String subDir, Path outputFile) throws IOException {
        String cmd = this.gitExecutable + " archive --format tgz --output " + outputFile.toString() + " " + branch;
        if (!StringUtils.isEmpty((Object)subDir)) {
            cmd = cmd + " " + subDir;
        }
        new CommandChain(localClone).addCommand(cmd).runAllOrFail();
    }

    private Path initAskPass() throws IOException {
        if (this.authType != AuthType.SSH) {
            return null;
        }
        if (!StringUtils.isEmpty((Object)this.account.getSshPrivateKeyPassphraseCmd())) {
            File pwdCmd = new File(this.account.getSshPrivateKeyPassphraseCmd());
            if (!pwdCmd.exists() || !pwdCmd.isFile()) {
                throw new IOException("SshPrivateKeyPassphraseCmd doesn't exist or is not a file: " + this.account.getSshPrivateKeyPassphraseCmd());
            }
            return Paths.get(this.account.getSshPrivateKeyPassphraseCmd(), new String[0]);
        }
        if (genericAskPassBinary == null) {
            File askpass = File.createTempFile("askpass", null);
            if (!askpass.setExecutable(true)) {
                throw new IOException("Unable to make executable askpass script at " + askpass.toPath().toString());
            }
            FileUtils.writeStringToFile((File)askpass, (String)"#!/bin/sh\necho \"$SSH_KEY_PWD\"", (Charset)Charset.defaultCharset());
            genericAskPassBinary = askpass.toPath();
        }
        return genericAskPassBinary;
    }

    private boolean isValidReference(String reference) {
        if (this.authType == AuthType.USER_PASS || this.authType == AuthType.USER_TOKEN || this.authType == AuthType.TOKEN) {
            return reference.startsWith("http");
        }
        if (this.authType == AuthType.SSH) {
            return reference.startsWith("ssh://") || reference.startsWith("git@");
        }
        return true;
    }

    private List<String> cmdToList(String cmd) {
        ArrayList<String> cmdList = new ArrayList<String>();
        switch (this.authType) {
            case USER_PASS: 
            case USER_TOKEN: 
            case TOKEN: {
                cmdList.add("sh");
                cmdList.add("-c");
                cmdList.add(cmd);
                break;
            }
            default: {
                cmdList.addAll(Arrays.asList(cmd.split(" ")));
            }
        }
        return cmdList;
    }

    private String repoUrlWithAuth(String repoUrl) {
        if (this.authType != AuthType.USER_PASS && this.authType != AuthType.USER_TOKEN && this.authType != AuthType.TOKEN) {
            return repoUrl;
        }
        String authPart = this.authType == AuthType.USER_PASS ? "$GIT_USER:$GIT_PASS" : (this.authType == AuthType.USER_TOKEN ? "$GIT_USER:$GIT_TOKEN" : "token:$GIT_TOKEN");
        try {
            URI uri = new URI(repoUrl);
            return String.format("%s://%s@%s%s%s", uri.getScheme(), authPart, uri.getHost(), uri.getPort() > 0 ? ":" + uri.getPort() : "", uri.getRawPath());
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException("Malformed git repo url " + repoUrl, e);
        }
    }

    private Map<String, String> addEnvVars(Map<String, String> env) {
        HashMap<String, String> result = new HashMap<String, String>(env);
        switch (this.authType) {
            case USER_PASS: {
                result.put("GIT_USER", GitJobExecutor.encodeURIComponent(this.account.getUsername()));
                result.put("GIT_PASS", GitJobExecutor.encodeURIComponent(this.account.getPassword()));
                break;
            }
            case USER_TOKEN: {
                result.put("GIT_USER", GitJobExecutor.encodeURIComponent(this.account.getUsername()));
                result.put("GIT_TOKEN", GitJobExecutor.encodeURIComponent((String)this.account.getTokenAsString().orElseThrow(() -> new IllegalArgumentException("Token or TokenFile must be present if using token auth."))));
                break;
            }
            case TOKEN: {
                result.put("GIT_TOKEN", GitJobExecutor.encodeURIComponent((String)this.account.getTokenAsString().orElseThrow(() -> new IllegalArgumentException("Token or TokenFile must be present if using token auth."))));
                break;
            }
            case SSH: {
                result.put("GIT_SSH_COMMAND", this.buildSshCommand());
                result.put("SSH_ASKPASS", this.askPassBinary.toString());
                result.put("DISPLAY", ":0");
                if (StringUtils.isEmpty((Object)this.account.getSshPrivateKeyPassphrase())) break;
                result.put(SSH_KEY_PWD_ENV_VAR, this.account.getSshPrivateKeyPassphrase());
            }
        }
        if (log.isDebugEnabled()) {
            result.put("GIT_CURL_VERBOSE", "1");
            result.put("GIT_TRACE", "1");
        }
        return result;
    }

    @NotNull
    private String buildSshCommand() {
        Object gitSshCmd = "setsid ssh";
        if (this.account.isSshTrustUnknownHosts()) {
            gitSshCmd = (String)gitSshCmd + " -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no";
        } else if (!StringUtils.isEmpty((Object)this.account.getSshKnownHostsFilePath())) {
            gitSshCmd = (String)gitSshCmd + " -o UserKnownHostsFile=" + this.account.getSshKnownHostsFilePath();
        }
        if (!StringUtils.isEmpty((Object)this.account.getSshPrivateKeyFilePath())) {
            gitSshCmd = (String)gitSshCmd + " -i " + this.account.getSshPrivateKeyFilePath();
        }
        return gitSshCmd;
    }

    private static String encodeURIComponent(String s) {
        if (StringUtils.isEmpty((Object)s)) {
            return s;
        }
        String result = URLEncoder.encode(s, StandardCharsets.UTF_8).replaceAll("\\+", "%20").replaceAll("\\*", "%2A").replaceAll("%21", "!").replaceAll("%27", "'").replaceAll("%28", "(").replaceAll("%29", ")").replaceAll("%7E", "~");
        return result;
    }

    @Generated
    public GitRepoArtifactAccount getAccount() {
        return this.account;
    }

    private static enum AuthType {
        USER_PASS,
        USER_TOKEN,
        TOKEN,
        SSH,
        NONE;

    }

    private class CommandChain {
        private final Collection<JobRequest> commands = new ArrayList<JobRequest>();
        private final Path workingDir;

        CommandChain(Path workingDir) {
            this.workingDir = workingDir;
        }

        CommandChain addCommand(String command) {
            this.commands.add(new JobRequest(GitJobExecutor.this.cmdToList(command), GitJobExecutor.this.addEnvVars(System.getenv()), this.workingDir.toFile()));
            return this;
        }

        void runAllOrFail() throws IOException {
            for (JobRequest command : this.commands) {
                log.debug("Executing command: \"{}\"", (Object)String.join((CharSequence)" ", command.getTokenizedCommand()));
                JobResult result = GitJobExecutor.this.jobExecutor.runJob(command);
                if (result.getResult() == JobResult.Result.SUCCESS) continue;
                throw new IOException(String.format("%s failed. Error: %s Output: %s", command.getTokenizedCommand(), result.getError(), result.getOutput()));
            }
        }

        JobResult<String> runAll() {
            JobResult result = null;
            for (JobRequest command : this.commands) {
                log.debug("Executing command: \"{}\"", (Object)String.join((CharSequence)" ", command.getTokenizedCommand()));
                result = GitJobExecutor.this.jobExecutor.runJob(command);
                if (result.getResult() == JobResult.Result.SUCCESS) continue;
                break;
            }
            return result;
        }
    }
}

