/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.aws.lifecycle;

import com.amazonaws.auth.policy.Action;
import com.amazonaws.auth.policy.Condition;
import com.amazonaws.auth.policy.Policy;
import com.amazonaws.auth.policy.Principal;
import com.amazonaws.auth.policy.Resource;
import com.amazonaws.auth.policy.Statement;
import com.amazonaws.auth.policy.actions.SNSActions;
import com.amazonaws.auth.policy.actions.SQSActions;
import com.amazonaws.services.sns.AmazonSNS;
import com.amazonaws.services.sns.model.SetTopicAttributesRequest;
import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.model.Message;
import com.amazonaws.services.sqs.model.ReceiptHandleIsInvalidException;
import com.amazonaws.services.sqs.model.ReceiveMessageRequest;
import com.amazonaws.services.sqs.model.ReceiveMessageResult;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.netflix.spinnaker.cats.agent.RunnableAgent;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.ARN;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.LaunchFailureConfigurationProperties;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.NotificationMessage;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.NotificationMessageWrapper;
import com.netflix.spinnaker.clouddriver.aws.provider.AwsProvider;
import com.netflix.spinnaker.clouddriver.aws.security.AmazonClientProvider;
import com.netflix.spinnaker.clouddriver.aws.security.NetflixAmazonCredentials;
import com.netflix.spinnaker.clouddriver.cache.CustomScheduledAgent;
import com.netflix.spinnaker.clouddriver.security.AccountCredentials;
import com.netflix.spinnaker.clouddriver.tags.EntityTagger;
import com.netflix.spinnaker.credentials.CredentialsRepository;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LaunchFailureNotificationAgent
implements RunnableAgent,
CustomScheduledAgent {
    private static final Logger log = LoggerFactory.getLogger(LaunchFailureNotificationAgent.class);
    private static final String SUPPORTED_LIFECYCLE_TRANSITION = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    private static final int AWS_MAX_NUMBER_OF_MESSAGES = 10;
    private final ObjectMapper objectMapper;
    private final AmazonClientProvider amazonClientProvider;
    private final CredentialsRepository<NetflixAmazonCredentials> credentialsRepository;
    private final LaunchFailureConfigurationProperties properties;
    private final EntityTagger serverGroupTagger;
    private final ARN topicARN;
    private final ARN queueARN;
    private String topicId = null;
    private String queueId = null;

    LaunchFailureNotificationAgent(ObjectMapper objectMapper, AmazonClientProvider amazonClientProvider, NetflixAmazonCredentials netflixAmazonCredentials, CredentialsRepository<NetflixAmazonCredentials> credentialsRepository, LaunchFailureConfigurationProperties properties, EntityTagger serverGroupTagger) {
        this.objectMapper = objectMapper;
        this.amazonClientProvider = amazonClientProvider;
        this.credentialsRepository = credentialsRepository;
        this.properties = properties;
        this.serverGroupTagger = serverGroupTagger;
        this.topicARN = new ARN(netflixAmazonCredentials, properties.getTopicARN());
        this.queueARN = new ARN(netflixAmazonCredentials, properties.getQueueARN());
    }

    public String getAgentType() {
        return this.queueARN.account.getName() + "/" + this.queueARN.region + "/" + LaunchFailureNotificationAgent.class.getSimpleName();
    }

    public String getProviderName() {
        return AwsProvider.PROVIDER_NAME;
    }

    public long getPollIntervalMillis() {
        return TimeUnit.MINUTES.toMillis(1L);
    }

    public long getTimeoutMillis() {
        return -1L;
    }

    public void run() {
        List<String> allAccountIds = this.credentialsRepository.getAll().stream().map(AccountCredentials::getAccountId).collect(Collectors.toList());
        AmazonSQS amazonSQS = this.amazonClientProvider.getAmazonSQS(this.queueARN.account, this.queueARN.region);
        this.queueId = LaunchFailureNotificationAgent.ensureQueueExists(amazonSQS, this.queueARN, this.topicARN);
        AmazonSNS amazonSNS = this.amazonClientProvider.getAmazonSNS(this.topicARN.account, this.topicARN.region);
        this.topicId = LaunchFailureNotificationAgent.ensureTopicExists(amazonSNS, this.topicARN, allAccountIds, this.queueARN);
        AtomicInteger messagesProcessed = new AtomicInteger(0);
        while (messagesProcessed.get() < this.properties.getMaxMessagesPerCycle()) {
            ReceiveMessageResult receiveMessageResult = amazonSQS.receiveMessage(new ReceiveMessageRequest(this.queueId).withMaxNumberOfMessages(Integer.valueOf(10)).withVisibilityTimeout(Integer.valueOf(this.properties.getVisibilityTimeout())).withWaitTimeSeconds(Integer.valueOf(this.properties.getWaitTimeSeconds())));
            receiveMessageResult.getMessages().forEach(message -> {
                try {
                    NotificationMessageWrapper notificationMessageWrapper = (NotificationMessageWrapper)this.objectMapper.readValue(message.getBody(), NotificationMessageWrapper.class);
                    NotificationMessage notificationMessage = (NotificationMessage)this.objectMapper.readValue(notificationMessageWrapper.message, NotificationMessage.class);
                    if (SUPPORTED_LIFECYCLE_TRANSITION.equalsIgnoreCase(notificationMessage.event)) {
                        LaunchFailureNotificationAgent.handleMessage(this.serverGroupTagger, notificationMessage);
                    }
                }
                catch (IOException e) {
                    log.error("Unable to convert NotificationMessage (body: {})", (Object)message.getBody(), (Object)e);
                }
                LaunchFailureNotificationAgent.deleteMessage(amazonSQS, this.queueId, message);
                messagesProcessed.incrementAndGet();
            });
            if (!receiveMessageResult.getMessages().isEmpty()) continue;
            break;
        }
        log.info("Processed {} messages (queueARN: {})", (Object)messagesProcessed.get(), (Object)this.queueARN.arn);
    }

    private static void handleMessage(EntityTagger serverGroupTagger, NotificationMessage notificationMessage) {
        log.info("Failed to launch instance (asgName: {}, reason: {})", (Object)notificationMessage.autoScalingGroupName, (Object)notificationMessage.statusMessage);
        Matcher sqsMatcher = ARN.PATTERN.matcher(notificationMessage.autoScalingGroupARN);
        if (!sqsMatcher.matches()) {
            throw new IllegalArgumentException(notificationMessage.autoScalingGroupARN + " is not a valid ARN");
        }
        String region = sqsMatcher.group(1);
        String accountId = sqsMatcher.group(2);
        serverGroupTagger.alert("aws", accountId, region, null, "servergroup", notificationMessage.autoScalingGroupName, notificationMessage.event, notificationMessage.statusMessage, null);
    }

    private static String ensureTopicExists(AmazonSNS amazonSNS, ARN topicARN, List<String> allAccountIds, ARN queueARN) {
        topicARN.arn = amazonSNS.createTopic(topicARN.name).getTopicArn();
        amazonSNS.setTopicAttributes(new SetTopicAttributesRequest().withTopicArn(topicARN.arn).withAttributeName("Policy").withAttributeValue(LaunchFailureNotificationAgent.buildSNSPolicy(topicARN, allAccountIds).toJson()));
        amazonSNS.subscribe(topicARN.arn, "sqs", queueARN.arn);
        return topicARN.arn;
    }

    private static String ensureQueueExists(AmazonSQS amazonSQS, ARN queueARN, ARN topicARN) {
        String queueUrl;
        try {
            queueUrl = amazonSQS.getQueueUrl(queueARN.name).getQueueUrl();
        }
        catch (Exception e) {
            queueUrl = amazonSQS.createQueue(queueARN.name).getQueueUrl();
        }
        amazonSQS.setQueueAttributes(queueUrl, Collections.singletonMap("Policy", LaunchFailureNotificationAgent.buildSQSPolicy(queueARN, topicARN).toJson()));
        return queueUrl;
    }

    private static Policy buildSNSPolicy(ARN topicARN, List<String> allAccountIds) {
        Statement statement = new Statement(Statement.Effect.Allow).withActions(new Action[]{SNSActions.Publish});
        statement.setPrincipals((Collection)allAccountIds.stream().map(Principal::new).collect(Collectors.toList()));
        statement.setResources(Collections.singletonList(new Resource(topicARN.arn)));
        return new Policy("allow-remote-account-send", Collections.singletonList(statement));
    }

    private static Policy buildSQSPolicy(ARN queue, ARN topic) {
        Statement statement = new Statement(Statement.Effect.Allow).withActions(new Action[]{SQSActions.SendMessage});
        statement.setPrincipals(new Principal[]{Principal.All});
        statement.setResources(Collections.singletonList(new Resource(queue.arn)));
        statement.setConditions(Collections.singletonList(new Condition().withType("ArnEquals").withConditionKey("aws:SourceArn").withValues(new String[]{topic.arn})));
        return new Policy("allow-sns-topic-send", Collections.singletonList(statement));
    }

    private static void deleteMessage(AmazonSQS amazonSQS, String queueUrl, Message message) {
        try {
            amazonSQS.deleteMessage(queueUrl, message.getReceiptHandle());
        }
        catch (ReceiptHandleIsInvalidException e) {
            log.warn("Error deleting lifecycle message, reason: {} (receiptHandle: {})", (Object)e.getMessage(), (Object)message.getReceiptHandle());
        }
    }
}

