/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.aws.lifecycle;

import com.amazonaws.auth.policy.Action;
import com.amazonaws.auth.policy.Condition;
import com.amazonaws.auth.policy.Policy;
import com.amazonaws.auth.policy.Principal;
import com.amazonaws.auth.policy.Resource;
import com.amazonaws.auth.policy.Statement;
import com.amazonaws.auth.policy.actions.SNSActions;
import com.amazonaws.auth.policy.actions.SQSActions;
import com.amazonaws.services.sns.AmazonSNS;
import com.amazonaws.services.sns.model.SetTopicAttributesRequest;
import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.model.Message;
import com.amazonaws.services.sqs.model.ReceiptHandleIsInvalidException;
import com.amazonaws.services.sqs.model.ReceiveMessageRequest;
import com.amazonaws.services.sqs.model.ReceiveMessageResult;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.netflix.frigga.Names;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import com.netflix.spinnaker.clouddriver.aws.deploy.ops.discovery.AwsEurekaSupport;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.ARN;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.InstanceTerminationConfigurationProperties;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.LifecycleMessage;
import com.netflix.spinnaker.clouddriver.aws.lifecycle.NotificationMessageWrapper;
import com.netflix.spinnaker.clouddriver.aws.security.AmazonClientProvider;
import com.netflix.spinnaker.clouddriver.aws.security.AmazonCredentials;
import com.netflix.spinnaker.clouddriver.aws.security.NetflixAmazonCredentials;
import com.netflix.spinnaker.clouddriver.eureka.api.Eureka;
import com.netflix.spinnaker.clouddriver.eureka.deploy.ops.AbstractEurekaSupport;
import com.netflix.spinnaker.clouddriver.security.AccountCredentials;
import com.netflix.spinnaker.credentials.CredentialsRepository;
import com.netflix.spinnaker.kork.retrofit.exceptions.SpinnakerHttpException;
import com.netflix.spinnaker.kork.retrofit.exceptions.SpinnakerNetworkException;
import com.netflix.spinnaker.kork.retrofit.exceptions.SpinnakerServerException;
import java.io.IOException;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import javax.inject.Provider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;

public class InstanceTerminationLifecycleWorker
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(InstanceTerminationLifecycleWorker.class);
    private static final int AWS_MAX_NUMBER_OF_MESSAGES = 10;
    private static final String SUPPORTED_LIFECYCLE_TRANSITION = "autoscaling:EC2_INSTANCE_TERMINATING";
    ObjectMapper objectMapper;
    AmazonClientProvider amazonClientProvider;
    CredentialsRepository<NetflixAmazonCredentials> credentialsRepository;
    InstanceTerminationConfigurationProperties properties;
    Provider<AwsEurekaSupport> discoverySupport;
    Registry registry;
    private final ARN queueARN;
    private final ARN topicARN;
    private String queueId = null;

    public InstanceTerminationLifecycleWorker(ObjectMapper objectMapper, AmazonClientProvider amazonClientProvider, CredentialsRepository<NetflixAmazonCredentials> credentialsRepository, InstanceTerminationConfigurationProperties properties, Provider<AwsEurekaSupport> discoverySupport, Registry registry) {
        this.objectMapper = objectMapper;
        this.amazonClientProvider = amazonClientProvider;
        this.credentialsRepository = credentialsRepository;
        this.properties = properties;
        this.discoverySupport = discoverySupport;
        this.registry = registry;
        Set accountCredentials = credentialsRepository.getAll();
        this.queueARN = new ARN(accountCredentials, properties.getQueueARN());
        this.topicARN = new ARN(accountCredentials, properties.getTopicARN());
    }

    public String getWorkerName() {
        return this.queueARN.account.getName() + "/" + this.queueARN.region + "/" + InstanceTerminationLifecycleWorker.class.getSimpleName();
    }

    @Override
    public void run() {
        log.info("Starting " + this.getWorkerName());
        while (true) {
            try {
                while (true) {
                    this.listenForMessages();
                }
            }
            catch (Throwable e) {
                log.error("Unexpected error running " + this.getWorkerName() + ", restarting", e);
                continue;
            }
            break;
        }
    }

    private void listenForMessages() {
        AmazonSQS amazonSQS = this.amazonClientProvider.getAmazonSQS(this.queueARN.account, this.queueARN.region);
        AmazonSNS amazonSNS = this.amazonClientProvider.getAmazonSNS(this.topicARN.account, this.topicARN.region);
        Set accountCredentials = this.credentialsRepository.getAll();
        List<String> allAccountIds = InstanceTerminationLifecycleWorker.getAllAccountIds(accountCredentials);
        this.queueId = InstanceTerminationLifecycleWorker.ensureQueueExists(amazonSQS, this.queueARN, this.topicARN, InstanceTerminationLifecycleWorker.getSourceRoleArns(accountCredentials), this.properties.getSqsMessageRetentionPeriodSeconds());
        InstanceTerminationLifecycleWorker.ensureTopicExists(amazonSNS, this.topicARN, allAccountIds, this.queueARN);
        while (true) {
            ReceiveMessageResult receiveMessageResult;
            if ((receiveMessageResult = amazonSQS.receiveMessage(new ReceiveMessageRequest(this.queueId).withMaxNumberOfMessages(Integer.valueOf(10)).withVisibilityTimeout(Integer.valueOf(this.properties.getVisibilityTimeout())).withWaitTimeSeconds(Integer.valueOf(this.properties.getWaitTimeSeconds())))).getMessages().isEmpty()) {
                continue;
            }
            receiveMessageResult.getMessages().forEach(message -> {
                LifecycleMessage lifecycleMessage = this.unmarshalLifecycleMessage(message.getBody());
                if (lifecycleMessage != null) {
                    if (!SUPPORTED_LIFECYCLE_TRANSITION.equalsIgnoreCase(lifecycleMessage.lifecycleTransition)) {
                        log.info("Ignoring unsupported lifecycle transition: " + lifecycleMessage.lifecycleTransition);
                        InstanceTerminationLifecycleWorker.deleteMessage(amazonSQS, this.queueId, message);
                        return;
                    }
                    this.handleMessage(lifecycleMessage);
                }
                InstanceTerminationLifecycleWorker.deleteMessage(amazonSQS, this.queueId, message);
                this.registry.counter(this.getProcessedMetricId(this.queueARN.region)).increment();
            });
        }
    }

    private LifecycleMessage unmarshalLifecycleMessage(String messageBody) {
        String body = messageBody;
        try {
            NotificationMessageWrapper wrapper = (NotificationMessageWrapper)this.objectMapper.readValue(messageBody, NotificationMessageWrapper.class);
            if (wrapper != null && wrapper.message != null) {
                body = wrapper.message;
            }
        }
        catch (IOException e) {
            log.debug("Unable unmarshal NotificationMessageWrapper. Assuming SQS message. (body: {})", (Object)messageBody, (Object)e);
        }
        LifecycleMessage lifecycleMessage = null;
        try {
            lifecycleMessage = (LifecycleMessage)this.objectMapper.readValue(body, LifecycleMessage.class);
        }
        catch (IOException e) {
            log.error("Unable to unmarshal LifecycleMessage (body: {})", (Object)body, (Object)e);
        }
        return lifecycleMessage;
    }

    private void handleMessage(LifecycleMessage message) {
        NetflixAmazonCredentials credentials = this.getAccountCredentialsById(message.accountId);
        if (credentials == null) {
            log.error("Unable to find credentials for account id: {}", (Object)message.accountId);
            return;
        }
        Names names = Names.parseName((String)message.autoScalingGroupName);
        Eureka eureka = ((AwsEurekaSupport)((Object)this.discoverySupport.get())).getEureka((Object)credentials, this.queueARN.region);
        if (!this.updateInstanceStatus(eureka, names.getApp(), message.ec2InstanceId)) {
            this.registry.counter(this.getFailedMetricId(this.queueARN.region)).increment();
        }
        this.recordLag(message.time, this.queueARN.region, message.accountId, message.autoScalingGroupName, message.ec2InstanceId);
    }

    private boolean updateInstanceStatus(Eureka eureka, String app, String instanceId) {
        for (int retry = 0; retry < this.properties.getEurekaUpdateStatusRetryMax(); ++retry) {
            try {
                eureka.updateInstanceStatus(app, instanceId, AbstractEurekaSupport.DiscoveryStatus.OUT_OF_SERVICE.getValue());
                return true;
            }
            catch (SpinnakerServerException e) {
                String recoverableMessage = "Failed marking app out of service (status: {}, app: {}, instance: {}, retry: {})";
                if (e instanceof SpinnakerHttpException && HttpStatus.NOT_FOUND.value() == ((SpinnakerHttpException)((Object)e)).getResponseCode()) {
                    log.warn("Failed marking app out of service (status: {}, app: {}, instance: {}, retry: {})", new Object[]{404, app, instanceId, retry});
                    continue;
                }
                if (e instanceof SpinnakerNetworkException) {
                    log.error("Failed marking app out of service (status: {}, app: {}, instance: {}, retry: {})", new Object[]{"none", app, instanceId, retry, e});
                    continue;
                }
                log.error("Irrecoverable error while marking app out of service (app: {}, instance: {}, retry: {})", new Object[]{app, instanceId, retry, e});
                break;
            }
        }
        return false;
    }

    private static void deleteMessage(AmazonSQS amazonSQS, String queueUrl, Message message) {
        try {
            amazonSQS.deleteMessage(queueUrl, message.getReceiptHandle());
        }
        catch (ReceiptHandleIsInvalidException e) {
            log.warn("Error deleting lifecycle message, reason: {} (receiptHandle: {})", (Object)e.getMessage(), (Object)message.getReceiptHandle());
        }
    }

    private NetflixAmazonCredentials getAccountCredentialsById(String accountId) {
        for (NetflixAmazonCredentials credentials : this.credentialsRepository.getAll()) {
            if (credentials.getAccountId() == null || !credentials.getAccountId().equals(accountId)) continue;
            return credentials;
        }
        return null;
    }

    private static String ensureTopicExists(AmazonSNS amazonSNS, ARN topicARN, List<String> allAccountIds, ARN queueARN) {
        topicARN.arn = amazonSNS.createTopic(topicARN.name).getTopicArn();
        amazonSNS.setTopicAttributes(new SetTopicAttributesRequest().withTopicArn(topicARN.arn).withAttributeName("Policy").withAttributeValue(InstanceTerminationLifecycleWorker.buildSNSPolicy(topicARN, allAccountIds).toJson()));
        amazonSNS.subscribe(topicARN.arn, "sqs", queueARN.arn);
        return topicARN.arn;
    }

    private static Policy buildSNSPolicy(ARN topicARN, List<String> allAccountIds) {
        Statement statement = new Statement(Statement.Effect.Allow).withActions(new Action[]{SNSActions.Publish});
        statement.setPrincipals((Collection)allAccountIds.stream().map(Principal::new).collect(Collectors.toList()));
        statement.setResources(Collections.singletonList(new Resource(topicARN.arn)));
        return new Policy("allow-remote-account-send", Collections.singletonList(statement));
    }

    private static String ensureQueueExists(AmazonSQS amazonSQS, ARN queueARN, ARN topicARN, Set<String> terminatingRoleArns, int sqsMessageRetentionPeriodSeconds) {
        String queueUrl = amazonSQS.createQueue(queueARN.name).getQueueUrl();
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put("Policy", InstanceTerminationLifecycleWorker.buildSQSPolicy(queueARN, topicARN, terminatingRoleArns).toJson());
        attributes.put("MessageRetentionPeriod", Integer.toString(sqsMessageRetentionPeriodSeconds));
        amazonSQS.setQueueAttributes(queueUrl, attributes);
        return queueUrl;
    }

    private static Policy buildSQSPolicy(ARN queue, ARN topic, Set<String> terminatingRoleArns) {
        Statement snsStatement = new Statement(Statement.Effect.Allow).withActions(new Action[]{SQSActions.SendMessage});
        snsStatement.setPrincipals(new Principal[]{Principal.All});
        snsStatement.setResources(Collections.singletonList(new Resource(queue.arn)));
        snsStatement.setConditions(Collections.singletonList(new Condition().withType("ArnEquals").withConditionKey("aws:SourceArn").withValues(new String[]{topic.arn})));
        Statement sqsStatement = new Statement(Statement.Effect.Allow).withActions(new Action[]{SQSActions.SendMessage, SQSActions.GetQueueUrl});
        sqsStatement.setPrincipals((Collection)terminatingRoleArns.stream().map(Principal::new).collect(Collectors.toList()));
        sqsStatement.setResources(Collections.singletonList(new Resource(queue.arn)));
        return new Policy("allow-sns-or-sqs-send", Arrays.asList(snsStatement, sqsStatement));
    }

    Id getLagMetricId(String region) {
        return this.registry.createId("terminationLifecycle.lag", new String[]{"region", region});
    }

    void recordLag(Date start, String region, String account, String serverGroup, String instanceId) {
        if (start != null) {
            Long lag = this.registry.clock().wallTime() - start.getTime();
            log.info("Lifecycle message processed (account: {}, serverGroup: {}, instance: {}, lagSeconds: {})", new Object[]{account, serverGroup, instanceId, Duration.ofMillis(lag).getSeconds()});
            this.registry.gauge(this.getLagMetricId(region), (Number)lag);
        }
    }

    Id getProcessedMetricId(String region) {
        return this.registry.createId("terminationLifecycle.totalProcessed", new String[]{"region", region});
    }

    Id getFailedMetricId(String region) {
        return this.registry.createId("terminationLifecycle.totalFailed", new String[]{"region", region});
    }

    private static List<String> getAllAccountIds(Set<? extends AccountCredentials> accountCredentials) {
        return accountCredentials.stream().map(AccountCredentials::getAccountId).filter(a -> a != null).collect(Collectors.toList());
    }

    private static <T extends AccountCredentials> Set<String> getSourceRoleArns(Set<T> allCredentials) {
        HashSet<String> sourceRoleArns = new HashSet<String>();
        for (AccountCredentials credentials : allCredentials) {
            NetflixAmazonCredentials c;
            if (!(credentials instanceof NetflixAmazonCredentials) || (c = (NetflixAmazonCredentials)credentials).getLifecycleHooks() == null) continue;
            sourceRoleArns.addAll(c.getLifecycleHooks().stream().filter(h -> SUPPORTED_LIFECYCLE_TRANSITION.equals(h.getLifecycleTransition())).map(AmazonCredentials.LifecycleHook::getRoleARN).collect(Collectors.toSet()));
        }
        return sourceRoleArns;
    }
}

