/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.fiat.shared;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import com.netflix.spinnaker.fiat.model.Authorization;
import com.netflix.spinnaker.fiat.model.UserPermission;
import com.netflix.spinnaker.fiat.model.resources.Account;
import com.netflix.spinnaker.fiat.model.resources.ResourceType;
import com.netflix.spinnaker.fiat.shared.FiatClientConfigurationProperties;
import com.netflix.spinnaker.fiat.shared.FiatService;
import com.netflix.spinnaker.fiat.shared.FiatStatus;
import com.netflix.spinnaker.kork.exceptions.IntegrationException;
import com.netflix.spinnaker.kork.telemetry.caffeine.CaffeineStatsCounter;
import com.netflix.spinnaker.security.AuthenticatedRequest;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Component;
import org.springframework.util.backoff.BackOffExecution;
import org.springframework.util.backoff.ExponentialBackOff;
import retrofit.RetrofitError;

@Component
public class FiatPermissionEvaluator
implements PermissionEvaluator {
    private static final Logger log = LoggerFactory.getLogger(FiatPermissionEvaluator.class);
    private static final ThreadLocal<AuthorizationFailure> authorizationFailure = new ThreadLocal();
    private final Registry registry;
    private final FiatService fiatService;
    private final FiatStatus fiatStatus;
    private final Cache<String, UserPermission.View> permissionsCache;
    private final Id getPermissionCounterId;
    private final RetryHandler retryHandler;

    @Autowired
    public FiatPermissionEvaluator(Registry registry, FiatService fiatService, FiatClientConfigurationProperties configProps, FiatStatus fiatStatus) {
        this(registry, fiatService, configProps, fiatStatus, FiatPermissionEvaluator.buildRetryHandler(configProps));
    }

    private static RetryHandler buildRetryHandler(FiatClientConfigurationProperties fiatClientConfigurationProperties) {
        return new ExponentialBackoffRetryHandler(fiatClientConfigurationProperties.getRetry());
    }

    FiatPermissionEvaluator(Registry registry, FiatService fiatService, FiatClientConfigurationProperties configProps, FiatStatus fiatStatus, RetryHandler retryHandler) {
        this.registry = registry;
        this.fiatService = fiatService;
        this.fiatStatus = fiatStatus;
        this.retryHandler = retryHandler;
        this.permissionsCache = Caffeine.newBuilder().maximumSize((long)configProps.getCache().getMaxEntries().intValue()).expireAfterWrite((long)configProps.getCache().getExpiresAfterWriteSeconds().intValue(), TimeUnit.SECONDS).recordStats(() -> new CaffeineStatsCounter(registry, "fiat.permissionsCache")).build();
        this.getPermissionCounterId = registry.createId("fiat.getPermission");
    }

    public boolean hasPermission(Authentication authentication, Object resource, Object authorization) {
        return false;
    }

    public boolean canCreate(String resourceType, Object resource) {
        if (!this.fiatStatus.isEnabled()) {
            return true;
        }
        String username = this.getUsername(SecurityContextHolder.getContext().getAuthentication());
        try {
            return (Boolean)AuthenticatedRequest.propagate(() -> this.retryHandler.retry("determine whether " + username + " can create resource " + resource, () -> {
                try {
                    this.fiatService.canCreate(username, resourceType, resource);
                    return true;
                }
                catch (RetrofitError re) {
                    boolean shouldRetry = true;
                    if (re.getKind() == RetrofitError.Kind.HTTP) {
                        switch (HttpStatus.valueOf((int)re.getResponse().getStatus())) {
                            case NOT_FOUND: {
                                return false;
                            }
                            case BAD_REQUEST: {
                                shouldRetry = false;
                            }
                        }
                    }
                    IntegrationException ie = new IntegrationException((Throwable)re);
                    ie.setRetryable(Boolean.valueOf(shouldRetry));
                    throw ie;
                }
            })).call();
        }
        catch (Exception e) {
            log.info(e.toString());
            return false;
        }
    }

    public boolean hasCachedPermission(String username) {
        if (!this.fiatStatus.isEnabled()) {
            return true;
        }
        return this.permissionsCache.getIfPresent((Object)username) != null;
    }

    public boolean hasPermission(String username, Serializable resourceName, String resourceType, Object authorization) {
        UserPermission.View permission;
        boolean hasPermission;
        if (!this.fiatStatus.isEnabled()) {
            return true;
        }
        if (resourceName == null || resourceType == null || authorization == null) {
            log.warn("Permission denied because at least one of the required arguments was null. resourceName={}, resourceType={}, authorization={}", new Object[]{resourceName, resourceType, authorization});
            return false;
        }
        ResourceType r = ResourceType.parse((String)resourceType);
        Authorization a = null;
        if (!r.equals((Object)ResourceType.SERVICE_ACCOUNT)) {
            a = Authorization.valueOf((String)authorization.toString());
        }
        if (a == Authorization.CREATE) {
            throw new IllegalArgumentException("This method should not be called for `CREATE`. Please call the other implementation");
        }
        if (r.equals((Object)ResourceType.APPLICATION) && StringUtils.isNotEmpty((CharSequence)resourceName.toString())) {
            resourceName = resourceName.toString();
        }
        authorizationFailure.set((hasPermission = this.permissionContains(permission = this.getPermission(username), resourceName.toString(), r, a)) ? null : new AuthorizationFailure(a, r, resourceName.toString()));
        if (permission != null && permission.isLegacyFallback() && hasPermission) {
            if (a == Authorization.READ) {
                log.debug("Legacy fallback granted {} access (type: {}, resource: {})", new Object[]{a, r, resourceName});
            } else {
                log.warn("Legacy fallback granted {} access (type: {}, resource: {})", new Object[]{a, r, resourceName});
            }
        }
        return hasPermission;
    }

    public boolean hasPermission(Authentication authentication, Serializable resourceName, String resourceType, Object authorization) {
        if (!this.fiatStatus.isEnabled()) {
            return true;
        }
        return this.hasPermission(this.getUsername(authentication), resourceName, resourceType, authorization);
    }

    public void invalidatePermission(String username) {
        this.permissionsCache.invalidate((Object)username);
    }

    public UserPermission.View getPermission(String username) {
        UserPermission.View view = null;
        if (StringUtils.isEmpty((CharSequence)username)) {
            return null;
        }
        AtomicBoolean cacheHit = new AtomicBoolean(true);
        AtomicBoolean successfulLookup = new AtomicBoolean(true);
        AtomicBoolean legacyFallback = new AtomicBoolean(false);
        AtomicReference<Exception> exception = new AtomicReference<Exception>();
        try {
            view = (UserPermission.View)this.permissionsCache.get((Object)username, loadUserName -> {
                cacheHit.set(false);
                try {
                    return (UserPermission.View)AuthenticatedRequest.propagate(() -> {
                        try {
                            return this.retryHandler.retry("getUserPermission for " + loadUserName, () -> this.fiatService.getUserPermission((String)loadUserName));
                        }
                        catch (Exception e) {
                            if (!this.fiatStatus.isLegacyFallbackEnabled()) {
                                throw e;
                            }
                            legacyFallback.set(true);
                            successfulLookup.set(false);
                            exception.set(e);
                            return this.buildFallbackView();
                        }
                    }).call();
                }
                catch (RuntimeException re) {
                    throw re;
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
        }
        catch (Exception e) {
            successfulLookup.set(false);
            exception.set((Exception)(e.getCause() != null ? e.getCause() : e));
        }
        Id id = this.getPermissionCounterId.withTag("cached", cacheHit.get()).withTag("success", successfulLookup.get());
        if (!successfulLookup.get()) {
            log.error("Cannot get whole user permission for user {}, reason: {} (fallbackAccounts: {})", new Object[]{username, ((Throwable)exception.get()).getMessage(), this.getAccountsForView(view)});
            id = id.withTag("legacyFallback", legacyFallback.get());
        }
        this.registry.counter(id).increment();
        if (view != null && view.isLegacyFallback() && view.getAccounts().isEmpty()) {
            view = this.buildFallbackView();
            log.debug("Rebuilt legacy fallback user permission for {} (fallbackAccounts: {})", (Object)username, (Object)this.getAccountsForView(view));
        }
        return view;
    }

    @Deprecated
    public boolean storeWholePermission() {
        if (!this.fiatStatus.isEnabled()) {
            return true;
        }
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        UserPermission.View permission = this.getPermission(this.getUsername(authentication));
        return permission != null;
    }

    public static Optional<AuthorizationFailure> getAuthorizationFailure() {
        return Optional.ofNullable(authorizationFailure.get());
    }

    private String getUsername(Authentication authentication) {
        String username = "anonymous";
        if (authentication != null && authentication.isAuthenticated() && authentication.getPrincipal() != null) {
            Object principal = authentication.getPrincipal();
            if (principal instanceof UserDetails) {
                username = ((UserDetails)principal).getUsername();
            } else if (StringUtils.isNotEmpty((CharSequence)principal.toString())) {
                username = principal.toString();
            }
        }
        return username;
    }

    private boolean permissionContains(UserPermission.View permission, String resourceName, ResourceType resourceType, Authorization authorization) {
        if (permission == null) {
            return false;
        }
        if (permission.isAdmin()) {
            return true;
        }
        Function<Set, Boolean> containsAuth = resources -> resources.stream().anyMatch(view -> {
            Set authorizations = Optional.ofNullable(view.getAuthorizations()).orElse(Collections.emptySet());
            return view.getName().equalsIgnoreCase(resourceName) && authorizations.contains(authorization);
        });
        if (resourceType.equals((Object)ResourceType.ACCOUNT)) {
            boolean authorized = containsAuth.apply(permission.getAccounts());
            if (!authorized) {
                Map<String, Set> accounts = permission.getAccounts().stream().collect(Collectors.toMap(Account.View::getName, Account.View::getAuthorizations));
                log.debug("Authorization={} denied to account={} for user permission={}, found={}", new Object[]{authorization.toString(), resourceName, permission.getName(), accounts.toString()});
            }
            return authorized;
        }
        if (resourceType.equals((Object)ResourceType.APPLICATION)) {
            boolean applicationHasPermissions = permission.getApplications().stream().anyMatch(a -> a.getName().equalsIgnoreCase(resourceName));
            if (!applicationHasPermissions && permission.isAllowAccessToUnknownApplications()) {
                return true;
            }
            return permission.isLegacyFallback() || containsAuth.apply(permission.getApplications()) != false;
        }
        if (resourceType.equals((Object)ResourceType.SERVICE_ACCOUNT)) {
            return permission.getServiceAccounts().stream().anyMatch(view -> view.getName().equalsIgnoreCase(resourceName));
        }
        if (resourceType.equals((Object)ResourceType.BUILD_SERVICE)) {
            return permission.isLegacyFallback() || containsAuth.apply(permission.getBuildServices()) != false;
        }
        if (permission.getExtensionResources() != null && permission.getExtensionResources().containsKey(resourceType)) {
            Set extensionResources = (Set)permission.getExtensionResources().get(resourceType);
            return permission.isLegacyFallback() || containsAuth.apply(extensionResources) != false;
        }
        return false;
    }

    private UserPermission.View buildFallbackView() {
        return new UserPermission.View(new UserPermission().setId(AuthenticatedRequest.getSpinnakerUser().orElse("anonymous")).setAccounts(Arrays.stream(AuthenticatedRequest.getSpinnakerAccounts().orElse("").split(",")).filter(a -> a != null && !a.isEmpty()).map(a -> new Account().setName(a)).collect(Collectors.toSet()))).setLegacyFallback(true).setAllowAccessToUnknownApplications(true);
    }

    private String getAccountsForView(UserPermission.View view) {
        String fallbackAccounts = "''";
        if (view != null && view.getAccounts() != null && !view.getAccounts().isEmpty()) {
            fallbackAccounts = view.getAccounts().stream().map(Account.View::getName).collect(Collectors.joining(","));
        }
        return fallbackAccounts;
    }

    public boolean isAdmin() {
        return true;
    }

    public boolean isAdmin(Authentication authentication) {
        if (!this.fiatStatus.isEnabled()) {
            return true;
        }
        UserPermission.View permission = this.getPermission(this.getUsername(authentication));
        return permission != null && permission.isAdmin();
    }

    public static class AuthorizationFailure {
        private final Authorization authorization;
        private final ResourceType resourceType;
        private final String resourceName;

        public AuthorizationFailure(Authorization authorization, ResourceType resourceType, String resourceName) {
            this.authorization = authorization;
            this.resourceType = resourceType;
            this.resourceName = resourceName;
        }

        public Authorization getAuthorization() {
            return this.authorization;
        }

        public boolean hasAuthorization() {
            return this.authorization != null;
        }

        @Nonnull
        public ResourceType getResourceType() {
            return this.resourceType;
        }

        @Nonnull
        public String getResourceName() {
            return this.resourceName;
        }
    }

    static class ExponentialBackoffRetryHandler
    implements RetryHandler {
        private final FiatClientConfigurationProperties.RetryConfiguration retryConfiguration;

        public ExponentialBackoffRetryHandler(FiatClientConfigurationProperties.RetryConfiguration retryConfiguration) {
            this.retryConfiguration = retryConfiguration;
        }

        @Override
        public <T> T retry(String description, Callable<T> callable) throws Exception {
            ExponentialBackOff backOff = new ExponentialBackOff(this.retryConfiguration.getInitialBackoffMillis(), this.retryConfiguration.getRetryMultiplier());
            backOff.setMaxElapsedTime(this.retryConfiguration.getMaxBackoffMillis());
            BackOffExecution backOffExec = backOff.start();
            while (true) {
                try {
                    return callable.call();
                }
                catch (Throwable e) {
                    long waitTime = backOffExec.nextBackOff();
                    if (waitTime == -1L) {
                        throw e;
                    }
                    log.warn(description + " failed. Retrying in " + waitTime + "ms", e);
                    TimeUnit.MILLISECONDS.sleep(waitTime);
                    continue;
                }
                break;
            }
        }
    }

    static interface RetryHandler {
        public static final RetryHandler NOOP = new RetryHandler(){};

        default public <T> T retry(String description, Callable<T> callable) throws Exception {
            return callable.call();
        }
    }
}

