/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.front50.model;

import com.amazonaws.services.sqs.model.Message;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.netflix.spectator.api.Registry;
import com.netflix.spinnaker.front50.config.S3MetadataStorageProperties;
import com.netflix.spinnaker.front50.model.ObjectKeyLoader;
import com.netflix.spinnaker.front50.model.ObjectType;
import com.netflix.spinnaker.front50.model.StorageService;
import com.netflix.spinnaker.front50.model.TemporarySQSQueue;
import com.netflix.spinnaker.front50.model.events.S3Event;
import com.netflix.spinnaker.front50.model.events.S3EventWrapper;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import net.logstash.logback.argument.StructuredArguments;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventingS3ObjectKeyLoader
implements ObjectKeyLoader,
Runnable {
    private static final Logger log = LoggerFactory.getLogger(EventingS3ObjectKeyLoader.class);
    private static final Executor executor = Executors.newFixedThreadPool(5);
    private final ObjectMapper objectMapper;
    private final TemporarySQSQueue temporarySQSQueue;
    private final StorageService storageService;
    private final Registry registry;
    private final Cache<KeyWithObjectType, Long> objectKeysByLastModifiedCache;
    private final LoadingCache<ObjectType, Map<String, Long>> objectKeysByObjectTypeCache;
    private final String rootFolder;
    private boolean pollForMessages = true;

    public EventingS3ObjectKeyLoader(ExecutorService executionService, ObjectMapper objectMapper, S3MetadataStorageProperties s3Properties, TemporarySQSQueue temporarySQSQueue, final StorageService storageService, Registry registry, boolean scheduleImmediately) {
        this.objectMapper = objectMapper;
        this.temporarySQSQueue = temporarySQSQueue;
        this.storageService = storageService;
        this.registry = registry;
        this.objectKeysByLastModifiedCache = CacheBuilder.newBuilder().expireAfterWrite(s3Properties.getEventing().getRefreshIntervalMs() + 60000L, TimeUnit.MILLISECONDS).recordStats().build();
        this.objectKeysByObjectTypeCache = CacheBuilder.newBuilder().refreshAfterWrite(s3Properties.getEventing().getRefreshIntervalMs(), TimeUnit.MILLISECONDS).recordStats().build((CacheLoader)new CacheLoader<ObjectType, Map<String, Long>>(){

            public Map<String, Long> load(ObjectType objectType) throws Exception {
                log.debug("Loading object keys for {}", (Object)StructuredArguments.value((String)"type", (Object)objectType));
                return storageService.listObjectKeys(objectType);
            }

            public ListenableFuture<Map<String, Long>> reload(ObjectType objectType, Map<String, Long> previous) throws Exception {
                ListenableFutureTask task = ListenableFutureTask.create(() -> {
                    log.debug("Refreshing object keys for {} (asynchronous)", (Object)StructuredArguments.value((String)"type", (Object)objectType));
                    return storageService.listObjectKeys(objectType);
                });
                executor.execute((Runnable)task);
                return task;
            }
        });
        this.rootFolder = s3Properties.getRootFolder();
        if (scheduleImmediately) {
            executionService.submit(this);
        }
    }

    @PreDestroy
    public void shutdown() {
        log.debug("Stopping ...");
        this.pollForMessages = false;
        log.debug("Stopped");
    }

    public Map<String, Long> listObjectKeys(ObjectType objectType) {
        if (!this.storageService.supportsEventing(objectType)) {
            return this.storageService.listObjectKeys(objectType);
        }
        try {
            Map objectKeys = (Map)this.objectKeysByObjectTypeCache.get((Object)objectType);
            this.objectKeysByLastModifiedCache.asMap().entrySet().stream().filter(e -> ((KeyWithObjectType)e.getKey()).objectType == objectType).forEach(e -> {
                String key = ((KeyWithObjectType)e.getKey()).key;
                if (objectKeys.containsKey(key)) {
                    Long currentLastModifiedTime = (Long)e.getValue();
                    Long previousLastModifiedTime = (Long)objectKeys.get(key);
                    if (currentLastModifiedTime > previousLastModifiedTime) {
                        log.info("Detected Recent Modification (type: {}, key: {}, previous: {}, current: {})", new Object[]{StructuredArguments.value((String)"type", (Object)objectType), StructuredArguments.value((String)"key", (Object)key), StructuredArguments.value((String)"previousTime", (Object)new Date(previousLastModifiedTime)), StructuredArguments.value((String)"currentTime", (Object)new Date((Long)e.getValue()))});
                        objectKeys.put(key, currentLastModifiedTime);
                    }
                } else {
                    log.info("Detected Recent Modification (type: {}, key: {}, current: {})", new Object[]{StructuredArguments.value((String)"type", (Object)objectType), StructuredArguments.value((String)"key", (Object)key), StructuredArguments.value((String)"currentTime", (Object)new Date((Long)e.getValue()))});
                    objectKeys.put(key, (Long)e.getValue());
                }
            });
            return objectKeys;
        }
        catch (ExecutionException e2) {
            log.error("Unable to fetch keys from cache", (Throwable)e2);
            return this.storageService.listObjectKeys(objectType);
        }
    }

    @Override
    public void run() {
        while (this.pollForMessages) {
            try {
                List<Message> messages = this.temporarySQSQueue.fetchMessages();
                if (messages.isEmpty()) continue;
                messages.forEach(message -> {
                    S3Event s3Event = EventingS3ObjectKeyLoader.unmarshall(this.objectMapper, message.getBody());
                    if (s3Event != null) {
                        this.tick(s3Event);
                    }
                    this.temporarySQSQueue.markMessageAsHandled(message.getReceiptHandle());
                });
            }
            catch (Exception e) {
                log.error("Failed to poll for messages", (Throwable)e);
                this.registry.counter("s3.eventing.pollErrors").increment();
            }
        }
    }

    private void tick(S3Event s3Event) {
        s3Event.records.forEach(record -> {
            if (record.s3.object.key.endsWith("last-modified.json")) {
                return;
            }
            String eventType = record.eventName;
            KeyWithObjectType keyWithObjectType = EventingS3ObjectKeyLoader.buildObjectKey(this.rootFolder, record.s3.object.key);
            DateTime eventTime = new DateTime((Object)record.eventTime);
            log.debug("Received Event (objectType: {}, type: {}, key: {}, delta: {})", new Object[]{StructuredArguments.value((String)"objectType", (Object)keyWithObjectType.objectType), StructuredArguments.value((String)"type", (Object)eventType), StructuredArguments.value((String)"key", (Object)keyWithObjectType.key), StructuredArguments.value((String)"delta", (Object)(System.currentTimeMillis() - eventTime.getMillis()))});
            this.objectKeysByLastModifiedCache.put((Object)keyWithObjectType, (Object)eventTime.getMillis());
        });
    }

    private static KeyWithObjectType buildObjectKey(String rootFolder, String s3ObjectKey) {
        if (!((String)rootFolder).endsWith("/")) {
            rootFolder = (String)rootFolder + "/";
        }
        s3ObjectKey = s3ObjectKey.replace((CharSequence)rootFolder, "");
        s3ObjectKey = s3ObjectKey.substring(s3ObjectKey.indexOf("/") + 1);
        String metadataFilename = s3ObjectKey.substring(s3ObjectKey.lastIndexOf("/") + 1);
        s3ObjectKey = s3ObjectKey.substring(0, s3ObjectKey.lastIndexOf("/"));
        try {
            s3ObjectKey = URLDecoder.decode(s3ObjectKey, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException("Invalid key '" + s3ObjectKey + "' (non utf-8)");
        }
        ObjectType objectType = Arrays.stream(ObjectType.values()).filter(o -> o.defaultMetadataFilename.equalsIgnoreCase(metadataFilename)).findFirst().orElseThrow(() -> new IllegalArgumentException("No ObjectType found (defaultMetadataFileName: " + metadataFilename + ")"));
        return new KeyWithObjectType(objectType, s3ObjectKey);
    }

    private static S3Event unmarshall(ObjectMapper objectMapper, String messageBody) {
        S3EventWrapper s3EventWrapper;
        try {
            s3EventWrapper = (S3EventWrapper)objectMapper.readValue(messageBody, S3EventWrapper.class);
        }
        catch (IOException e) {
            log.debug("Unable unmarshal S3EventWrapper (body: {})", (Object)StructuredArguments.value((String)"message", (Object)messageBody), (Object)e);
            return null;
        }
        try {
            return (S3Event)objectMapper.readValue(s3EventWrapper.message, S3Event.class);
        }
        catch (IOException e) {
            log.debug("Unable unmarshal S3Event (body: {})", (Object)StructuredArguments.value((String)"body", (Object)s3EventWrapper.message), (Object)e);
            return null;
        }
    }

    private static class KeyWithObjectType {
        final ObjectType objectType;
        final String key;

        KeyWithObjectType(ObjectType objectType, String key) {
            this.objectType = objectType;
            this.key = key;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            KeyWithObjectType that = (KeyWithObjectType)o;
            if (this.objectType != that.objectType) {
                return false;
            }
            return this.key.equals(that.key);
        }

        public int hashCode() {
            int result = this.objectType.hashCode();
            result = 31 * result + this.key.hashCode();
            return result;
        }
    }
}

